#include "target.h"
#include "runestree.h"
#include "exception.h"
#include "color.h"
#include <pthread.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <string.h>
#include <cstdlib>

#include <iostream>

namespace RUNES {

namespace Applyer {


// コンストラクタ
Target::Target(const Bundle& source, const std::string& config_path) :
	m_resolving(false),
	m_resolved(false),
	m_circle_connect(false),
	m_source(source),
	m_config_path(config_path),
	m_applied(false)
{
	pthread_mutex_init(&m_mutex, NULL);
	pthread_cond_init(&m_cond, NULL);
}


// デストラクタ
Target::~Target() {}


// public:

// for linking
const std::list<std::string> Target::getSourcePreOrder (void) const
{
	return m_source.getPreOrder();
}

const std::list<std::string> Target::getSourcePostOrder(void) const
{
	return m_source.getPostOrder();
}

void Target::pushPreRequre(Target* target)
{
	m_pre_requres.push_back(target);
}


// for resolving
std::list<Target*> Target::resolve(void)
{
	if( m_resolved ) return std::list<Target*>();
	if( m_resolving ) {
		// 自分が解決中なのに、またresolve()が呼ばれた
		// 循環を検知
		// 自分が循環の連結点
		m_circle_connect = true;
		std::list<Target*> circle_list;
		circle_list.push_back(this);
		return circle_list;
	}

	m_resolving = true;

	for( std::list<Target*>::iterator req( m_pre_requres.begin() );
			req != m_pre_requres.end();
			++req ) {
		std::list<Target*> circle_list( (*req)->resolve() );
		if( ! circle_list.empty() ) {
			// 循環が検知されている
			if( m_circle_connect == true ) {
				// 自分が循環の連結点
				// 循環依存の検出を完了
				// 例外を投げる
				std::string msg;
				msg += "Circled dependency:  ";
				for( std::list<Target*>::const_iterator ptgt( circle_list.begin() );
						ptgt != circle_list.end();
						++ptgt ) {
					msg = msg + (*ptgt)->getName() + " -> ";
				}
				msg += this->getName();
				throw CircularPreRequresError(msg, circle_list);
			} else {
				circle_list.push_back(this);
				return circle_list;
			}
		}
	}

	m_resolved = true;
	m_resolving = false;
	return std::list<Target*>();
}

bool Target::isResolved(void) const
{
	return m_resolved;
}


// for applying
void Target::start(void)
{
	if( m_applied == true ) return;
	pthread_create(&m_thread, 0, startThread, this);
}

void* Target::startThread(void* self)
{
	Target* const me( static_cast<Target*>(self) );
	me->startIMPL();
	return NULL;
}

void Target::startIMPL(void)
{
	// 順番制御 pre指定されたサービスを待つ
	for( std::list<Target*>::iterator req( m_pre_requres.begin() );
			req != m_pre_requres.end();
			++req ) {
		(*req)->wait();
	}

	std::cout << Color::MSG << "Applying " << m_source.getPath() << std::endl;

	// 実行
	{
		const std::string& bundle_path(m_source.getPath());
		std::string exe(bundle_path + "/" + BundleComponent::EXECUTE_FILE_NAME);

		// 環境変数を設定
		::setenv("RUNES_CONFIG_PATH", m_config_path.c_str(), 1);
		::setenv("RUNES_RUNE_NAME",   m_source.getName().c_str(), 1);


		// 引数を作成
		// XXX: エラー処理
		char* args[2];

		// args[0] == exe
		args[0] = new char[ exe.size() + 1 ];
		::strcpy(args[0], exe.c_str());

		args[1] = NULL;


		// fork & exec
		int pid;
		if( (pid=::fork()) == 0 ) {
			// 子プロセス
			// XXX: エラー処理
			if( ::chdir( (bundle_path + "/" + BundleComponent::RESOURCE_DIR_NAME).c_str() ) < 0 ) {
				// RESOURCE_DIR_NAMEにcd失敗
				::chdir( bundle_path.c_str() );	// XXX: エラー無視？
			}
			::execv(exe.c_str(), args);
			// exec失敗
			// XXX: エラー処理
			std::cerr << "Can't exec " << exe.c_str() << errorDescribe() << std::endl;
			::exit(1);
		} else if( pid == -1 ) {
			// fork失敗
			// XXX: 例外処理 ただし引数を破棄しなければならない
			std::cerr << "Can't fork for " << exe << std::endl;
		} else {
			// fork成功、親プロセス
			int status_raw;
			::waitpid(pid, &status_raw, 0);
			// FIXME: WEXITSTATUS(status)が0で無い場合は？
		}


		// 引数を破棄
		delete[] args[0];
	}

	// このサービスを待っているサービスを再開
	pthread_mutex_lock(&m_mutex);
	m_applied = true;
	pthread_cond_broadcast(&m_cond);
	pthread_mutex_unlock(&m_mutex);
}

void Target::wait(void)
{
	pthread_mutex_lock(&m_mutex);
	while( ! m_applied ) {
		pthread_cond_wait(&m_cond, &m_mutex);
	}
	pthread_mutex_unlock(&m_mutex);
}

void Target::collect(void)
{
	wait(); // XXX: 必要？
	pthread_join(m_thread, NULL);
}


// for debugging
const std::string& Target::getName(void) const
{
	return m_source.getName();
}

void Target::show(void) const
{
	std::cout << "Target " << getName() << std::endl;

	for( std::list<Target*>::const_iterator req( m_pre_requres.begin() );
			req != m_pre_requres.end();
			++req ) {
		std::cout << "  " << (*req)->getName() << "\n";
	}
	std::cout << std::endl;
	// XXX: 未実装
}


}  // namespace Applyer

}  // namespace RUNES
