#include "rune.h"
#include "bundle.h"
#include "bundlemgr.h"
#include "runemgr.h"
#include "filesystem.h"

namespace RUNES {

namespace Manager {


// コンストラクタ
Rune::Rune(const Meta& source) :
	m_source(source)
{}


// デストラクタ
Rune::~Rune() {}


// public:

// isResolved
// public:
bool Rune::isResolved(const BundleManager& bmgr) const
{
	return isResolvedAND(bmgr, &m_source.getDepend());
}

// private:
bool Rune::isResolvedBASE   (const BundleManager& bmgr, const BundleDepend* obj) const
{
	switch( obj->getType() ) {
	case BundleDepend::DEP_AND:
		return isResolvedAND    (bmgr, static_cast<const BDLogicAND*>(obj)   );
	case BundleDepend::DEP_OR:
		return isResolvedOR     (bmgr, static_cast<const BDLogicOR*>(obj)    );
	case BundleDepend::DEP_BUNDLE:
		return isResolvedBundle (bmgr, static_cast<const BDTBundle*>(obj) );
	case BundleDepend::DEP_FILE:
		return isResolvedFile (static_cast<const BDTHostFile*>(obj) );
	case BundleDepend::DEP_LIB:
		return isResolvedLib  (static_cast<const BDTHostLib*>(obj)  );
	case BundleDepend::DEP_BIN:
		return isResolvedBin  (static_cast<const BDTHostBin*>(obj)  );
	}
	return false; // XXX:
}

bool Rune::isResolvedAND    (const BundleManager& bmgr, const BDLogicAND*   obj) const
{
	for( BDLogic::entries_t::const_iterator ent( obj->getEntries().begin() );
			ent != obj->getEntries().end();
			++ent ) {
		if( ! isResolvedBASE(bmgr, (*ent).get() ) ) {
			return false;
		}
	}
	return true;
}
bool Rune::isResolvedOR     (const BundleManager& bmgr, const BDLogicOR*    obj) const
{
	for( BDLogic::entries_t::const_iterator ent( obj->getEntries().begin() );
			ent != obj->getEntries().end();
			++ent ) {
		if( isResolvedBASE(bmgr, (*ent).get() ) ) {
			return true;
		}
	}
	return false;
}
bool Rune::isResolvedBundle (const BundleManager& bmgr, const BDTBundle* obj) const
{
	if( bmgr.find(obj->getName()) != NULL ) {
		return true;
	} else {
		return false;
	}
}
bool Rune::isResolvedFile   (const BDTHostFile* obj) const
{
	if( FileSystem::checkFileExists(obj->getName()) ) {
		return true;
	} else {
		return false;
	}
}
bool Rune::isResolvedLib    (const BDTHostLib*  obj) const
{
	static const char* path[] = {
		"/lib/",
		"/usr/lib/",
		"/usr/local/lib/",
		"/usr/X11R6/lib/",
		"/usr/X11R7/lib/",
		NULL,
	};
	for(const char** p = &path[0]; *p != NULL; ++p) {
		if( FileSystem::checkFileExists( std::string(*p) + obj->getName()) ) {
			return true;
		}
	}
	return false;
}
bool Rune::isResolvedBin    (const BDTHostBin*  obj) const
{
	static const char* path[] = {
		"/bin/",
		"/sbin/",
		"/usr/bin/",
		"/usr/sbin/",
		"/usr/local/bin/",
		"/usr/local/sbin/",
		"/usr/X11R6/bin/",
		"/usr/X11R7/bin/",
		NULL,
	};
	for(const char** p = &path[0]; *p != NULL; ++p) {
		if( FileSystem::checkFileExists( std::string(*p) + obj->getName()) ) {
			return true;
		}
	}
	return false;
}


// getRecursiveBundleDepend
// public:
void Rune::getRecursiveBundleDepend(
		const BundleManager& bmgr,
		const RuneManager& nmgr,
		std::list<const Rune*>* list,
		boost::function<const Rune* (std::list<const Rune*>*)> askmethod
		) const
{
	// XXX: 途中
	getRecursiveBundleDependAND(bmgr, nmgr, list, askmethod, &m_source.getDepend() );
}

// private:
bool Rune::isResolvedHostRequires(const BDTHost* obj) const
{
	switch( obj->getType() ) {
	case BundleDepend::DEP_FILE:
		return isResolvedFile (static_cast<const BDTHostFile*>(obj) );
	case BundleDepend::DEP_LIB:
		return isResolvedLib  (static_cast<const BDTHostLib*>(obj)  );
	case BundleDepend::DEP_BIN:
		return isResolvedBin  (static_cast<const BDTHostBin*>(obj)  );
	case BundleDepend::DEP_AND:
	case BundleDepend::DEP_OR:
	case BundleDepend::DEP_BUNDLE:
		assert(false);
	}
	assert(false); // XXX:
	return false;
}

void Rune::getRecursiveBundleDependAND (
		const BundleManager& bmgr,
		const RuneManager& nmgr,
		std::list<const Rune*>* list,
		boost::function<const Rune* (std::list<const Rune*>*)> askmethod,
		const BDLogicAND* obj
		) const
{
	BDLogic::entries_t entries( obj->getEntries() );

	// 仕分け
	std::list<const BDLogicAND*> and_list;
	std::list<const BDLogicOR*>  or_list;
	std::list<const BDTBundle*>  bundle_list;
	std::list<const BDTHost*>    hostreq_list;
	for( BDLogic::entries_t::const_iterator ent( entries.begin() );
			ent != entries.end();
			++ent ) {
		switch( (*ent)->getType() ) {
		case BundleDepend::DEP_AND:
			and_list.push_back(     static_cast<const BDLogicAND*>(   (*ent).get() ) );
			break;
		case BundleDepend::DEP_OR:
			or_list.push_back(      static_cast<const BDLogicOR*>(    (*ent).get() ) );
			break;
		case BundleDepend::DEP_BUNDLE:
			bundle_list.push_back(  static_cast<const BDTBundle*>( (*ent).get() ) );
			break;
		case BundleDepend::DEP_FILE:
		case BundleDepend::DEP_LIB:
		case BundleDepend::DEP_BIN:
			hostreq_list.push_back( static_cast<const BDTHost*>(       (*ent).get() ) );
			break;
		}
	}

	// I.   runeが実行される前に、hostが既に保持していなければならないもの
	//      （File, Lib, Bin = BDTHost）がホストに存在するかチェック
	//      存在すればそのBDTHostは通過
	//      存在しなければthrow LackOfFile
	//
	// II.  BDTBundleに対して、そのBundleがインストール済み（BundleManagerに入っている）かどうかチェック
	//      存在すればそのBDTBundleは通過
	//      存在しなければインストールする必要があるので、IIIに進む
	//
	// III. BDTBundleに対して、そのBundleがWebリポジトリに存在する（RuneManagerに入っている）かどうかチェック
	//      存在しなければthrow NoRuneFound
	//      存在すれば、そのRuneをインストールすることが決定。list_candidateに追加し、さらにgetRecursiveBundleDepend()
	//
	// IV.  BDLoigcAND, BDLoigcORに対してgetRecursiveBundleDepend(AND|OR)
	//
	// 以上の条件がすべて満たされたら、listにlist_candidateを追加して終了
	std::list<const Rune*> list_candidate;


	// I.   File, Lib, Binがホストに存在するかチェック
	for( std::list<const BDTHost*>::const_iterator hr( hostreq_list.begin() );
			hr != hostreq_list.end();
			++hr ) {
		if( ! isResolvedHostRequires(*hr) ) {
			// 必要なファイルがホストに存在しない
			throw LackOfFile(*hr);
		}
	}

	// II.  BDTBundle
	for( std::list<const BDTBundle*>::const_iterator bdl( bundle_list.begin() );
			bdl != bundle_list.end();
			++bdl ) {
		// II.  Bundleがインストールされているかチェック
		const Bundle* installed_bundle( bmgr.find( (*bdl)->getName() ) );
		if( installed_bundle != NULL &&	(*bdl)->getVersionRange().isSatisfy( installed_bundle->getVersion() ) ) {
			// 依存しているBundleがインストールされており、かつ、バージョンなどの依存も満たしている
			continue;
		}

		// インストールされていないか、インストールされていてもバージョンなどの依存を満たさない

		// III. Webリポジトリに存在するかチェック
		const Rune* found_rune( nmgr.find( (*bdl)->getName() ) );
		if( found_rune == NULL ) {
			// Webリポジトリに依存しているRuneが見つからない
			throw NoRuneFound(*bdl);
		} else {
			// Webリポジトリに依存しているRuneを発見
			if( ! (*bdl)->getVersionRange().isSatisfy( found_rune->m_source.getVersion() ) ) {
				// リポジトリにあるRuneはバージョンなどの依存を満たしていない
				throw NoRuneFound(*bdl);
			}

			// Webリポジトリに、バージョンなどの依存も満たすRuneを発見
			// getRecursiveBundleDepend()を呼び出し、例外が発生しなかったらlist_candidateに追加
			found_rune->getRecursiveBundleDepend(bmgr, nmgr, list, askmethod);
			list_candidate.push_back( found_rune );
		}
	}

	// IV.   BDLoigcAND, BDLoigcORに対してgetRecursiveBundleDepend(AND|OR)
	for( std::list<const BDLogicAND*>::const_iterator land( and_list.begin() );
			land != and_list.end();
			++land ) {
		getRecursiveBundleDependAND(bmgr, nmgr, list, askmethod, *land);
	}
	for( std::list<const BDLogicOR*>::const_iterator lor( or_list.begin() );
			lor != or_list.end();
			++lor ) {
		getRecursiveBundleDependOR(bmgr, nmgr, list, askmethod, *lor);
	}

	// すべての条件が満たされたので、listにlist_candidateを追加
	list->merge(list_candidate);
}


namespace {
struct UnsatisfiedORDependencyDetected : public std::runtime_error {
	UnsatisfiedORDependencyDetected() : std::runtime_error("") {}
};
struct AskMethodThrow {
	const Rune* operator() (std::list<const Rune*>* list)
	{
		throw UnsatisfiedORDependencyDetected();
	}
};
}  // noname namespace

void Rune::getRecursiveBundleDependOR  (
		const BundleManager& bmgr,
		const RuneManager& nmgr,
		std::list<const Rune*>* list,
		boost::function<const Rune* (std::list<const Rune*>*)> askmethod,
		const BDLogicOR* obj
		) const
{
	BDLogic::entries_t entries( obj->getEntries() );

	// 仕分け
	std::list<const BDLogicAND*> and_list;
	std::list<const BDLogicOR*>  or_list;
	std::list<const BDTBundle*>  bundle_list;
	std::list<const BDTHost*>    hostreq_list;
	for( BDLogic::entries_t::const_iterator ent( entries.begin() );
			ent != entries.end();
			++ent ) {
		switch( (*ent)->getType() ) {
		case BundleDepend::DEP_AND:
			and_list.push_back(     static_cast<const BDLogicAND*>(   (*ent).get() ) );
			break;
		case BundleDepend::DEP_OR:
			or_list.push_back(      static_cast<const BDLogicOR*>(    (*ent).get() ) );
			break;
		case BundleDepend::DEP_BUNDLE:
			bundle_list.push_back(  static_cast<const BDTBundle*>( (*ent).get() ) );
			break;
		case BundleDepend::DEP_FILE:
		case BundleDepend::DEP_LIB:
		case BundleDepend::DEP_BIN:
			hostreq_list.push_back( static_cast<const BDTHost*>(       (*ent).get() ) );
			break;
		}
	}

	// I.   runeが実行される前に、hostが既に保持していなければならないもの
	//      （File, Lib, Bin = BDTHost）がホストに存在するかチェック
	//      存在すれば終了
	//
	// II.  BDTBundleに対して、そのBundleがインストール済み（BundleManagerに入っている）かどうかチェック
	//      存在すれば終了
	//
	// III. にBDTBundleに対して、そのBundleがWebリポジトリに存在する（RuneManagerに入っている）かどうかチェック
	//      存在すれば、try{}の中でgetRecursiveBundleDepend()し、例外が発生しなければlist_candidateに追加し、終了
	//
	// IV.  BDLoigcOR, BDLoigcANDに対してgetRecursiveBundleDepend(AND|OR)
	//      例外が発生しなかった = 満たされたら、終了
	//
	// 以上の条件が一つも満たされなかったら、例外を投げる
	std::list<const Rune*> candidates;


	// I.   File, Lib, Binがホストに存在するかチェック
	for( std::list<const BDTHost*>::const_iterator hr( hostreq_list.begin() );
			hr != hostreq_list.end();
			++hr ) {
		if( isResolvedHostRequires(*hr) ) {
			// 必要なファイルがホストに存在
			return;
		}
	}


	// II.  BDTBundle
	for( std::list<const BDTBundle*>::const_iterator bdl( bundle_list.begin() );
			bdl != bundle_list.end();
			++bdl ) {
		// II.  Bundleがインストールされているかチェック
		const Bundle* installed_bundle( bmgr.find( (*bdl)->getName() ) );
		if( installed_bundle != NULL &&	(*bdl)->getVersionRange().isSatisfy( installed_bundle->getVersion() ) ) {
			// 依存しているBundleがインストールされており、かつ、バージョンなどの依存も満たしている
			return;
		}


		// 以下、インストールされていないか、インストールされていてもバージョンなどの依存を満たさない

		// III. Webリポジトリに存在するかチェック
		const Rune* found_rune( nmgr.find( (*bdl)->getName() ) );
		if( found_rune != NULL ) {
			// Webリポジトリに依存しているRuneが見つからない
			continue;
		} else {
			// Webリポジトリに依存しているRuneを発見
			if( ! (*bdl)->getVersionRange().isSatisfy( found_rune->m_source.getVersion() ) ) {
				// リポジトリにあるRuneはバージョンなどの依存を満たしていない
				continue;
			}

			// Webリポジトリに、バージョンなどの依存も満たすRuneを発見
			// OR依存で例外を投げるaskmethodでgetRecursiveBundleDepend()を呼び出し、
			// 例外が発生せず、かつtmp_list.empty()==trueなら（=既に満たされている）終了
			try {
				std::list<const Rune*> tmp_list;
				found_rune->getRecursiveBundleDepend(bmgr, nmgr, &tmp_list, AskMethodThrow());
				list->insert(list->end(), tmp_list.begin(), tmp_list.end());
				list->push_back( found_rune );
				if( tmp_list.empty() ) return;
			} catch (const DependencyUnsatisfied& ex) {
				candidates.push_back(found_rune);
				continue;
			} catch (const UnsatisfiedORDependencyDetected& ex) {
				candidates.push_back(found_rune);
				continue;
			}
		}
	}

	// IV.   BDLoigcOR, BDLoigcANDに対してgetRecursiveBundleDepend(AND|OR)
	//       OR依存で例外を投げるaskmethodで呼び出す
	//       例外が発生せず、かつtmp_list.empty()==trueなら（=既に満たされている）終了
	for( std::list<const BDLogicOR*>::const_iterator lor( or_list.begin() );
			lor != or_list.end();
			++lor ) {
		try {
			std::list<const Rune*> tmp_list;
			getRecursiveBundleDependOR(bmgr, nmgr, &tmp_list, AskMethodThrow(), *lor);
			list->insert(list->end(), tmp_list.begin(), tmp_list.end());
			if( tmp_list.empty() ) return;
		} catch (const DependencyUnsatisfied& ex) {
			continue;
		} catch (const UnsatisfiedORDependencyDetected& ex) {
			continue;
		}
	}
	for( std::list<const BDLogicAND*>::const_iterator land( and_list.begin() );
			land != and_list.end();
			++land ) {
		try {
			std::list<const Rune*> tmp_list;
			getRecursiveBundleDependAND(bmgr, nmgr, &tmp_list, AskMethodThrow(), *land);
			list->insert(list->end(), tmp_list.begin(), tmp_list.end());
			if( tmp_list.empty() ) return;
		} catch (const DependencyUnsatisfied& ex) {
			continue;
		} catch (const UnsatisfiedORDependencyDetected& ex) {
			continue;
		}
	}


	// 既に満たされている依存は１つもない
	// askmethodを使って１つ選択する
	if( candidates.empty() ) {
		throw NoneOfEntriesSatisfied(obj);
	} else if( candidates.size() == 1 ) {
		const Rune* use( *candidates.begin() );
		list->push_back(use);
		use->getRecursiveBundleDepend(bmgr, nmgr, list, askmethod);
	} else {
		const Rune* selected( askmethod(&candidates) );
		selected->getRecursiveBundleDepend(bmgr, nmgr, list, askmethod);
		list->push_back(selected);
	}
}


}  // namespace Manager

}  // namespace RUNES
