#include "bundle_parser.h"
#include "bundle_info.h"
#include "exception.h"
#include <list>
#include <sstream>
#include <algorithm>
#include <boost/lexical_cast.hpp>

#include <iostream>

namespace RUNES {


// コンストラクタ
BundleParser::BundleParser(const Element& document, BundleInfo* result) :
	m_document(document),
	m_result(result)
{
	// XXX: resultのNULLチェック
}

// デストラクタ
BundleParser::~BundleParser()
{}


// public:
namespace {
const bool findPreferredValue(const Element& parent, const std::string& name, std::string* result) {
	Element::const_reverse_iterator ite( parent.getLastChildByName(name) );
	if( ite != parent.rend() && ite->hasText() ) {
		*result = *ite->getTexts().begin();
		return true;
	}
	return false;
}

const bool makeMargedElement(const Element& parent, const std::string& name, std::list<const Element*>* result) {
	result->clear();
	for(Element::const_iterator ch(parent.begin()), ch_end(parent.end());
			ch != ch_end;
			++ch ) {
		if( ch->getName() == name ) {
			for(Element::const_iterator marg(ch->begin()), marg_end(ch->end());
					marg != marg_end;
					++marg ) {
				result->push_back(&(*marg));
			}
		}
	}
	return result->empty();
}

bool findPreferredAttribute(const Element& element, const std::string& key, std::string* result)
{
	typedef Element::attributes_type::const_reverse_iterator rite;
	for(rite attr(element.getAttributes().rbegin()), attr_end(element.getAttributes().rend());
				attr != attr_end;
				++attr ) {
		if( attr->first == key ) {
			*result = attr->second;
			return true;
		}
	}
	return false;
}

void DependContext(const Element& source, BDLogic* target) 
{
	if( source.getName() == "and" ) {
		BDLogicAND diland;
		for(Element::const_iterator el(source.begin()), el_end(source.end());
				el != el_end;
				++el ) {
			DependContext(*el, &diland);
		}
		target->addTarget(diland);

	} else if( source.getName() == "or" ) {
		BDLogicOR dilor;
		for(Element::const_iterator el(source.begin()), el_end(source.end());
				el != el_end;
				++el ) {
			DependContext(*el, &dilor);
		}
		target->addTarget(dilor);

	} else if( source.getName() == "bundle" ) {
		if( source.hasText() ) {
			std::string ver;
			findPreferredAttribute(source, "version", &ver);

			// XXX: デフォルトorderはどこかに定数を定義
			BDTBundle::order_t order( BDTBundle::ORDER_PRE );
			std::string order_text;
			findPreferredAttribute(source, "order", &order_text);
			if( order_text == "pre" ) {
				order = BDTBundle::ORDER_PRE;
			} else if ( order_text == "post" ) {
				order = BDTBundle::ORDER_POST;
			} else if ( order_text == "any" ) {
				order = BDTBundle::ORDER_ANY;
			}

			BDTBundle b(*source.getTexts().begin(), VersionRange(ver), order);
			target->addTarget(b);
		}

	} else if( source.getName() == "file" ) {
		if( source.hasText() ) {
			BDTHostFile h( *source.getTexts().begin() );
			target->addTarget(h);
		}

	} else if( source.getName() == "lib" ) {
		if( source.hasText() ) {
			BDTHostLib  h( *source.getTexts().begin() );
			target->addTarget(h);
		}

	} else if( source.getName() == "bin" ) {
		if( source.hasText() ) {
			BDTHostBin  h( *source.getTexts().begin() );
			target->addTarget(h);
		}

	}
}

}  // noname namespace

void BundleParser::parse(void)
{
	const Element::list_type& root_list( m_document.getChildren() );
	if( root_list.empty() ) {
		return;
	}
	const Element& root( *root_list.begin() );

	findPreferredValue(root, "name", &m_result->m_name);
	{
		std::string version_string;
		if( findPreferredValue(root, "version", &version_string) ) {
			m_result->m_version = Version(version_string);
		}
	}
	findPreferredValue(root, "release", &m_result->m_release);
	findPreferredValue(root, "build", &m_result->m_build);
	findPreferredValue(root, "arch", &m_result->m_arch);
	findPreferredValue(root, "license", &m_result->m_license);
	findPreferredValue(root, "packager", &m_result->m_packager);

	{
		std::list<const Element*> sums;
		makeMargedElement(root, "summary", &sums);
		for(std::list<const Element*>::const_iterator lang(sums.begin()), lang_end(sums.end());
				lang != lang_end;
				++lang ) {
			if( (*lang)->hasText() ) {
				m_result->m_summary.insert(
						std::make_pair( (*lang)->getName(), *(*lang)->getTexts().begin() )
						);
			}
		}
	}

	{
		std::list<const Element*> sums;
		makeMargedElement(root, "description", &sums);
		for(std::list<const Element*>::const_iterator lang(sums.begin()), lang_end(sums.end());
				lang != lang_end;
				++lang ) {
			if( (*lang)->hasText() ) {
				m_result->m_description.insert(
						std::make_pair( (*lang)->getName(), *(*lang)->getTexts().begin() )
						);
			}
		}
	}

	// depend
	{
		std::list<const Element*> deps;
		makeMargedElement(root, "depend", &deps);
		for(std::list<const Element*>::const_iterator pel(deps.begin()), pel_end(deps.end());
				pel != pel_end;
				++pel ) {
			DependContext(**pel, &m_result->m_depend);
		}
	}
}


#if 0
// コンストラクタ
BundleParser::BundleParser(std::istream& source_stream, BundleInfo* result) :
	SAX(source_stream),
	m_result(result)
{
	// XXX: resultのNULLチェック
}

// デストラクタ
BundleParser::~BundleParser()
{}


// private:
const std::string& BundleParser::getTopElement(void) const
{
	// XXX: m_path.empty()チェックが必要
	return *(m_path.rbegin());
}
const std::string& BundleParser::getBottomElement(void) const
{
	// XXX: m_path.size() > 1 チェックが必要
	return *(++m_path.begin());
}


// private:
void BundleParser::initializer(void)
{
	m_logic_path.clear();
	m_logic_path.push_back(&m_depend);
}

void BundleParser::finalizer(void)
{
	m_result->m_depend = m_depend;
}

void BundleParser::startElementHandler(const std::string& name, const attributes_t& attributes)
{
	m_path.push_back(name);
	m_current_atts = attributes;	// current_attsはpath状になっていないので、再帰的に保持されない

	const std::string& element( getTopElement() );
	if( element == "or" ) {
		startDependLogicAND(attributes);
	} else if( element == "and" ) {
		startDependLogicOR(attributes);
	}
}

void BundleParser::endElementHandler(const std::string& name)
{
	const std::string& element( getTopElement() );
	if( element == "or" ) {
		endDependLogicAND();
	} else if ( element == "and" ) {
		endDependLogicOR();
	}

	m_path.pop_back();
}

void BundleParser::textHandler(const std::string& text)
{
	// *m_path.begin()は常に<bundle>
	const std::string& element( getTopElement() );

	if( m_path.size () == 2 ) {
		if( element == "name" ) {
			m_result->m_name = text;

		} else if( element == "arch" ) {
			m_result->m_arch = text;

		} else if( element == "version" ) {
			// XXX: 未実装
			// XXX: 例外処理？
			m_result->m_version = Version(text);

		} else if( element == "release" ) {
			m_result->m_release = text;

		} else if( element == "summary" ) {
			// XXX: 未実装

		} else if( element == "description" ) {
			// XXX: 未実装
		}


	} else if( m_path.size() > 2 ) {
		const std::string& bottom_element( getBottomElement() );

		if( bottom_element == "depend" ) {
			startDependContext(text);
		} else if ( bottom_element == "summary" ) {
		} else if ( bottom_element == "port" ) {
		} else if ( bottom_element == "description" ) {
		} else if ( bottom_element == "parameters" ) {
		}
	}
}

void BundleParser::errorHandler(const std::string& message, unsigned long long line, unsigned long long column)
{
	// XXX:
	std::cout << "XML Parse Error " << message << std::endl;
}


// private:
std::list<std::string> BundleParser::getAttrValue(const std::string& key) const
{
	std::list<std::string> vals;
	for( SAX::attributes_t::const_iterator att( m_current_atts.begin() );
			att != m_current_atts.end();
			++att ) {
		if( att->first == key ) {
			vals.push_back( att->second );
		}
	}
	return vals;
}


// dependency parser
void BundleParser::startDependLogicAND(const attributes_t& attributes)
{
	BDLogic* current_logic = *(m_logic_path.rbegin());
	BDLogicAND diland;
	BDLogicAND* latest = current_logic->addTarget(diland);
	m_logic_path.push_back(latest);
}
void BundleParser::startDependLogicOR(const attributes_t& attributes)
{
	BDLogic* current_logic = *(m_logic_path.rbegin());
	BDLogicOR dilor;
	BDLogicOR* latest = current_logic->addTarget(dilor);
	m_logic_path.push_back( latest );
}
void BundleParser::endDependLogicAND(void)
{
	m_logic_path.pop_back();
}
void BundleParser::endDependLogicOR(void)
{
	m_logic_path.pop_back();
}
void BundleParser::startDependContext(const std::string& text)
{
	const std::string& element( getTopElement() );
	BDLogic* current_logic = *(m_logic_path.rbegin() );

	if( element == "bundle" ) {
		// <bundle>
		BDTBundle::order_t order = BDTBundle::ORDER_POST;	// XXX: デフォルトorderはどこかに定数を定義
		std::list<std::string> attrs_order( getAttrValue("order") );
		if( ! attrs_order.empty() ) {
			const std::string& order_text( *(attrs_order.rbegin()) );
			if( order_text == "pre" ) {
				order = BDTBundle::ORDER_PRE;
			} else if ( order_text == "post" ) {
				order = BDTBundle::ORDER_POST;
			} else {
				// XXX: エラーを出すべきか？
			}
		}
		// XXX: バージョン周り未実装
		BDTBundle d(text, VersionRange("0.0"), order);
		current_logic->addTarget(d);

	} else if( element == "file" ) {
		// <file>
		BDTHostFile f(text);
		current_logic->addTarget(f);

	} else if( element == "lib" ) {
		// <lib>
		BDTHostLib l(text);
		current_logic->addTarget(l);

	} else if( element == "bin" ) {
		// <bin>
		BDTHostBin b(text);
		current_logic->addTarget(b);
	}
}
#endif


}  // namespace RUNES

