/***********************************************************************
 * VirtualDub Modification for OGM
 *
 * Copyright (C) 2002 Cyrius
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *   
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *   
 * You should have received a copy of the GNU General Public License along
 * with this program (see the file COPYING); if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * or visit http://www.gnu.org/copyleft/gpl.html
 *
 ***********************************************************************
 *
 *
 *
 */

#include "OGMSerialGenerator.h"

/*
 * Constructor
 */
SerialGenerator::SerialGenerator(int type, bool force) {
	registered_serials = NULL;
	creation_type = type;
	force_creation = force;
}

/*
 * Destructor
 */
SerialGenerator::~SerialGenerator(void) {
	if(registered_serials) {
		int_mesh *next = NULL;
		while(registered_serials) {
			next = registered_serials->next;
			delete registered_serials;
			registered_serials = next;
		}
	}
}

/*
 * Create a unique serial (according to all the serials already registered)
 */
int SerialGenerator::create_unique_serial(int serial) {
	int unique_serial = 0;
	if(creation_type == T_RANDOM)
		unique_serial = rand();

	if( (!force_creation) && (serial>=0) && !serial_in_use(serial)) {
		// The provided serial is not yet registered, so why don't use it ?
		unique_serial = serial;
	} else {
		// Loop until we find an unique one
		if(creation_type == T_RANDOM) {
			while(serial_in_use(unique_serial))
				unique_serial = rand();
		} else {
			while(serial_in_use(unique_serial))
				unique_serial++;
		}
	}
	return unique_serial;
}

/*
 * Add a serial to the list of registered ones
 */
void SerialGenerator::register_serial(int serial) {
	if(serial<0)
		return;

	if(registered_serials) {
		int_mesh *mesh = registered_serials;
		if(mesh->num == serial)
			return;
		while(mesh->next) {
			mesh = mesh->next;
			if(mesh->num == serial)
				return;
		}
		// The serial have not already been registered, so let's register it
		mesh->next = new int_mesh;
		if(!mesh->next)
			throw MyMemoryError();
		mesh->next->num = serial;
	} else {
		// No serial have been registered yet, this will be the first one
		registered_serials = new int_mesh;
		if(!registered_serials)
			throw MyMemoryError();
		registered_serials->num = serial;
	}
}

/*
 * Remove a serial from the list of registered ones
 */
void SerialGenerator::unregister_serial(int serial) {
	if( (serial<0) || (!registered_serials) )
		return;

	int_mesh *previous_mesh = NULL;
	int_mesh *mesh = registered_serials;
	while(mesh) {
		if(mesh->num == serial) {
			if(previous_mesh) {
				previous_mesh->next = mesh->next;
			} else {
				registered_serials = mesh->next;
			}
			// Free allocated memory for the serial that have been unregistered
			delete mesh;
			return;
		} else {
			previous_mesh = mesh;
			mesh = mesh->next;
		}
	}
}

/*
 * Tell if a serial is registered or not
 */
bool SerialGenerator::serial_in_use(int serial) {
	if(serial < 0)
		return true;

	int_mesh *mesh = registered_serials;
	while(mesh) {
		if(mesh->num == serial)
			return true;
		mesh = mesh->next;
	}

	return false;
}