require 'dom-utility'
require 'extension/string'

module Wiki
	class WikiStyle
		#*******************************************************************
		# Format Rule
		#*******************************************************************
		#-------------------------------------------------------------------
		# Base Regexp Definitions
		#-------------------------------------------------------------------
		TX_WORD          = "A-Za-z0-9_"
		TX_IDENT         = "[A-Za-z][#{TX_WORD}]*"
		TX_PATH          = "[#{TX_WORD}/]+"
		TX_NOWORD        = "[^#{TX_WORD}]"
		TX_MARK          = "\\#\\$%&\\-=~@;\\+:\\./\\?\\\\"
		TX_PROT          = "(?:http://|https://|ftp://|mailto:|news://|file://)"
		TX_IMGEXT        = "(?i:\.jpg|\.jpeg|\.png|\.gif)"
		TX_WIKINAME      = "(?:[A-Z][a-z][a-z0-9]*[A-Z][a-z0-9]+[A-Za-z0-9]*)"
		TX_WIKINAMES     = "(?:#{TX_WIKINAME}(?:/#{TX_WIKINAME})*)"
		TX_BRACKETNAME   = "(?:[^\\:\\x5b\\s]+)"
		TX_WRIT          = "(?:#{TX_PATH}\\:)*#{TX_WIKINAMES}(?:\\:#{TX_PATH})*(?:\\##{TX_IDENT})?"
		TX_WRIX          = "#{TX_BRACKETNAME}(?:\\:#{TX_BRACKETNAME})*(?:\\##{TX_IDENT})?"
		TX_URL           = "#{TX_PROT}[#{TX_WORD}#{TX_MARK}]+"
		TX_INURL         = "(?:#{TX_PROT})?[#{TX_WORD}#{TX_MARK}]+"
		TX_NOBLANK       = "\\S+"
		TX_NOPARE_S      = "[^)]+"
		TX_NOPARE_M      = "[^}]+"
		TX_NOPARE_L      = "[^\\x5b]+"
		TX_NOQUOTE       = "[^']+"
		TX_NODBLQ        = "[^\\\"]+"

		#-------------------------------------------------------------------
		# Block
		#-------------------------------------------------------------------

		# <pre>
		#   space
		FM_PRE           = ' %s'
		RE_PRE           = /^\s(.*)$/

		# <h1> to <h6>
		#   !,!!,!!!,!!!!,!!!!!
		FM_H1            = nil
		FM_H2            = '!%s'
		FM_H3            = '!!%s'
		FM_H4            = '!!!%s'
		FM_H5            = '!!!!%s'
		FM_H6            = '!!!!!%s'
		RE_H1            = nil
		RE_H2            = /^!{1}([^!].*)$/
		RE_H3            = /^!{2}([^!].*)$/
		RE_H4            = /^!{3}([^!].*)$/
		RE_H5            = /^!{4}([^!].*)$/
		RE_H6            = /^!{5}([^!].*)$/

		# <hr />
		#   ----
		FM_HR            = '----'
		RE_HR            = /^----+$/

		# <ul>
		#   *,**,***,...
		TX_UL            = '*'
		RE_UL            = /^(\*+)(.*)$/

		# <ol>
		#   #,##,###,...
		TX_OL            = '#'
		RE_OL            = /^(\#+)(.*)$/

		# <dl>
		#   :xxx:yyy,::xxx:yyy,:::xxx:yyy,...
		TX_DT            = ':'
		TX_DD            = ':'
		RE_DL            = /^(\:+)([^\:]+)\:(.*)$/

		# <blockquote>
		#   >,>>,>>>,...
		TX_BLOCKQUOTE    = '>'
		RE_BLOCKQUOTE    = /^(>+)(.*)$/

		# <table>
		# ,td,td,...
		TX_TABLE         = ','
		RE_TABLE         = /^\,(.*)$/

		#-------------------------------------------------------------------
		# Inline
		#-------------------------------------------------------------------

		# WikiName
		FM_WIKINAME      = '%s'
		RE_WIKINAME      = /^(#{TX_WRIT})/

		# BracketName
		#   [[...]]
		FM_BRACKETNAME   = '[[%s]]'
		RE_BRACKETNAME   = /^\[\[(#{TX_WRIX})\]\]/

		# AliasName
		#   [[WRI alias]]]
		FM_ALIASNAME     = '[[%s %s]]'
		RE_ALIASNAME     = /^\[\[(#{TX_WRIX})\s+(#{TX_NOPARE_L})\]\]/

		# <em>
		#   '''...'''
		FM_EM            = "'''%s'''"
		RE_EM            = /^\'{3}(#{TX_NOQUOTE})\'{3}/

		# <strong>
		#   ''...''
		FM_STRONG        = "''%s''"
		RE_STRONG        = /^\'{2}(#{TX_NOQUOTE})\'{2}/

		# <del>
		#   ==...==
		FM_DEL           = '==%s=='
		RE_DEL           = /^={2}([^=]+)={2}/

		# <a>
		#   [URL alias]
		FM_A             = '[%s %s]'
		RE_A             = /^\[(#{TX_NOBLANK})\s+(#{TX_NOPARE_L})\]/

		# URL
		FM_URL           = '%s'
		RE_URL           = /^(#{TX_URL})/

		# word
		RE_WORD          = /\A([#{TX_WORD}]+)/
		RE_LETTER        = /\A(.|\n)/

		#-------------------------------------------------------------------
		# Plugin (VikiWiki style)
		#   $1 => name, $2 => ident, $3 => parameters
		#-------------------------------------------------------------------

		# plugin
		FM_PLUGIN        = '#%s%s%s'
		RE_PLUGIN        = /^\#\s*(#{TX_IDENT})(?:\#(\S+))?(?:\s+(.+))?$/

		# plugin (inline)
		FM_INPLUGIN      = '#{%s%s%s}'
		RE_INPLUGIN      = /^\#\{(#{TX_IDENT})(?:\#(\S+))?(?:\s+(#{TX_NOPARE_M}))?\}/

		# plugin parameters
		FM_PLUGIN_PRM    = " %s"
		RE_PLUGIN_PRM    = /^(?:"(?:[^\\]*\\.)*(?:[^"]*)"|'(?:[^\\]*\\.)*(?:[^']*)')/
		FM_PLUGIN_HDC    = "<<%s"
		RE_PLUGIN_HDC    = /^<<(#{TX_IDENT})/
		FM_PLUGIN_SEP    = " "
		RE_PLUGIN_SEP    = /\s+/

		#-------------------------------------------------------------------
		# Others
		#-------------------------------------------------------------------
		# indent
		RE_INDENT        = /^(\s*)/

		#*******************************************************************
		# Definitions
		#*******************************************************************
		# Definitions for parser and generator
		BLOCK_PARSING_ORDER = [
			'h6','h5','h4','h3','h2',
			'pre','blockquote','table',
			'hr','ul','ol','dl',
			'plugin'
		]
		INLINE_PARSING_ORDER = [
			'wikiname','bracketname','aliasname','url',
			'em','strong','del',
			'plugin'
		]
		ELEMENT_GENERATING_TAGS = [
			'h6','h5','h4','h3','h2',
			'pre','blockquote','p',
			'hr','ul','ol','dl',
			'plugin',
			'wikiname',
			'em','strong','del',
			'plugin'
		]

		# Default settings
		INDENT_STYLE = false
		TOP_LEVEL_BREAK = false
		ALIAS_FORWARD = false
		DEFAULT_CONF = {
			'TABSIZE' => 4,
			'HNTOP'   => 2,
			'MAXLEVEL'=> 3,
		}

		#*******************************************************************
		# Public Methods
		#*******************************************************************
		#===================================================================
		# Method Name   : initialize
		# Explanations  : Initialize a style object.
		# Return values : style object
		#===================================================================
		def initialize(sys=nil, conf={})
			return if @sys
			@sys = sys
			@my = self.class
			@my_consts = @my.constants
			@block_root = Array::new
			@block_parsing_order = expand_order(@my::BLOCK_PARSING_ORDER)
			@inline_parsing_order = expand_order(@my::INLINE_PARSING_ORDER)
			@conf = @my::DEFAULT_CONF.dup
			if conf and conf.respond_to?(:[]) then
				@conf.keys.each do |key|
					@conf[key] = conf[key] if conf[key]
				end
			end
			@re_wri_href = /^#{Regexp::escape(@sys['SCRIPT_NAME'])}(?:\?p=([^&;]+))?$/ if @sys and @sys['SCRIPT_NAME']
		end
		#===================================================================
		# Method Name   : parse
		# Explanations  : Wiki text parser
		# Parameters    : root - root Element object
		#                 text - Wiki formatted text
		# Return values : root Element object
		#===================================================================
		def parse(root, text)
			raise "duplicate function call for parse" if @parse_call
			initialize(@sys, @conf)
			@parse_call = true
			begin
				parse_block(root, text.to_a)
			ensure
				@parse_call = false
			end
			return root
		end
		#===================================================================
		# Method Name   : generate
		# Explanations  : Wiki text generator
		# Parameters    : node - Element object
		#                 outobj - object to output with method <<
		# Return values : output object
		#===================================================================
		def generate(node, outobj='')
			raise "duplicate function call for generator" if @generate_call
			initialize(@sys, @conf)
			@generate_call = true
			begin
				generateBlock(node, outobj)
			ensure
				@generate_call = false
			end
			return outobj
		end
		#===================================================================
		# Method Name   : parse_block
		# Explanations  : Wiki text parser for block
		# Parameters    : root - root Element object
		#                 lines - array of Wiki formatted text
		#                 endtag - Regexp object to detect the end of block
		# Return values : root Element object
		#===================================================================
		def parse_block(root, lines, endtag=nil)
			@endtag_match = nil
			@block_root.push(root)
			nelm = root
			ptxt = Array::new
			pindent = 0
			while line = lines.shift do
				# Split indent and sentence
				sentence, indent = split_indent(line)
				# Move up to same level
				if endtag === sentence then
					@endtag_match = $~
					nelm = appendParagraph(nelm, ptxt, pindent)
					break
				elsif sentence.empty? then
					nelm = appendParagraph(nelm, ptxt, pindent)
					nelm = @block_root.last if @my::TOP_LEVEL_BREAK
					next
				elsif pindent != indent then
					nelm = appendParagraph(nelm, ptxt, pindent)
				end
				match, method = match_method(@block_parsing_order, sentence)
				if method then
					nelm = appendParagraph(nelm, ptxt, pindent)
					nelm = method.call(nelm, match, line, indent, lines)
				else
					ptxt << sentence
				end
				pindent = indent
			end
			appendParagraph(nelm, ptxt, pindent)
			@block_root.pop
			return root
		end
		#===================================================================
		# Method Name   : parse_inline
		# Explanations  : Inline based parser.
		# Parameters    : root - root Element object
		#                 lines - Wiki formatted inline text
		# Return values : node and next string
		#===================================================================
		def parse_inline(root, lines)
			return root if lines.nil? or lines.empty?
			ptxt = ''
			lines.each do |line|
				until line.empty? do
					match, method = match_method(@inline_parsing_order, line)
					if method then
						root.appendTextSimply(ptxt)
						ptxt = ''
						method.call(root, match)
					elsif @my::RE_WORD === line || @my::RE_LETTER === line then
						ptxt << $1
						match = $~
					else
						break
					end
					line = match.post_match
				end
			end
			root.appendTextSimply(ptxt.chomp) unless ptxt.empty?
			return root
		end
		#*******************************************************************
		# Common Parser's Methods
		#*******************************************************************
		#===================================================================
		# Method Name   : appendChild_block
		# Explanations  : Append child node for block
		# Parameters    : root - root Element object
		#                 name - tag name
		#                 attr - attributes (Hash)
		#                 text - unparsed text
		#                 indent - indent size
		#                 mode - appending mode
		#                        APPEND_CHILD
		#                          append normally
		#                        APPEND_IF_NOT_EXIST
		#                          append a new node if not exist
		#                        APPEND_TO_SAME_LEVEL
		#                          append to the same level
		# Return vlaues : appended Element object
		#===================================================================
		APPEND_TO_SAME_LEVEL = 'same'
		APPEND_IF_NOT_EXIST = 'none'
		APPEND_CHILD = 'child'
		def appendChild_block(root, name, attr, text, indent, mode=APPEND_TO_SAME_LEVEL)
			case mode
			when APPEND_IF_NOT_EXIST then
				root, curlevel = getSameLevelAncestor(root, name, indent)
				return root if root.nodeName == name and curlevel == indent
			when APPEND_TO_SAME_LEVEL then
				root, curlevel = getSameLevelAncestor(root, name, indent)
			end
			nelm = root.createElementSimply(name, attr, nil, true)
			nelm.setAttribute('wiki-indent-level', indent.to_s)
			nelm.setAttribute('wiki-node-type', 'block')
			parse_inline(nelm, text) if text
			return nelm
		end
		#===================================================================
		# Method Name   : appendChild_inline
		# Explanations  : Append child node for inline
		# Parameters    : root - root Element object
		#                 name - tag name
		#                 attr - attributes (Hash)
		#                 text - unparsed text
		# Return vlaues : appended Element object
		#===================================================================
		def appendChild_inline(root, name, attr=nil, text=nil)
			nelm = root.createElementSimply(name, attr, text, true)
			nelm.setAttribute('wiki-node-type', 'inline')
			return nelm
		end
		#===================================================================
		# Method Name   : appendParagraph
		# Explanations  : Append paragraph
		# Parameters    : root - root Element object
		#                 text - unparsed text
		#                 indent - indent size
		# Return vlaues : root node of appended Element object
		#===================================================================
		def appendParagraph(root, text, indent)
			return root if text.empty?
			nelm = appendChild_block(root, 'p', nil, text.join, indent)
			text.clear
			return nelm.parentNode
		end
		#===================================================================
		# Method Name   : getSameLevelAncestor
		# Explanations  : Search the same level ancestor node.
		#                 If not exist, it returns a node which is greater
		#                 than the specified level.
		# Parameters    : node - starting node
		#                 name - node name
		#                 level - indent level
		#                 upper - upper bound node
		# Return vlaues : node and level
		#===================================================================
		def getSameLevelAncestor(node, name, level, upper=@block_root.last)
			prev = node
			curlevel = 0
			while node do
				curlevel = node.respond_to?(:getAttribute) ? \
					node.getAttribute('wiki-indent-level').to_i : 0
				case node
				when upper then
					return node, curlevel
				when Element then
					curlevel = node.getAttribute('wiki-indent-level').to_i
					if curlevel > level then
					elsif curlevel < level then
						return node, curlevel
					else
						return node, curlevel if name.nil? or node.nodeName == name
					end
					prev = node
				when Text, Comment then
					break
				end
				node = node.parentNode
			end
			return prev, curlevel
		end
		#===================================================================
		# Method Name   : expnd_order
		# Explanations  : Expand order strings to regexps and methods.
		# Parameters    : order - array of order strings
		# Return vlaues : array of regexps and methods
		#===================================================================
		def expand_order(order)
			order.map do |name|
				regname = "RE_#{name.upcase}"
				if @my_consts.include?(regname+"_S") then
					reg = @my.const_get(regname+"_S")
				elsif @my_consts.include?(regname) then
					reg = @my.const_get(regname)
				else
					reg = nil
				end
				methname = "#{name.downcase}_onparse"
				if @sys.respond_to?(methname) then
					meth = @sys.method(methname)
				elsif self.respond_to?(methname) then
					meth = self.method(methname)
				end
				[reg, meth]
			end
		end
		#===================================================================
		# Method Name   : split_indent
		# Explanations  : Split sentence deleted indent and size of indent
		# Parameters    : line - one line text
		# Return vlaues : sentence deleted indent and size of indent
		#===================================================================
		def split_indent(line)
			return line, 0 unless @my::INDENT_STYLE
			return line, 0 unless @my::RE_INDENT === line
			sentence, indent = $', $1
			size = 0
			indent.each_byte do |ch|
				size += 1
				size = (size+@conf['TABSIZE']-1)/@conf['TABSIZE']*@conf['TABSIZE'] if ch.chr == "\t"
			end
			return sentence, size
		end
		#===================================================================
		# Method Name   : get_here_document
		# Explanations  : Get here document with mark (Imitation of Ruby)
		# Parameters    : lines - array of text
		#                 mark - end of here document
		# Return vlaues : text of here document
		#===================================================================
		def get_here_document(lines, mark)
			doc = ''
			until lines.empty? do
				line = lines.shift
				break if line.chomp == mark
				doc << line
			end
			return doc
		end
		#===================================================================
		# Method Name   : match_method
		# Explanations  : Select element from array of order.
		# Parameters    : order - array of order
		#                 line - line to be parsed
		# Return vlaues : match data and method
		#===================================================================
		def match_method(order, line)
			order.each do |reg, meth|
				next if reg.nil? or meth.nil?
				return $~, meth if reg === line
			end
			return nil
		end
		#===================================================================
		# Method Name   : generateBlock
		# Explanations  : Wiki text generator
		# Parameters    : node - Element object
		#                 outobj - object to output with method <<
		# Return values : output object
		#===================================================================
		def generateBlock(node, outobj='')
			case node
			when nil then
				return outobj
			when NodeList then
				list = node
				0.upto(list.length-1) do |i|
					generateBlock(list.item(i), outobj)
				end
				return outobj
			end
			name = node.nodeName.downcase.gsub(/[^a-zA-Z0-9]/, '_')
			meth = name + '_ontext'
			if @my.method_defined?(meth) then
				if not Element === node then
					method(meth).call(node, outobj)
				else
					if @my::ELEMENT_GENERATING_TAGS.include?(name) then
						method(meth).call(node, outobj)
					else
						generateBlock(node.childNodes, outobj)
						outobj << "\n"
					end
				end
			else
				generateBlock(node.childNodes, outobj)
			end
			return outobj
		end
		def get_alias_uri(prm1, prm2)
			# alias, uri/wri
			return prm1, prm1 unless prm2
			return prm2, prm2 unless prm1
			return prm1, prm2 if @my::ALIAS_FORWARD
			return prm2, prm1
		end
		#*******************************************************************
		# Common Generator's Methods
		#*******************************************************************
		#===================================================================
		# Method Name   : generateChild
		# Explanations  : Generate child nodes
		# Parameters    : node - Element object
		#                 outobj - object to output with method <<
		# Return values : output object
		#===================================================================
		def generateChild(node, outobj='')
			generateBlock(node.childNodes, outobj)
			return outobj
		end
		#===================================================================
		# Method Name   : generateFirst
		# Explanations  : Generate first node
		# Parameters    : node - Element object
		#                 outobj - object to output with method <<
		# Return values : output object
		#===================================================================
		def generateFirst(node, outobj='')
			ncur = node.firstChild
			generateBlock(ncur, outobj)
			return outobj, ncur.nextSibling
		end
		def generateInline(node, outobj='')
			ncur = node.firstChild
			while ncur do
				break if Element === ncur and ncur.getAttribute('wiki-node-type') == 'block'
				generateBlock(ncur, outobj)
				ncur = ncur.nextSibling
			end
			return outobj, ncur
		end
		def generateInlineText(node, outobj='')
			text, ncur = generateInline(node, outobj)
			return text
		end
		def generateSurplus(node, outobj='')
			ncur = node
			while ncur do
				generateBlock(ncur, outobj)
				ncur = ncur.nextSibling
			end
			return outobj
		end
		def indent_line(node)
			size = node.getAttribute('wiki-indent-level')
			size = @nest if size.to_s.empty?
			return ' ' * size.to_i
		end
		def get_align(node)
			return nil unless Element === node
			align = node.getAttribute('align')
			return align if /^left|right|center$/ === align
			style = node.getAttribute('style').to_s.gsub(/\s/,'')
			return $1 if /text-align:(left|right|center);/ === style
			return nil
		end
		def split_if_same(node, outobj='')
			if nprev = node.previousSibling and
				nprev.nodeName == node.nodeName then
				outobj << "\n"
			end
			return outobj
		end
		def pre_eql?(node)
			node.previousSibling and node.previousSibling.nodeName == node.nodeName
		end
		def text_format(fmt, *prms)
			return prms.join(' ') unless fmt
			begin
				return fmt % prms.flatten
			rescue ArgumentError
				raise "#{$!} #{fmt} % #{prms.inspect}"
			end
		end
		#*******************************************************************
		# Block Parser
		#*******************************************************************
		#===================================================================
		# Method Name   : xxx_onparse
		# Explanations  : xxx parser.
		# Parameters    : root - root Element object
		#                 match - Match object
		#                 line - currect line
		#                 indent - indent level
		#                 lines - surplus lines
		# Return values : Node object
		#===================================================================
		def hn_onparse(root, match, line, indent, lines, level=1)
			level += @conf['HNTOP'] - @my::DEFAULT_CONF['HNTOP']
			level = 1 if level < 1
			level = 6 if level > 6
			hn = "h%d" % level
			nhn = appendChild_block(root, hn, nil, match[1], indent)
			return nhn.parentNode
		end
		def h6_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 6)
		end
		def h5_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 5)
		end
		def h4_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 4)
		end
		def h3_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 3)
		end
		def h2_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 2)
		end
		def h1_onparse(root, match, line, indent, lines)
			hn_onparse(root, match, line, indent, lines, 1)
		end
		def hr_onparse(root, match, line, indent, lines)
			nhr = appendChild_block(root, 'hr', nil, nil, indent)
			return nhr.parentNode
		end
		def pre_onparse(root, match, line, indent, lines)
			text = match[1]+"\n"
			while @my::RE_PRE === lines.first do
				text << $1+"\n"
				lines.shift
			end
			npre = appendChild_block(root, 'pre', nil, nil, indent)
			npre.appendTextSimply(text)
			return npre.parentNode
		end
		def li_onparse(root, match, line, indent, lines, tag='ul')
			depth = match[1].size - 1
			item = match[2]
			nlist = appendChild_block(root, tag, nil, nil, depth, APPEND_IF_NOT_EXIST)
			nli = appendChild_block(nlist, 'li', nil, item, depth, APPEND_CHILD)
			return nli
		end
		def ul_onparse(root, match, line, indent, lines)
			li_onparse(root, match, line, indent, lines, 'ul')
		end
		def ol_onparse(root, match, line, indent, lines)
			li_onparse(root, match, line, indent, lines, 'ol')
		end
		def dl_onparse(root, match, line, indent, lines)
			depth = match[1].size - 1
			dt, dd = match[2], match[3]
			root = appendChild_block(root, 'dl', nil, nil, depth, APPEND_IF_NOT_EXIST)
			appendChild_block(root, 'dt', nil, dt, depth, APPEND_CHILD)
			ndd = appendChild_block(root, 'dd', nil, dd, depth, APPEND_CHILD)
			return ndd
		end
		def blockquote_onparse(root, match, line, indent, lines)
			depth = match[1].size - 1
			text = match[2]+"\n"
			while @my::RE_BLOCKQUOTE === lines.first and depth == $1.to_i do
				text << $2+"\n"
				lines.shift
			end
			nbq = appendChild_block(root, 'blockquote', nil, text, depth, APPEND_CHILD)
			return nbq
		end
		def table_onparse(root, match, line, indent, lines)
			tr = match[1]
			ntable = appendChild_block(root, 'table', {'border'=>'1'}, nil, indent)
			loop do
				ntr = appendChild_block(ntable, 'tr', nil, nil, indent, APPEND_CHILD)
				tr.split(TX_TABLE).each do |td|
					appendChild_block(ntr, 'td', nil, td, indent, APPEND_CHILD)
				end
				break unless @my::RE_TABLE === lines.first
				tr = $1
				lines.shift
			end
			return ntable
		end
		#*******************************************************************
		# Inline Parser
		#*******************************************************************
		#===================================================================
		# Method Name   : xxx_onparse
		# Explanations  : xxx parser.
		# Parameters    : root - root Element object
		#                 match - Match object
		# Return values : Node object
		#===================================================================
		def wikiname_onparse(root, match, bracket=false)
			text, wri = get_alias_uri(match[1], match[2])
			appendChild_inline(root, 'WikiName', {'wri'=>wri, 'bracket'=>bracket}, text)
		end
		def bracketname_onparse(root, match)
			wikiname_onparse(root, match, true)
		end
		def aliasname_onparse(root, match)
			wikiname_onparse(root, match, true)
		end
		def a_onparse(root, match)
			text, href = get_alias_uri(match[1], match[2])
			appendChild_inline(root, 'a', {'href'=>href}, text)
		end
		def url_onparse(root, match)
			appendChild_inline(root, 'a', {'href'=>match[1]}, match[1])
		end
		def img_onparse(root, match)
			alt, src = get_alias_uri(match[1], match[2])
			appendChild_inline(root, 'img', {'src'=>src, 'alt'=>alt})
		end
		def em_onparse(root, match)
			appendChild_inline(root, 'em', nil, match[1])
		end
		def strong_onparse(root, match)
			appendChild_inline(root, 'strong', nil, match[1])
		end
		def del_onparse(root, match)
			appendChild_inline(root, 'del', nil, match[1])
		end
		def inplugin_onparse(root, match)
			plugin_onparse(root, match, match[0], nil, nil, true)
		end
		#*******************************************************************
		# Plugin Parser
		#*******************************************************************
		def plugin_onparse(root, match, line, indent, lines, inline=false)
			attr = {'name' => match[1], 'plain'=>line.chomp, 'id'=>match[2]}
			if inline then
				nplugin = appendChild_inline(root, 'plugin', attr)
			else
				nplugin = appendChild_block(root, 'plugin', attr, nil, indent)
			end
			plugin_onparse_prm(nplugin, match[3], lines, inline)
			return root
		end
		def plugin_onparse_prm(root, prmstr, lines, inline)
			return unless prmstr
			until prmstr.empty? do
				str, prmstr = parse_expression(prmstr, @my::RE_PLUGIN_SEP)
				str = $1.gsub(/\\./){eval(%Q["#{$&}"].untaint)} if /^\"(.*)\"$/ === str
				attr = nil
				if not inline and @my::RE_PLUGIN_HDC === str then
					mark = $1
					str = get_here_document(lines, mark)
					attr = {'mark' => mark}
				end
				root.createElementSimply('param', attr, str, true)
			end
			return root
		end
		def parse_expression(prmstr, sep)
			case prmstr
			when @my::RE_PLUGIN_PRM then
				prm = $&
				prmstr = sep === $' ? $' : ''
			when sep then
				prm = $`
				prmstr = $'
			else
				prm = prmstr
				prmstr = ''
			end
			return prm, prmstr
		end
		#*******************************************************************
		# Wiki Text Block Generator
		#*******************************************************************
		#===================================================================
		# Method Name   : xxx_ontext
		# Explanations  : xxx generator.
		# Parameters    : node - Node object
		#                 outobj - output object
		# Return values : output object
		#===================================================================
		def _document_ontext(node, outobj='')
			generateBlock(node.documentElement, outobj)
		end
		def _document_fragment_ontext(node, outobj='')
			generateBlock(node.childNodes, outobj)
		end
		def _text_ontext(node, outobj='')
			outobj << node.nodeValue
		end
		def pre_ontext(node, outobj='')
			generateChild(node).each do |line|
				outobj << ' ' + line
			end
			return outobj
		end
		def hn_ontext(node, outobj='', level=1)
			level += @conf['HNTOP'] - @my::DEFAULT_CONF['HNTOP']
			level = 1 if level < 1
			level = 6 if level > 6
			format = @my::const_get("FM_H%d" % level)
			format = "%s" unless format
			outobj << indent_line(node)
			outobj << format % generateChild(node)
			outobj << "\n"
			return outobj
		end
		def h1_ontext(node, outobj='')
			hn_ontext(node, outobj, 1)
		end
		def h2_ontext(node, outobj='')
			hn_ontext(node, outobj, 2)
		end
		def h3_ontext(node, outobj='')
			hn_ontext(node, outobj, 3)
		end
		def h4_ontext(node, outobj='')
			hn_ontext(node, outobj, 4)
		end
		def h5_ontext(node, outobj='')
			hn_ontext(node, outobj, 5)
		end
		def h6_ontext(node, outobj='')
			hn_ontext(node, outobj, 6)
		end
		def hr_ontext(node, outobj='')
			outobj << indent_line(node)
			outobj << text_format(@my::FM_HR)
			outobj << "\n"
			return outobj
		end
		def li_ontext(node, outobj='')
			size = node.getAttribute('wiki-indent-level').to_i + 1
			size = @conf['MAXLEVEL'] if @conf['MAXLEVEL'] and size > @conf['MAXLEVEL']
			mark = node.parentNode.nodeName == 'ol' ? @my::TX_OL : @my::TX_UL
			text, ncur = generateInline(node)
			text = ' ' + text if text[0].chr == mark
			outobj << mark * size + text
			outobj << "\n"
			generateSurplus(ncur, outobj)
			return outobj
		end
		def dl_ontext(node, outobj='')
			outobj << "\n" if pre_eql?(node)
			size = node.getAttribute('wiki-indent-level').to_i + 1
			size = @conf['MAXLEVEL'] if @conf['MAXLEVEL'] and size > @conf['MAXLEVEL']
			dt = ''
			cur = node.firstChild
			while cur do
				ctxt, ncur = generateInline(cur)
				case cur.nodeName
				when 'dt' then
					dt = ctxt
				when 'dd' then
					outobj << @my::TX_DT * size + dt
					outobj << @my::TX_DD + ctxt
					outobj << "\n"
					generateSurplus(ncur, outobj)
				end
				cur = cur.nextSibling
			end
			return outobj
		end
		def blockquote_ontext(node, outobj='')
			outobj << "\n" if pre_eql?(node)
			size = node.getAttribute('wiki-indent-level').to_i + 1
			size = @conf['MAXLEVEL'] if @conf['MAXLEVEL'] and size > @conf['MAXLEVEL']
			text, ncur = generateInline(node)
			text.each do |t|
				t = ' ' + t if t[0].chr == @my::TX_BLOCKQUOTE
				outobj << @my::TX_BLOCKQUOTE * size + t
			end
			if node.hasChildNodes then
				outobj << generateInlineText(node)
				outobj << "\n"
			end
			outobj << "\n"
			generateSurplus(ncur, outobj)
			return outobj
		end
		def table_ontext(node, outobj='')
			node.each_node('tr') do |ntr|
				tds = [nil]
				list = ntr.childNodes
				0.upto(list.length-1) do |i|
					item = list.item(i)
					tds << generateInlineText(item)
				end
				outobj << tds.join(@my::TX_TABLE) + "\n"
			end
			return outobj
		end
		def p_ontext(node, outobj='')
			split_if_same(node, outobj)
			outobj << generateChild(node)
			outobj << "\n"
		end
		#*******************************************************************
		# Wiki Text Inline Generator
		#*******************************************************************
		def wikiname_ontext(node, outobj='')
			if node.nodeName == 'a' then
				href = node.getAttribute('href')
				return nil unless @re_wri_href === href
				wri = $1
				wri = @sys['TOPPAGE'] if wri.nil? and @sys
				wri = 'FrontPage' unless wri
			else
				wri = node.getAttribute('wri')
			end
			bracket = node.getAttribute('bracket') == 'true'
			text = generateChild(node)
			if wri != text then
				outobj << text_format(@my::FM_ALIASNAME, get_alias_uri(text, wri))
			elsif bracket or not @my::RE_WIKINAME === wri then
				outobj << text_format(@my::FM_BRACKETNAME, wri)
			else
				outobj << text_format(@my::FM_WIKINAME, wri)
			end
			return outobj
		end
		def del_ontext(node, outobj='')
			outobj << text_format(@my::FM_DEL, generateChild(node))
			return outobj
		end
		def strong_ontext(node, outobj='')
			outobj << text_format(@my::FM_STRONG, generateChild(node))
			return outobj
		end
		def em_ontext(node, outobj='')
			outobj << text_format(@my::FM_EM, generateChild(node))
			return outobj
		end
		def a_ontext(node, outobj='')
			return outobj if wikiname_ontext(node, outobj)
			href = node.getAttribute('href')
			text = generateChild(node)
			if href == text then
				outobj << text_format(@my::FM_URL, href)
			else
				outobj << text_format(@my::FM_A, get_alias_uri(text, href))
			end
			return outobj
		end
		def img_ontext(node, outobj='')
			src = node.getAttribute('src')
			alt = node.getAttribute('alt')
			if src == alt then
				outobj << text_format(@my::FM_URL, src)
			else
				outobj << text_format(@my::FM_IMG, get_alias_uri(alt, src))
			end
			return outobj
		end
		#*******************************************************************
		# Wiki Text Plugin Generator
		#*******************************************************************
		def plugin_ontext(node, outobj='')
			name = node.getAttribute('name')
			id = node.getAttribute('id')
			id = text_format(@my::FM_PLUGIN_ID, id) unless id.to_s.empty?
			inline = node.getAttribute('wiki-node-type') == 'inline'
			prmstr = ''
			hdcs = Array::new
			list = node.childNodes
			0.upto(list.length-1) do |i|
				nprm = list.item(i)
				mark = nprm.getAttribute('mark')
				text = generateBlock(nprm)
				if mark.to_s.empty? then
					text = text.dump if /\s/ === text
				else
					hdcs[i] = [mark, text]
					text = text_format(@my::FM_PLUGIN_HDC, mark)
				end
				prmstr << text_format(@my::FM_PLUGIN_PRM, text)
			end
			if inline then
				outobj << text_format(@my::FM_INPLUGIN, name, id, prmstr)
			else
				outobj << text_format(@my::FM_PLUGIN, name, id, prmstr)
				outobj << "\n"
				hdcs.each do |mark, text|
					next unless mark
					outobj << text.chomp
					outobj << "\n"
					outobj << mark
					outobj << "\n"
				end
			end
			return outobj
		end
	end
end
