require 'ftools'
require 'difftool'
require 'lockfile'
require 'message'

class DiffCVS
	CVSDIR = '.cvs'
	DSTDIR = '.dst'
	def initialize(mode='400', backup=true, user=nil)
		@mode = String === mode ? mode.oct : mode
		@backup = backup
		@user = user
	end
	#===================================================================
	# Method Name   : readFile
	# Explanations  : Read file.
	# Parameters    : file - file path
	#                 ver - version
	# Return values : content of file
	#===================================================================
	def readFile(file, ver=nil)
		lines = File::readlines(file)
		return lines.join unless ver
		ver = ver.to_i
		vers = getVersions(file)
		return lines.join if vers.empty? or vers.last < ver
		return lines.join unless bakfile = getBackupFile(file)
		lines = File::readlines(bakfile).join
		vers.last.downto(ver) do |cur|
			diffile = getDiffFile(file, cur)
			break unless File::exist?(diffile)
			diflines = File::readlines(diffile)
			lines = DiffTool::detachDiff(lines, diflines)
		end
		return lines
	end
	#===================================================================
	# Method Name   : writeFile
	# Explanations  : Writing file.
	# Parameters    : file - file path
	#                 text - content to write
	#                 check - check timestamp
	#                 user - user name
	# Return values : none
	#===================================================================
	def writeFile(file, text=nil, check=nil, user=nil)
		File::mkpath(File::dirname(file))
		LockFile::lock(file) do
			begin
				cur = nil
				if File::exist?(file) then
					cur = File::stat(file).mtime
					raise Message::new(:FILE_ALREADY_MODIFIED) if check and cur != check
					File::chmod(@mode|0666, file)
				end
				if text then
					File::open(file, "w") do |f|
						f.write(text)
					end
				end
				if @backup then
					bakfile = getBackupFile(file)
					if File::exist?(bakfile) then
						vers = getVersions(file)
						ver = vers.last ? vers.last+1 : 0
						lastuser = /^#{Regexp::escape(File::basename(file))}\./ === File::basename(bakfile) ? $' : nil
						diffile = getDiffFile(file, ver, lastuser)
						if DiffTool::executeDiff(bakfile, file, diffile) then
							vers = getVersions(file)
							if Integer === @backup and vers.size > @backup then
								0.upto(vers.size-@backup-1) do |i|
									diffile = getDiffFile(file, vers[i])
									File::unlink(diffile)
								end
							end
						elsif cur then
							File::utime(cur, cur, file)
						end
						File::chmod(0666|@mode, bakfile)
						File::unlink(bakfile)
					end
					user = @user unless user
					bakfile = getBackupFile(file, user)
					File::open(bakfile, "w") do |f|
						File::foreach(file) do |line|
							f.write(line)
						end
					end
					mtime = File::mtime(file)
					File::utime(mtime, mtime, bakfile)
					File::chmod(@mode, bakfile)
				end
			ensure
				File::chmod(@mode, file) if File::exist?(file)
			end
		end
	end
	#===================================================================
	# Method Name   : renameFile
	# Explanations  : Renaming file.
	# Parameters    : from - file path
	#                 to - file path
	# Return values : none
	#===================================================================
	def renameFile(from, to)
		raise Message::new(:FILE_AREADY_EXISTS, to) if File::exist?(to)
		raise Message::new(:FILE_NOT_FOUND, from) unless File::exist?(from)
		renames = Array::new
		File::mkpath(File::dirname(to))
		LockFile::lock(from, to) do
			begin
				vers = getVersions(from)
				File::rename(from, to)
				renames << [from, to]
				from_bakfile = getBackupFile(from)
				if File::exist?(from_bakfile) then
					to_bakfile = getBackupFile(to)
					File::rename(from_bakfile, to_bakfile)
					renames << [from_bakfile, to_bakfile]
					vers.each do |ver|
						from_diffile = getDiffFile(from, ver)
						to_diffile = getDiffFile(to, ver)
						File::rename(from_diffile, to_diffile)
						renames << [from_diffile, to_diffile]
					end
				end
			rescue StandardError, ScriptError
				renames.each do |f1, f2|
					File::rename(f2, f1)
				end
				raise
			end
		end
	end
	#===================================================================
	# Method Name   : removeFile
	# Explanations  : Removing file.
	# Parameters    : file - file path
	# Return values : none
	#===================================================================
	def removeFile(file)
		raise Message::new(:FILE_NOT_FOUND, file) unless File::exist?(file)
		dir = File::dirname(file)
		vers = getVersions(file)
		vers.each do |ver|
			diffile = getDiffFile(file, ver)
			desertFile(dir, diffile)
		end
		bakfile = getBackupFile(file)
		desertFile(dir, bakfile)
		desertFile(dir, file)
	end
	#===================================================================
	# Method Name   : getDiff
	# Explanations  : Getting diff.
	# Parameters    : file - file path
	#                 ver - version
	# Return values : diff string
	#===================================================================
	def getDiff(file, ver)
		file = getDiffFile(file, ver)
		return nil unless file and File::exist?(file)
		return File::readlines(file).join
	end
	#===================================================================
	# Method Name   : getHistry
	# Explanations  : Getting history.
	# Parameters    : file - file path
	# Return values : all version information with Array
	#                 [version, file-name, update-time]
	#===================================================================
	def getHistory(file)
		dir, base = File::split(file)
		cvspath = "#{dir}/#{CVSDIR}/#{base}"
		reg = /^#{Regexp::escape(cvspath)}(?:\.(\w+))?\.(\d+)\.diff$/
		vers = Array::new
		@diffile = file
		@diffiles = Hash::new
		Dir::glob("#{cvspath}.*.diff") do |diffile|
			next unless reg === diffile
			diffile.untaint
			vers << [$2.to_i, $1, File::stat(diffile).mtime]
			@diffiles[$2.to_i] = diffile
		end
		return vers.sort
	end
	#===================================================================
	# Method Name   : getVersions
	# Explanations  : Getting all version.
	# Parameters    : file - file path
	# Return values : all version with Array
	#===================================================================
	def getVersions(file)
		getHistory(file).map{|his| his[0]}
	end

private
	def getDiffFile(file, ver, user=nil)
		user = @user unless user
 		ver.untaint if (/^\d+$/ === ver)
		return @diffiles[ver] if user.nil? and @diffile and @diffile == file and @diffiles[ver]
		dir, base = File::split(file)
		dir = "#{dir}/#{CVSDIR}"
		File::mkpath(dir)
		diffile = Dir::glob("#{dir}/#{base}*.#{ver}.diff").first
		diffile = ["#{dir}/#{base}", user, ver.to_s, "diff"].compact.join('.') if diffile.nil?
		return diffile.untaint
	end
	def getBackupFile(file, user=nil)
		dir, base = File::split(file)
		dir = "#{dir}/#{CVSDIR}"
		File::mkpath(dir)
		bakfile = nil
		Dir::glob("#{dir}/#{base}.*"){|fnam|
			next if /\.diff$/ === fnam
			bakfile = fnam
			break
		}
		unless bakfile then
			bakfile = "#{dir}/#{base}"
			bakfile << ".#{user}" if user
		end
		return bakfile.untaint
	end
	def desertFile(dir, file)
		raise "file path mismatch" unless /^#{Regexp::escape(dir)}/ === file
		base = $'
		dstfile = "#{dir}/#{DSTDIR}/#{base}"
		File::mkpath(File::dirname(dstfile))
		File::rename(file, dstfile) if File::exist?(file)
	end
end

if __FILE__ == $0 then

def Dir::rm_f(dir)
	Dir::foreach(dir) do |file|
		next if /^\.+$/ === file
		file = "#{dir}/#{file}"
		if File::directory?(file) then
			Dir::rm_f(file)
			Dir::unlink(file)
		else
			File::unlink(file)
		end
	end
	Dir::unlink(dir)
end

cvs = DiffCVS::new('666', true, ENV['USERNAME'])
loop do
	fout = STDOUT
	if ARGV.empty? then
		print "> "
		line, out = $`, $1 if /\>\s*(\S+)$/ === (line = gets.chomp)
		fout = File::open(out, "w") if out
		cmd, *prms  = line.split
	else
		cmd, *prms  = ARGV
	end
	case cmd
	when 'read' then
		ver = prms[1].to_i if prms[1]
		Dir::glob(prms[0]) do |file|
			next if File::directory?(file)
			next if File::expand_path(file).include?('/.')
			lines = cvs.readFile(file, ver)
			printf "<< file=%s, version=%s >>\n", file, ver.inspect
			fout.write lines
		end
	when 'write' then
		Dir::glob(prms[0]) do |file|
			next if File::directory?(file)
			next if File::expand_path(file).include?('/.')
			puts file
			cvs.writeFile(file)
		end
	when 'rename' then
		file1 = prms[0]
		file2 = prms[2]
		cvs.renameFile(file1, file2)
	when 'remove' then
		file = prms[0]
		cvs.removeFile(file)
	when 'history' then
		file = prms[0]
		vers = cvs.getHistory(file)
		vers.each do |ver, user, time|
			printf "<< version=%s, user=%s, time=%s >>\n", ver.inspect, user.inspect, time.strftime('%Y-%m-%d %H:%M:%S')
		end
	when 'clear' then
		print "Do you want to delete all backup files? [y,n]"
		if gets.chomp == 'y' then
			Dir::glob('**/.cvs **/.dst') do |dir|
				Dir::rm_f(dir)
			end
		end
	when 'exit' then
		break
	else
		system([cmd, prms].join(' '))
	end
	fout.close if fout != STDOUT
	if ARGV.empty? then
		puts "finished."
	else
		break
	end
end

end
