#
# vikiwikiio.rb: Wiki base IO class for VikiWiki format.
#
# if you want to store the page datas into VikiWiki system,
# add @style into tdiary.conf below:
#
#    require 'tdiary/vikiwikiio'
#    @io_class = TDiary::VikiwikiIO
#
# Copyright (C) 2003,2004, ASAI Etsuhisa <mopia@aqua.livedoor.com>
# You can distribute this under GPL.
#
require 'tdiary/vikiwikiconf'
require 'diffcvs'

module TDiary

	def TDiary::parse_tdiary( data )
		header, body = data.split( /\r?\n\r?\n/, 2 )
		headers = {}
		if header then
			header.each do |l|
				l.chomp!
				next unless VIKIWIKIFORMAT['HEAD_REGEXP'] === l
				key, val = $1, $2
				headers[key] = val ? val.chomp : nil
			end
		end
		[headers, body]
	end

	module CommentIO
		def comment_file( data_path, date )
			comment_file_replace( data_path, date )
		end
		def comment_file_replace( data_path, date, replace = nil )
			format = "#{VIKIWIKICONF['BASEDIR']}/text/#{VIKIWIKICONF['TDCPAGE']}.#{VIKIWIKICONF['STYLE_NAME']}Style.#{VIKIWIKICONF['FILEEXT']}"
			replace.each{|s,t| format.gsub!( s, t ) } if replace
			date.strftime( format )
		end

		def restore_comment( file, diaries )
			begin
				Dir::glob( file ) do |cfile|
					File::open( cfile.untaint, 'r' ) do |fh|
						body = ''
						date,  mail,  name,  last,  visible  = nil
						loop do
							l = fh.gets
							l = VIKIWIKICONV['RESTORE'].call( l ) if VIKIWIKICONV['RESTORE'] and l
							if l.nil? or VIKIWIKIFORMAT['CMT_HEAD_REGEXP'] === l then
								sdate, slast, smail, sname, svisible = \
								$1,    $2,    $3,    $4,    $5
								if date then
									date.gsub!( '-', '' )
									tlast = Time::local( *last.scan( /\d+/ ) )
									comment = Comment::new( name, mail, body, tlast )
									comment.show = false if visible == 'false'
									diaries[date].add_comment( comment )
								end
								break unless l
								date,  mail,  name,  last,  visible  = \
								sdate, smail, sname, slast, svisible
								body = ''
							elsif VIKIWIKIFORMAT['CMT_BODY_REGEXP'] === l then
								body << $1
							end
						end
					end
				end
			rescue Errno::ENOENT
			end
		end

		def store_comment( file, diaries )
			arr = Array::new
			cvs = DiffCVS::new( VIKIWIKICONF['FILEMODE'].oct, VIKIWIKICONF['BACKUP'] )
			diaries.each do |date,diary|
				diary.each_comment( diary.count_comments( true ) ) do |com|
					wdate = *date.scan( /^(\d{4})(\d{2})(\d{2})$/ ).join( '-' )
					wname = com.name
					wname = VIKIWIKICONV['STORE'].call( wname ) if VIKIWIKICONV['STORE']
					wmail = com.mail || ''
					wlast = com.date.strftime( "%Y-%m-%d %H:%M:%S" )
					wvisible = com.visible? ? 'true' : 'false'
					arr << VIKIWIKIFORMAT['CMT_HEAD_FORMAT'] %
						[ wdate, wlast, wmail, wname, wvisible ]
					l = com.body.gsub( /\r/, '' ).sub( /\n+\Z/, '' )
					l = VIKIWIKICONV['STORE'].call( l ) if VIKIWIKICONV['STORE']
					l.each do |i|
						arr << VIKIWIKIFORMAT['CMT_BODY_FORMAT'] % i
					end
				end
				if file.include?( '??' ) then
					cfile = comment_file_replace( @data_path, Time::local( *date.scan( /(\d{4})(\d\d)(\d\d)/ )[0] ) )
					cvs.writeFile( cfile, arr.join("\n") )
				end
			end
			cvs.writeFile( file, arr.join("\n") ) unless file.include?( '??' )
		end
	end

	module RefererIO
		def referer_file( data_path, date )
			referer_file_replace( data_path, date )
		end
		def referer_file_replace( data_path, date, replace=nil )
			format = "#{VIKIWIKICONF['BASEDIR']}/text/#{VIKIWIKICONF['TDRPAGE']}.#{VIKIWIKICONF['STYLE_NAME']}Style.#{VIKIWIKICONF['FILEEXT']}"
			replace.each{|s,t| format.gsub!( s, t ) } if replace
			date.strftime( format )
		end

		def restore_referer( file, diaries )
			begin
				Dir::glob( file ) do |cfile|
					File::open( cfile.untaint, 'r' ) do |fh|
						while l = fh.gets do
							next unless VIKIWIKIFORMAT['REF_REGEXP'] === l
							date, count, ref = $1, $2, $3
							diaries[date].add_referer( ref, count.to_i )
						end
					end
				end
			rescue Errno::ENOENT
			end
		end

		def store_referer( file, diaries )
			arr = Array::new
			cvs = DiffCVS::new( VIKIWIKICONF['FILEMODE'].oct, VIKIWIKICONF['BACKUP'] )
			diaries.each do |date,diary|
				diary.each_referer( diary.count_referers ) do |count,ref|
					arr << VIKIWIKIFORMAT['REF_FORMAT'] %
						[ date, count.to_s, ref ]
				end
				if file.include?( '??' ) then
					rfile = diary_file_replace( @data_path, Time::local( *date.scan( /(\d{4})(\d\d)(\d\d)/ )[0] ) )
					cvs.writeFile( rfile, arr.join("\n") )
				end
			end
			cvs.writeFile( file, arr.join("\n") ) unless file.include?( '??' )
		end
	end

	class VikiwikiIO < IOBase
		include CommentIO
		include RefererIO

		def initialize( tdiary )
			@tdiary = tdiary
			@data_path = @tdiary.conf.data_path
			load_styles
		end

		#
		# block must be return boolean which dirty diaries.
		#
		def transaction( date )
			diaries = {}
			@dfile = diary_file_replace( @data_path, date, {'%d' => '??'} )
			cfile = comment_file_replace( @data_path, date, {'%d' => '??'} )
			rfile = referer_file_replace( @data_path, date, {'%d' => '??'} )
			begin
				cache = @tdiary.restore_parser_cache( date, 'defaultio' )
				unless cache then
					restore( @dfile, diaries )
					restore_comment( cfile, diaries )
					restore_referer( rfile, diaries )
				else
					diaries.update( cache )
				end
				dirty = yield( diaries ) if iterator?
				store( @dfile, diaries ) if dirty & TDiaryBase::DIRTY_DIARY != 0
				store_comment( cfile, diaries ) if dirty & TDiaryBase::DIRTY_COMMENT != 0
				store_referer( rfile, diaries ) if dirty & TDiaryBase::DIRTY_REFERER != 0
				if dirty != TDiaryBase::DIRTY_NONE or not cache then
					@tdiary.store_parser_cache( date, 'defaultio', diaries )
				end
			end
		end

		def calendar
			arr = {}
			td2wc = diary_file_replace( @data_path, nil,
				{ '%Y' => '????', '%m' => '??', '%d' => '??' })
			td2re = Regexp::new(diary_file_replace( @data_path, nil,
				{ '%Y' => '(\d{4})', '%m' => '(\d{2})', '%d' => '(\d{2})' }))
			Dir[td2wc].sort.each do |file|
				next unless td2re === file
				year, month = $1, $2
				arr[year] = [] unless arr[year]
				arr[year] << month unless arr[year].last == month
			end
			arr
		end

		def diary_factory( date, title, body, style = VIKIWIKICONF['STYLE_NAME'] )
			styled_diary_factory( date, title, body, style )
		end

	private
		def diary_file_replace( data_path, date=nil, replace=nil )
			format = "#{VIKIWIKICONF['BASEDIR']}/text/#{VIKIWIKICONF['TD2PAGE']}.#{VIKIWIKICONF['STYLE_NAME']}Style.#{VIKIWIKICONF['FILEEXT']}"
			replace.each{|s,t| format.gsub!( s, t ) } if replace
			format = date.strftime( format ) if date
			return format
		end
		def restore( file, diaries )
			begin
				Dir::glob( file ) do |dfile|
					File::open( dfile.untaint, 'r' ) do |fh|
						# read and parse diary
						while l = fh.gets( "\n#{VIKIWIKIFORMAT['SEP']}\n" )
							l = VIKIWIKICONV['RESTORE'].call( l ) if VIKIWIKICONV['RESTORE']
							begin
								headers, body = TDiary::parse_tdiary( l )
								modified = headers['Last-Modified'].scan(/\d+/)
								style = headers['Format'] || VIKIWIKICONF['STYLE_NAME']
								diary = eval( "#{style( style )}::new( headers['Date'], headers['Title'], body, Time::local( *modified ) )" )
								diary.show( headers['Visible'] == 'true' ? true : false )
								diaries[headers['Date']] = diary
							rescue NameError
							rescue
								raise
							end
						end
					end
				end
			rescue Errno::ENOENT
			end
		end

		def store( file, diaries )
			arr = Array::new
			cvs = DiffCVS::new( VIKIWIKICONF['FILEMODE'].oct, VIKIWIKICONF['BACKUP'] )
			diaries.each do |date,diary|
				# save diaries
				[
					["Date", date],
					["Title", diary.title],
					["Last-Modified", diary.last_modified.strftime( '%Y-%m-%d %H:%M:%S' )],
					["Visible", diary.visible? ? 'true' : 'false'],
					["Format", diary.style]
				].each{|key, val|
					val = VIKIWIKICONV['STORE'].call( val ) if VIKIWIKICONV['STORE'] and val.class == String
					arr << VIKIWIKIFORMAT['HEAD_FORMAT'] % [key, val.to_s]
				}
				arr << ''
				l = diary.to_src.gsub( /\r/, '' )
				l = VIKIWIKICONV['STORE'].call( l ) if VIKIWIKICONV['STORE']
				arr << l
				arr << VIKIWIKIFORMAT['SEP']
				if file.include?( '??' ) then
					dfile = diary_file_replace( @data_path, Time::local( *date.scan( /(\d{4})(\d{2})(\d{2})/ )[0] ) )
					cvs.writeFile( dfile, arr.join("\n") )
					arr.clear
				end
			end
			cvs.writeFile( file, arr.join("\n") ) unless file.include?( '??' )
		end
	end
end
