require 'vikiwikiplugin'
require 'date'

if RUBY_VERSION < "1.8" then
	class Date
		def Date::parse(str)
			raise "Invalid date format" unless /^(\d{4})-(\d{2})-(\d{2})$/ === str
			Date::new($1.to_i, $2.to_i, $3.to_i)
		end
		def strftime(fmt)
			Time::local(year, month, mday).strftime(fmt)
		end
	end
end

module VikiWiki
	module Plugins
		class ListscheduleNode
			attr_reader :page, :indent, :title, :sdate, :edate, :person, :rate, :child
			attr_accessor :parent
			def initialize(page=nil, indent=0, title=nil, sdate=nil, edate=nil, person=nil, rate=nil)
				@page = page.strip if page
				@indent = indent if indent
				@title = CGI::escapeHTML(title.strip) if title
				@sdate = Date::parse(sdate) if sdate
				@edate = Date::parse(edate) if edate
				@person = person.strip if person
				@rate = rate if rate
				@parent = nil
				@child = Array::new
			end
			def normalize()
				@child.map! do |node|
					node.normalize
				end
				@child.compact!
				return self if @rate
				return nil if @child.empty?
				sdate = nil
				edate = nil
				sum = 0
				cnt = 0
				@child.each do |node|
					sdate = node.sdate if sdate.nil? or node.sdate < sdate
					edate = node.edate if edate.nil? or node.edate > edate
					if node.sdate and node.edate then
						len = (node.edate - node.sdate).to_f + 1
						rate = node.rate >= 0 ? node.rate : 100
						sum += len * rate / 100
						cnt += len
					end
				end
				@sdate = sdate unless @sdate
				@edate = edate unless @edate
				@rate = (cnt == 0 ? 0 : (sum / cnt * 100).to_i) if @rate.nil?
				@person = '-' unless @person
				return self
			end
			def appendChild(node)
				cur = self
				while cur.indent >= node.indent do
					break unless cur.parent
					cur = cur.parent
				end
				cur.child << node
				node.parent = cur
				return node
			end
			def each
				@child.each do |node|
					yield node
					node.each do |child_node|
						yield child_node
					end
				end
			end
		end
		class Listschedule
			include BaseModule
			HEADER_REGEXP = /^(?:(!+)|(\s*)[\*\-](?:\s+\S\s+)?)(.+)$/
			RANGE_REGEXP = /^(\s*)[\-\*\+]\s+(\S)\s+(.+)\s+(\d{4}\-\d{2}\-\d{2})\s*\S+\s*(\d{4}\-\d{2}\-\d{2})\s*(\S+)(?:\s*(\d+)\%)?/
			UNIT_REGEXP = /^(\s*)[\-\*\+]\s+(\S)\s+(.+)\s+(\d{4}\-\d{2}\-\d{2})(\!)?\s*(\S+)(?:\s*(\d+)\%)?/
			TABLE_REGEXP = /^(\s*)\|\|\s*(\S)\s*\|\|(.+)\s*\|\|(\d{4}\-\d{2}\-\d{2})(?:\s*\S+\s*(\d{4}\-\d{2}\-\d{2}))?\|\|(.+)(?:\|\|(\d+)\%)?/
			LBL = Hash::new unless defined? LBL
			LBL['SCD_PAG'] = 'Page'
			LBL['SCD_TIT'] = 'Title'
			LBL['SCD_PSN'] = 'Charge'
			def ondesc; <<DSC; end
The plugin can show a chart generated by list format.
{{{
#listschedule [{short|now|all} [pages ... ]]
}}}
DSC
			def safe; 5 ; end
			def onpost
			end
			def onview
				mode, *pages = @prms
				pages = [@sys.page.name] if pages.empty?
				pages = @sys.pages.names if pages[0] == '*'
				cur = schedules = ListscheduleNode::new
				pages.each do |page|
					head_indent = 0
					list_indent = 0
					next unless @sys.pages[page].exist?
					@sys.pages[page].load.each do |line|
						case line
						when RANGE_REGEXP, UNIT_REGEXP, TABLE_REGEXP then
							indent = $1.size
							list_indent = indent if list_indent > indent
							node = ListscheduleNode::new(page,
								head_indent + indent + 1,
								$3, $4, ($5 || $4), $6, getRate($2, $7))
							cur.appendChild(node)
						when HEADER_REGEXP then
							if $1 then
								head_indent = $1.size
								node = ListscheduleNode::new(page, head_indent, $3)
							else
								list_indent = $2.size
								node = ListscheduleNode::new(page, head_indent+list_indent+1, $3)
							end
							cur = cur.appendChild(node)
						end
					end
				end
				schedules.normalize
				case mode
				when 'now' then
					srange = Date::today - 15
					erange = Date::today + 15
				else
					srange = schedules.sdate
					erange = schedules.edate
					return unless srange
					if mode != 'all' then
						srange_now = Date::today - 15
						erange_now = Date::today + 15
						srange = srange_now if srange < srange_now
						erange = erange_now if erange > erange_now
					end
				end
				raise "too large range from #{srange.strftime('%Y-%m-%d')} to #{erange.strftime('%Y-%m-%d')}" if erange - srange > 365
				chart_view(srange, erange, schedules)
			end
			def chart_view(srange, erange, schedules)
				res = Array::new
				res << %Q[<table class="chart" border="1">]
				# Header
				res << %Q[<tr>]
				res << %Q[<th class="title">#{LBL['SCD_TIT']}</th>]
				res << %Q[<th class="person">#{LBL['SCD_PSN']}</th>]
				prev_mon = nil
				srange.upto(erange) do |day|
					day_str = prev_mon == day.mon ? day.day.to_s : "#{day.mon}/#{day.day}"
					day_str = day_str.rjust(5).gsub(' ', '&nbsp;')
					prev_mon = day.mon
					date_class = day.strftime('%A')
					date_class = 'Today' if day == Date::today
					res << %Q[<th class="#{date_class}">#{day_str}</th>]
				end
				res << %Q[</tr>]
				width = erange - srange + 1
				prev_page = nil
				schedules.each do |schedule|
					next if schedule.sdate > erange or schedule.edate < srange
					if prev_page != schedule.page then
						pagelink = @sys.wri_onview(schedule.page)
						res << %Q[<tr class="page"><td colspan="#{width+2}">#{pagelink}</td></tr>]
						prev_page = schedule.page
					end
					res << %Q[<tr>]
					tree = schedule.indent > 1 ? '&nbsp;' * schedule.indent + '+-' : ''
					res << %Q[<td class="title">#{tree}#{schedule.title}</td>]
					res << %Q[<td class="person">#{schedule.person}</td>]

					lcols = schedule.sdate - srange
					acols = mcols = schedule.edate - schedule.sdate + 1
					rcols = erange - schedule.edate
					lcols, mcols = 0, mcols + lcols if lcols < 0
					rcols, mcols = 0, mcols + rcols if rcols < 0
					lcols, mcols = width, 0 if lcols > width
					rcols, mcols = width, 0 if rcols > width
					0.upto(lcols-1) do |i|
						day = srange + i
						date_class = day.strftime('%A')
						date_class = 'Today' if day == Date::today
						res << %Q[<td class="#{date_class}">&nbsp;</td>]
					end
					if mcols > 0 then
						res << %Q[<td colspan="#{mcols}">]
						res << %Q[<table class="result" width="100%" title="#{schedule.title} (#{schedule.rate}%)">]
						res << %Q[<tr>]
						if schedule.rate == 0 then
							res << %Q[<td class="yet" width="100%">&nbsp;</td>]
						elsif schedule.rate == 100 then
							res << %Q[<td class="end" width="100%">&nbsp;</td>]
						elsif schedule.rate < 0 then
							res << %Q[<td class="del" width="100%">&nbsp;</td>]
						elsif srange < schedule.sdate and schedule.edate < erange then
							res << %Q[<td class="end" width="#{schedule.rate}%">&nbsp;</td>]
							res << %Q[<td class="yet" width="*">&nbsp;</td>]
						else
							endlen = ((schedule.edate - schedule.sdate + 1) * schedule.rate / 100).to_i
							curdate = schedule.sdate + endlen
							endlen -= srange - schedule.sdate if srange > schedule.sdate
							endlen -= curdate - erange if curdate > erange
							res << %Q[<td class="end" width="#{(endlen*100/mcols).to_i}%">&nbsp;</td>] if endlen > 0
							res << %Q[<td class="yet" width="*">&nbsp;</td>] if endlen <= mcols
						end
						res << %Q[</tr>]
						res << %Q[</table>]
						res << %Q[</td>]
					end
					0.upto(rcols-1) do |i|
						day = srange + lcols + mcols + i
						date_class = day.strftime('%A')
						date_class = 'Today' if day == Date::today
						res << %Q[<td class="#{date_class}">&nbsp;</td>]
					end
					res << %Q[</tr>]
				end
				res << %Q[</table>]
				return res.flatten.join("\n")
			end
			def getRate(mark, rate)
				case mark
				when 'o' then
					return 100
				when 'x' then
					return 0
				when '+' then
					return rate.to_i if rate
					return 50
				when '-' then
					return -1
				end
				return -1
			end
		end
	end
end
