require 'ftools'
require 'lockfile'
require 'message'

class CVS
	def initialize(cvsroot, localdir)
		@cvsroot = cvsroot || ENV['CVSROOT']
		@cvsroot = @cvsroot.gsub(/\\/, '/')
		@localdir = localdir
		@localdir = @localdir.gsub(/\\/, '/')
		@localdir += '/' unless /\/$/ === @localdir
		@tmpdir = ENV['TMPDIR'] || ENV['TMP'] || ENV['TEMP'] || '/tmp'
		@tmpdir = @tmpdir.gsub(/\\/, '/').untaint + "/cvs_work"
		Dir::mkdir(@tmpdir) unless File::exist?(@tmpdir)
		@relfile = nil
		@outfile = nil
		@errfile = nil
	end
	def transaction(file)
		@relfile = file.sub(/^#{Regexp::escape(@localdir)}/i, '')
		now = Time::now.to_i
		@outfile = "#{@tmpdir}/#{$$}_#{now}.out"
		@errfile = "#{@tmpdir}/#{$$}_#{now}.err"
		pwd = Dir::pwd.untaint
		Dir::chdir(@localdir)
		dirs = Array::new
		File::dirname(@relfile).scan(/([^\/]+)/) do
			dirs << $1
			dir = dirs.join('/')
			next if File::exist?(dir)
			Dir::mkdir(dir)
			exec("add", dir)
			exec("checkout", dir)
		end
		begin
			yield @relfile
		ensure
			Dir::chdir(pwd)
			File::unlink(@outfile) if File::exist?(@outfile)
			File::unlink(@errfile) if File::exist?(@errfile)
			@outfile = nil
			@errfile = nil
			@relfile = nil
		end
	end
	EXCEPT_MSG = [
		/Checking out/,
		/scheduling file/,
		/already been entered/,
		/already exists/,
		/Updating/,
		""
	]
	def exec(options, file=@relfile)
		raise "not in transaction" unless @relfile
		cmd = "cvs -d #{@cvsroot} #{options} #{file} 1> #{@outfile} 2> #{@errfile}"
		puts cmd if $DEBUG
		system(cmd.untaint)
		errstr = File::readlines(@errfile).join
		err = true
		noterr = EXCEPT_MSG.select{|reg| reg === errstr}
		raise errstr if noterr.empty?
		outstr = File::readlines(@outfile).join
		return outstr
	end
end

class CVSwrapper
	def initialize(cvsroot=ENV['CVSROOT'], localdir=Dir::pwd)
		@cvs = CVS::new(cvsroot, localdir)
	end
	#===================================================================
	# Method Name   : readFile
	# Explanations  : Read file.
	# Parameters    : file - file path
	#                 ver - version
	# Return values : content of file
	#===================================================================
	def readFile(file, ver=nil)
		return File::readlines(file).join unless ver
		@cvs.transaction(file) do
			return @cvs.exec("checkout -p -r #{ver}")
		end
	end
	#===================================================================
	# Method Name   : writeFile
	# Explanations  : Writing file.
	# Parameters    : file - file path
	#                 text - content to write
	#                 check - check timestamp
	#                 user - user name
	# Return values : none
	#===================================================================
	def writeFile(file, text=nil, check=nil, user=nil, mtime=nil)
		if File::exist?(file.untaint) then
			exist_flag = true
			cur = File::stat(file).mtime
			raise Message::new(:FILE_ALREADY_MODIFIED) if check and cur != check
			return if File::readlines(file).join == text
		else
			exist_flag = false
		end
		@cvs.transaction(file) do |relfile|
			File::open(relfile, "w") do |fout|
				fout.write(text)
			end
			File::utime(mtime, mtime, relfile) if mtime
			@cvs.exec("add") unless exist_flag
			@cvs.exec("commit -m \"#{self.class.name} author: #{user}\"")
		end
	end
	#===================================================================
	# Method Name   : renameFile
	# Explanations  : Renaming file.
	# Parameters    : from - file path
	#                 to - file path
	# Return values : none
	#===================================================================
	def renameFile(from, to)
		raise Message::new(:FUNC_NOT_SUPPORTED, "rename")
	end
	#===================================================================
	# Method Name   : removeFile
	# Explanations  : Removing file.
	# Parameters    : file - file path
	# Return values : none
	#===================================================================
	def removeFile(file)
		@cvs.transaction(file) do
			@cvs.exec("rm -f")
		end
	end
	#===================================================================
	# Method Name   : getDiff
	# Explanations  : Getting diff.
	# Parameters    : file - file path
	#                 ver - version
	# Return values : diff string
	#===================================================================
	def getDiff(file, ver)
		vers = getVersions(file)
		idx = ver ? vers.index(ver) : 0
		return nil if idx.nil? or idx+1 >= vers.size
		rev1 = vers[idx+1]
		rev2 = vers[idx]
		@cvs.transaction(file) do
			lines = @cvs.exec("diff -r #{rev1} -r #{rev2}").to_a
			lines.shift while lines.first and not /^\d/ === lines.first
			return lines.join
		end
	end
	#===================================================================
	# Method Name   : getHistry
	# Explanations  : Getting history.
	# Parameters    : file - file path
	# Return values : all version information with Array
	#                 [version, user-name, update-time]
	#===================================================================
	def getHistory(file)
		history = Array::new
		ver, tms, usr = nil
		@cvs.transaction(file) do
			@cvs.exec("log").each do |line|
				case line
				when /^revision (.+)$/ then
					ver = $1
					history << [ver, nil, nil]
				when /^date: (\d{4})\/(\d{2})\/(\d{2}) (\d{2}):(\d{2}):(\d{2});/ then
					tms = Time::local($1.to_i, $2.to_i, $3.to_i, $4.to_i, $5.to_i, $6.to_i)
					history.last[2] = tms
				when /^#{self.class.name} author: (\w+);/ then
					history.last[1] = $1
				end
			end
		end
		return history
	end
	#===================================================================
	# Method Name   : getVersions
	# Explanations  : Getting all version.
	# Parameters    : file - file path
	# Return values : all version with Array
	#===================================================================
	def getVersions(file)
		getHistory(file).map{|his| his[0]}
	end
end

if __FILE__ == $0 then

def Dir::rm_f(dir)
	Dir::foreach(dir) do |file|
		next if /^\.+$/ === file
		file = "#{dir}/#{file}"
		if File::directory?(file) then
			Dir::rm_f(file)
			Dir::unlink(file)
		else
			File::unlink(file)
		end
	end
	Dir::unlink(dir)
end

print "CVSROOT : "
cvsroot = gets.chomp
print "Local Dir : "
localdir = gets.chomp

cvs = CVSwrapper::new(cvsroot, localdir)
loop do
	fout = STDOUT
	if ARGV.empty? then
		print "> "
		line, out = $`, $1 if /\>\s*(\S+)$/ === (line = gets.chomp)
		fout = File::open(out, "w") if out
		cmd, *prms  = line.split
	else
		cmd, *prms  = ARGV
	end
	case cmd
	when 'read' then
		ver = prms[1] if prms[1]
		Dir::glob(prms[0]) do |file|
			next if File::directory?(file)
			next if File::expand_path(file).include?('/.')
			lines = cvs.readFile(file, ver)
			printf "<< file=%s, version=%s >>\n", file, ver.inspect
			fout.write lines
		end
	when 'write' then
		Dir::glob(prms[0]) do |file|
			next if File::directory?(file)
			next if File::expand_path(file).include?('/.')
			puts file
			cvs.writeFile(file)
		end
	when 'rename' then
		file1 = prms[0]
		file2 = prms[2]
		cvs.renameFile(file1, file2)
	when 'remove' then
		file = prms[0]
		cvs.removeFile(file)
	when 'history' then
		file = prms[0]
		vers = cvs.getHistory(file)
		vers.each do |ver, user, time|
			printf "<< version=%s, user=%s, time=%s >>\n", ver.inspect, user.inspect, time.strftime('%Y-%m-%d %H:%M:%S')
		end
	when 'diff' then
		file = prms[0]
		ver = prms[1]
		lines = cvs.getDiff(file, ver)
		printf "<< file=%s, version=%s >>\n", file, ver.inspect
		fout.write lines
	when 'exit' then
		break
	else
		system([cmd, prms].join(' '))
	end
	fout.close if fout != STDOUT
	if ARGV.empty? then
		puts "finished."
	else
		break
	end
end

end
