require 'wiki/pukiwiki13style'

module Wiki
	class PukiWiki14Style < PukiWiki13Style
		#*******************************************************************
		# Format Rule
		#*******************************************************************
		RE_MARKUP        = /^(?:[\~\-\+\*\#\:\|\/\>\<\s]|(?:LEFT|RIGHT|CENTER)\:)/
		# Block

		# <ul>
		RE_UL            = /^(\-{1,3})(\~)?(.*)$/

		# <ol>
		RE_OL            = /^(\+{1,3})(\~)?(.*)$/

		# <dl>
		RE_DL            = /^(\:{1,3})([^\|]+)\|(\~)?(.*)$/

		# <blockquote>
		RE_BLOCKQUOTE_OUT= /^(<{1,3})$/

		# <table>
		RE_TABLE         = /^\|(.+)\|([hHfFcC]?)$/
		RE_TH            = /^\~(.+)$/
		RE_TD            = /\|/

		# Table Querifier
		RE_TD_CENTER     = /^CENTER:(.*)$/
		RE_TD_LEFT       = /^LEFT:(.*)$/
		RE_TD_RIGHT      = /^RIGHT:(.*)$/
		RE_TD_COLOR      = /^COLOR\((#{TX_NOPARE_S})\):(.*)$/
		RE_TD_SIZE       = /^SIZE\((\d+)\):(.*)$/
		RE_TD_BGCOLOR    = /^BGCOLOR\((#{TX_NOPARE_S})\):(.*)$/

		# <table> by CSV
		RE_CSV           = /^,((?:\"[^\"]*\")+|[^,]+)/
		RE_CSV_QUOT      = /^\"(.+)\"$/
		RE_CSV_CENTER    = /^\s+(\S+)\s+$/
		RE_CSV_LEFT      = /^(\S+)\s+$/
		RE_CSV_RIGHT     = /^\s+(\S+)$/

		# <p>
		RE_P             = /^\~(.*)$/

		# Inline

		# <img>
		RE_IMG           = /^\[\[([^:>]+)[:>](#{TX_URL}#{TX_IMGEXT})\]\]/

		#*******************************************************************
		# Definitions
		#*******************************************************************
		BLOCK_PARSING_ORDER = [
			'plugin','div','pre','blockquote','blockquote_out','comment',
			'h4','h3','h2',
			'hr','ul','ol','dl','table', 'csv','p',
		]
		INLINE_PARSING_ORDER = [
			'wikiname','uri','em','strong','del','inplugin',
			'img','a','aliasname','bracketname','notes','lineend_br'
		]
		ELEMENT_GENERATING_TAGS = [
			'wikiname', 'plugin', 'del', 'strong',
			'em', 'img', 'a', 'sup', 'br',
			'pre', 'h2', 'h3', 'h4', 'hr',
			'div', 'li', 'dl', 'blockquote',
			'table', 'tr', 'p'
		]
		INDENT_STYLE = false
		# character reference
		CHAR_REFS = ['lt', 'gt', 'amp', 'quot', 'apos',
					 'nbsp', 'copy', 'reg', 'trade', 'sup2', 'sup3']

		#*******************************************************************
		# Public Methods
		#*******************************************************************
		def initialize(sys, conf={})
			super
			@puki_indent = 0
			@puki_ind_base = 0
			@puki_bq_base = Array::new
		end
		#*******************************************************************
		# Common Parser's Methods
		#*******************************************************************
		def split_indent(line)
			return line, @puki_indent
		end
		#*******************************************************************
		# Block Parser
		#*******************************************************************
		# h1-h6, hr, comment => PukiWiki 1.3
		def pre_onparse(root, match, line, indent, lines)
			super(root, match, line, @puki_indent, lines)
		end
		def li_onparse(root, match, line, indent, lines, tag='ul')
			depth = match[1].size
			@puki_ind_base = @puki_indent + 1 if @puki_prev and @puki_prev == 'bq'
			@puki_indent = @puki_ind_base + depth - 1
			@puki_prev = 'li'
			p, item = match[2], match[3]+"\n"
			item << get_paragraph(lines)
			nlist = appendChild_block(root, tag, nil, nil, @puki_indent, APPEND_IF_NOT_EXIST)
			if p.nil? or p.empty? then
				nli = appendChild_block(nlist, 'li', nil, item, @puki_indent, APPEND_CHILD)
			else
				nli = appendChild_block(nlist, 'li', nil, nil, @puki_indent, APPEND_CHILD)
				appendChild_block(nli, 'p', nil, item, @puki_indent, APPEND_CHILD)
			end
			@puki_indent += 1
			return nli
		end
		def dl_onparse(root, match, line, indent, lines)
			depth = match[1].size
			@puki_ind_base = @puki_indent + 1 if @puki_prev and @puki_prev == 'bq'
			@puki_indent = @puki_ind_base + depth - 1
			@puki_prev = 'dl'
			dt, p, dd = match[2], match[3], match[4]+"\n"
			dd << get_paragraph(lines)
			root = appendChild_block(root, 'dl', nil, nil, @puki_indent, APPEND_IF_NOT_EXIST)
			appendChild_block(root, 'dt', nil, dt, @puki_indent, APPEND_CHILD)
			if p then
				ndd = appendChild_block(root, 'dd', nil, nil, @puki_indent, APPEND_CHILD)
				appendChild_block(ndd, 'p', nil, dd, @puki_indent+=1, APPEND_CHILD)
			else
				ndd = appendChild_block(root, 'dd', nil, dd, @puki_indent, APPEND_CHILD)
			end
			@puki_indent += 1
			return ndd
		end
		def blockquote_onparse(root, match, line, indent, lines)
			depth = match[1].size
			@puki_bq_base << {'depth'=>depth, 'indent'=>@puki_indent, 'ind_base'=>@puki_ind_base}
			@puki_ind_base = @puki_indent + 1 if @puki_prev and @puki_prev != 'bq'
			@puki_indent = @puki_ind_base + depth - 1
			@puki_prev = 'bq'
			text = match[2]+"\n"
			text << get_paragraph(lines)
			nbq = appendChild_block(root, 'blockquote', nil, nil, @puki_indent)
			appendChild_block(nbq, 'p', nil, text, @puki_indent, APPEND_CHILD)
			@puki_bq_base.last['node'] = nbq
			@puki_indent += 1
			return nbq
		end
		def blockquote_out_onparse(root, match, line, indent, lines)
			depth = match[1].size
			bq = nil
			while bq = @puki_bq_base.pop do
				break if bq['depth'] == depth
			end
			@puki_indent = bq ? bq['indent'] : 0
			@puki_ind_base = bq ? bq['ind_base'] : 0
			nbq = bq ? bq['node'].parentNode : root
			text = get_paragraph(lines)
			appendChild_block(nbq, 'p', nil, text, @puki_indent, APPEND_CHILD) unless text.empty?
			return nbq
		end
		def table_onparse(root, match, line, indent, lines)
			node = appendChild_block(root, 'table', nil, nil, @puki_indent)
			thead = appendChild_block(node, 'thead', nil, nil, @puki_indent, APPEND_CHILD)
			tbody = appendChild_block(node, 'tbody', nil, nil, @puki_indent, APPEND_CHILD)
			tfoot = appendChild_block(node, 'tfoot', nil, nil, @puki_indent, APPEND_CHILD)
			loop do
				cols, kind = match[1], match[2].downcase
				case kind
				when 'h' then
					tparent = thead
				when 'f' then
					tparent = tfoot
				else
					tparent = tbody
				end
				tr = appendChild_block(tparent, 'tr', nil, nil, @puki_indent, APPEND_CHILD)
				colspan = 1
				cols.split(RE_TD).each{|td|
					style = ''
					loop do
						case td
						when RE_TD_CENTER then
							td = $1
							style << 'text-align:center;'
						when RE_TD_LEFT then
							td = $1
							style << 'text-align:left;'
						when RE_TD_RIGHT then
							td = $1
							style << 'text-align:right;'
						when RE_TD_BGCOLOR then
							td = $2
							style << "background-color:#{$1};"
						when RE_TD_COLOR then
							td = $2
							style << "color:#{$1};"
						when RE_TD_SIZE then
							td = $2
							style << "fomt-size: #{$1}px;"
						else
							break
						end
					end
					case td
					when '>' then
						colspan += 1
					when '~' then
						appendChild_block(tr, 'td', {'span'=>'row'}, nil, @puki_indent, APPEND_CHILD)
					else
						attr = style.empty? ? {} : {'style' => style}
						tag = 'td'
						tag, td = 'th', $1 if RE_TH === td
						appendChild_block(tr, tag, attr, td, @puki_indent, APPEND_CHILD)
						2.upto(colspan){
							appendChild_block(tr, 'td', {'span'=>'column'}, nil, @puki_indent, APPEND_CHILD)
						}
						colspan = 1
					end
				}
				break if lines.empty? or not RE_TABLE === lines.first
				match = $~
				lines.shift
			end
			node.removeChild(thead) unless thead.hasChildNodes
			node.removeChild(tbody) unless tbody.hasChildNodes
			node.removeChild(tfoot) unless tfoot.hasChildNodes
			return root
		end
		def csv_onparse(root, match, line, indent, lines)
			node = appendChild_block(root, 'table', nil, nil, @puki_indent)
			loop do
				tr = appendChild_block(node, 'tr', nil, nil, @puki_indent, APPEND_CHILD)
				while RE_CSV === line do
					td, line = $1.chomp, $'
					td = $1.gsub(/""/,'"') if RE_CSV_QUOT === td
					attr = Hash::new
					case td
					when RE_CSV_CENTER then
						td = $1
						attr['align'] = 'center'
					when RE_CSV_LEFT then
						td = $1
						attr['align'] = 'left'
					when RE_CSV_RIGHT then
						td = $1
						attr['align'] = 'right'
					end
					appendChild_block(node, 'td', attr, td, @puki_indent, APPEND_CHILD)
				end
				break unless RE_CSV === lines.first
				line = lines.shift
			end
		end
		def p_onparse(root, match, line, indent, lines)
			p = match[1]+"\n"
			p << get_paragraph(lines)
			appendChild_block(root, 'p', nil, p, @puki_indent)
		end
		#*******************************************************************
		# Inline Parser
		#*******************************************************************
		# WikiName, uri, em, strong, del => PukiWiki 1.3
		def img_onparse(root, match)
			appendChild_inline(root, 'img', {'src'=>match[2], 'alt'=>match[1]})
		end
		def inplugin_onparse(root, match)
			name, prm, intext, plain = match[1], match[2], match[3], match[0]
			case name
			when 'size' then
				return appendChild_inline(root, 'font', {'size'=>prm}, intext)
			when 'color' then
				return appendChild_inline(root, 'font', {'color'=>prm}, intext)
			when 'ref' then
				return appendChild_inline(root, 'a', {'href'=>prm}, prm)
			when 'ruby' then
				node = appendChild_inline(root, 'ruby')
				appendChild_inline(node, 'rb', nil, intext)
				appendChild_inline(node, 'rt', nil, prm)
				return node
			when 'aname' then
				return appendChild_inline(root, 'a', {'name'=>prm})
			when 'br' then
				return appendChild_inline(root, 'br')
			else
				return plugin_onparse(root, match, match[0], nil, nil, true)
			end
		end
		#===================================================================
		# Method Name   : get_paragraph
		# Explanations  : get text block of a paragraph
		# Parameters    : lines - array of lines
		# Return values : paragraph text
		#===================================================================
		def get_paragraph(lines)
			p = ''
			until lines.empty? or RE_MARKUP === lines.first do
				line = lines.shift
				p << line
			end
			return p
		end
		#*******************************************************************
		# Wiki Text Block generator
		#*******************************************************************
		def table_ontext(node, outobj='')
			generateBlock(node.childNodes, outobj)
		end
		def tr_ontext(node, outobj='')
			tds = [nil]
			cur = node.firstChild
			while cur do
				case cur.getAttribute('span')
				when 'row' then
					tds << '~'
				when 'column' then
					col = tds.pop
					tds << '>'
					tds << col if col
				else
					text = generateInlineText(cur)
					align = get_align(cur)
					text = align.upcase+':'+text if align
					tds << text
				end
				cur = cur.nextSibling
			end
			case node.parentNode.nodeName
			when 'thead' then
				tds << 'h'
			when 'tfoot' then
				tds << 'f'
			else
				tds << nil
			end
			outobj << tds.join('|')
			outobj << "\n"
			return outobj
		end
		def dl_ontext(node, outobj='')
			level, n = 1, node
			loop do
				n = n.parentNode
				case n.nodeName
				when 'dl' then
					level += 1
				when 'dt', 'dd' then
				else
					break
				end
			end
			level = 3 if level > 3
			dt = ''
			cur = node.firstChild
			while cur do
				ctxt, ncur = generateInline(cur)
				case cur.nodeName
				when 'dt' then
					dt = ctxt
				when 'dd' then
					dt = ' ' + dt if /^:/ === dt
					outobj << ':'*level+dt+'|'+ctxt+"\n"
					generateSurplus(ncur, outobj)
				end
				cur = cur.nextSibling
			end
			return outobj
		end
		def div_ontext(node, outobj='')
			align = node.getAttribute('align')
			outobj << align.upcase+':' if align
			outobj << text
		end
		def li_ontext(node, outobj='')
			level, n = 0, node
			loop do
				break unless n = n.parentNode
				case n.nodeName
				when 'ul', 'ol' then
					level += 1
				when 'li' then
				when 'dl','blockquote','div','table' then
					break
				end
			end
			level = 3 if level > 3
			text, ncur = generateInline(node)
			if node.parentNode.nodeName == 'ul' then
				text = ' ' + text if /^\-/ === text
				outobj << '-'*level+text
			else
				text = ' ' + text if /^\+/ === text
				outobj << '+'*level+text
			end
			outobj << "\n"
			generateSurplus(ncur, outobj)
			return outobj
		end
		def blockquote_ontext(node, outobj='')
			level, n = 1, node
			level += 1 while (n=n.parentNode).nodeName == 'blockquote'
			level = 3 if level > 3
			text, ncur = generateInline(node)
			text = ' ' + text if /^>/ === text
			outobj << '>'*level+text
			generateSurplus(ncur, outobj)
			outobj << '<'*level
			outobj << "\n"
			return outobj
		end
		def p_ontext(node, outobj='')
			prev = node.previousSibling
			plevel = prev.getAttribute('wiki-indent-level') if Element === prev
			text = generateChild(node)
			if prev and plevel.to_i > 0 and prev.nodeName == 'p' then
				outobj << '~' + text
			else
				outobj << text
			end
			outobj << "\n"
		end
		#*******************************************************************
		# Wiki Text Inline Generator
		#*******************************************************************
		def font_ontext(node, outobj='')
			attr = node.getAttributeHash
			text = generateChild(node)
			if attr.key?('color') then
				outobj << "&color(#{attr['color']}){#{text}}"
			elsif attr.key?('size') then
				outobj << "&size(#{attr['size']}){#{text}}"
			else
				outobj << text
			end
			return outobj
		end
		def ruby_ontext(node, outobj='')
			rb = rt = nil
			list = node.childNodes
			0.upto(list.size-1) do |i|
				ncur = list.item(i)
				case ncur.nodeName
				when 'rb' then
					rb = generateBlock(ncur)
				when 'rt' then
					rt = generateBlock(ncur)
				end
			end
			outobj << "&ruby(#{rt}){#{rb}};"
		end
		def a_ontext(node, outobj='')
			return outobj if wikiname_ontext(node, outobj)
			attr = node.getAttributeHash
			text = generateInlineText(node).strip
			if attr['name'] then
				if text.empty? then
					text = "&aname(#{attr['name']});"
				else
					text = "&aname(#{attr['name']}){#{text}};"
				end
			else
				text = '[['+text+':'+attr['href']+']]'
			end
			outobj << text
		end
		def img_ontext(node, outobj='')
			attr = node.getAttrbuteHash
			outobj << '[['+attr['alt']+':'+attr['src']+']]'
		end
		def wikiname_ontext(node, outobj='')
			attr = node.getAttributeHash
			text = generateInlineText(node)
			if text.nil? or attr['wri'] == text then
				outobj << (attr['bracket'] ? '[['+attr['wri']+']]' : attr['wri'])
			else
				outobj << '[['+attr['wri']+'>'+text+']]'
			end
		end
		#*******************************************************************
		# Wiki Text Plugin Generator
		#*******************************************************************
		def plugin_ontext(node, outobj='')
			list = node.childNodes
			prms = Array::new
			0.upto(list.length-1) do |i|
				prms << generateInlineText(list.item(i))
			end
			if node.getAttribute('wiki-node-type') == 'block' then
				outobj << "##{node.getAttribute('name')}"
				outobj << "(#{prms.join(',')})" unless prms.empty?
				outobj << "\n"
			else
				outobj << "&#{node.getAttribute('name')}"
				unless prms.empty? then
					prm = prms.pop if prms.size > 1
					outobj << "(#{prms.join(',')})" unless prms.empty?
					outobj << "{#{prm}}" if prm
				end
				outobj << ';'
			end
			return outobj
		end
		#*******************************************************************
		# HTML Generator
		#*******************************************************************
		def plugin_onhtml(node, outobj='')
			name = node.getAttribute('name')
			if CHAR_REFS.include?(name) or /^\#(\d+|x[0-9A-Fa-f]+)$/ === name then
				outobj << "&#{name};"
				return outobj
			end
			return false
		end
	end
end
