require 'wiki/wikistyle'

module Wiki
	class PukiWiki13Style < WikiStyle
		#*******************************************************************
		# Format Rule
		#*******************************************************************
		#-------------------------------------------------------------------
		# Block
		#-------------------------------------------------------------------
		# <h1> to <h6>
		FM_H1            = nil
		FM_H2            = '*%s'
		FM_H3            = '**%s'
		FM_H4            = '***%s'
		FM_H5            = nil
		FM_H6            = nil
		RE_H1            = nil
		RE_H2            = /^\*{1}([^\*].*)$/
		RE_H3            = /^\*{2}([^\*].*)$/
		RE_H4            = /^\*{3}([^\*].*)$/
		RE_H5            = nil
		RE_H6            = nil

		# <hr />
		RE_HR            = /^-{4,}$/

		# <ul>
		TX_UL            = '-'
		RE_UL            = /^(\-{1,3})(.*)$/

		# <ol>
		TX_OL            = '+'
		RE_OL            = /^(\+{1,3})(.*)$/

		# <dl>
		TX_DT            = ':'
		TX_DD            = ':'
		RE_DL            = /^(\:{1,3})([^\:]+)\:(.*)$/

		# <blockquote>
		TX_BLOCKQUOTE    = '>'
		RE_BLOCKQUOTE    = /^(>{1,3})(.*)$/

		# <table>
		TX_TABLE         = '|'
		RE_TABLE         = /^\|(.+)\|$/

		# <!-- -->
		RE_COMMENT       = /^\/\/(.+)$/

		# <pre>

		# Querifier
		RE_CENTER        = /^CENTER:(.*)$/
		RE_LEFT          = /^LEFT:(.*)$/
		RE_RIGHT         = /^RIGHT:(.*)$/

		#-------------------------------------------------------------------
		# Inline
		#-------------------------------------------------------------------
		# <strong>

		# <em>

		# <del>
		FM_DEL           = "%%%%%s%%%%"
		RE_DEL           = /^%{2}([^%]+)%{2}/

		# notes
		RE_NOTES          = /^\({2}([^)]+)\){2}/

		# <a>
		FM_A             = '[[%s>%s]]'
		RE_A             = /^\[\[([^:>]+)[:>](#{TX_URL})\]\]/

		# WikiName
		FM_WIKINAME      = '%s'
		RE_WIKINAME      = /^(#{TX_WRIT})/

		# BracketName
		FM_BRACKETNAME   = '[[%s]]'
		RE_BRACKETNAME   = /^\[\[(#{TX_WRIX})\]\]/

		# AliasName
		FM_ALIASNAME     = '[[%s>%s]]'
		RE_ALIASNAME     = /^\[\[(#{TX_WRIX})>(#{TX_NOPARE_L})\]\]/

		# URL

		# <br>
		RE_LINEEND_BR    = /^\~$/

		# Querifier
		FM_COLOR         = "COLOR(%s){%s}"
		RE_COLOR         = /^COLOR\((#{TX_NOPARE_S})\)\{(#{TX_NOPARE_M})\}/
		FM_SIZE          = "SIZE(%s){%s}"
		RE_SIZE          = /^SIZE\((\d+)\)\{(#{TX_NOPARE_M})\}/

		#-------------------------------------------------------------------
		# Plugin
		#-------------------------------------------------------------------

		# plugin
		RE_PLUGIN        = /^\#(#{TX_IDENT})(?:\#(\S+))?(?:\((.+)\))?$/

		# plugin (inline)
		RE_INPLUGIN       = /^\&(\#?\w+)(?:\((#{TX_NOPARE_S})\))?(?:\{(#{TX_NOPARE_M})\})?;/

		# plugin parameters
		RE_PLUGIN_SEP    = /^\s*,\s*/

		#*******************************************************************
		# Definitions
		#*******************************************************************
		BLOCK_PARSING_ORDER = [
			'plugin','pre','blockquote','comment',
			'center', 'left', 'right',
			'h4','h3','h2',
			'hr','ul','ol','dl','table'
		]
		INLINE_PARSING_ORDER = [
			'wikiname','uri','em','strong','del',
			'inplugin', 'color', 'size',
			'a','aliasname','bracketname','notes','lineend_br'
		]
		ELEMENT_GENERATING_TAGS = [
			'wikiname', 'plugin', 'del', 'strong',
			'em', 'a', 'sup', 'br','font',
			'pre', 'h2', 'h3', 'h4', 'hr',
			'div', 'li', 'dl', 'blockquote','p',
			'table'
		]
		INDENT_STYLE = false
		ALIAS_FORWARD = true

		#*******************************************************************
		# Block Parser
		#*******************************************************************
		# h1-h6, hr => base
		def table_onparse(root, match, line, indent, lines)
			ntable = appendChild_block(root, 'table', nil, nil, 0)
			loop do
				ntr = appendChild_block(ntable, 'tr', nil, nil, 0, APPEND_CHILD)
				match[1].split(@my::TX_TABLE).each{|td|
					style = ''
					case td
					when @my::RE_CENTER then
						td = $1
						style = 'text-align:center;'
					when @my::RE_LEFT then
						td = $1
						style = 'text-align:left;'
					when @my::RE_RIGHT then
						td = $1
						style = 'text-align:right;'
					end
					attr = style.empty? ? {} : {'style' => style}
					appendChild_block(ntr, 'td', attr, td, 0, APPEND_CHILD)
				}
				break if lines.empty? or not @my::RE_TABLE === lines.first
				match = $~
				lines.shift
			end
			return ntable.parentNode
		end
		def comment_onparse(root, match, line, indent, lines)
			text = match[1]+"\n"
			while @my::RE_COMMENT === lines.first do
				text << $1+"\n"
				lines.shift
			end
			root.appendCommentSimply(text)
			return root
		end
		def center_onparse(root, match, line, indent, lines)
			appendChild_block(root, 'div', {'align'=>'center'}, match[1], @puki_indent)
		end
		def left_onparse(root, match, line, indent, lines)
			appendChild_block(root, 'div', {'align'=>'left'}, match[1], @puki_indent)
		end
		def right_onparse(root, match, line, indent, lines)
			appendChild_block(root, 'div', {'align'=>'right'}, match[1], @puki_indent)
		end
		#*******************************************************************
		# Inline Parser
		#*******************************************************************
		# WikiName, uri, em, strong, del => base
		def a_onparse(root, match)
			appendChild_inline(root, 'a', {'href'=>match[2]}, match[1])
		end
		def size_onparse(root, match)
			appendChild_inline(root, 'font', {'size'=>match[1]+'px'}, match[2])
		end
		def coor_onparse(root, match)
			appendChild_inline(root, 'font', {'color'=>match[1]}, match[2])
		end
		def notes_onparse(root, match)
			appendChild_inline(root, 'sup', nil, match[1])
		end
		def lineend_br_onparse(root, match)
			appendChild_inline(root, 'br')
		end
		def inplugin_onparse(root, match)
			name, prm, intext, plain = match[1], match[2], match[3], match[0]
			case name
			when 'aname' then
				return appendChild_inline(root, 'a', {'name'=>prm})
			else
				return plugin_onparse(root, match, match[0], nil, nil, true)
			end
		end
		#*******************************************************************
		# Wiki Text Block Generator
		#*******************************************************************
		def _comment_ontext(node, outobj='')
			outobj << "\n"
			node.nodeValue.each do |line|
				outobj << '//' + line
			end
		end
		def hn_ontext(node, outobj='', level=1)
			outobj << '*' * (level - @conf['HNTOP'] + 1)
			outobj << generateChild(node)
			outobj << "\n"
			return outobj
		end
		def hr_ontext(node, outobj='')
			outobj << "----\n"
		end
		def table_ontext(node, outobj='')
			node.each_node('tr') do |ntr|
				tds = [nil]
				list = ntr.childNodes
				0.upto(list.length-1) do |i|
					item = list.item(i)
					text = generateInlineText(item)
					align = get_align(item)
					text = align.upcase+':'+text if align
					tds << text
				end
				tds << nil
				outobj << tds.join(@my::TX_TABLE) + "\n"
			end
			return outobj
		end
		#*******************************************************************
		# Wiki Text Inline Generator
		#*******************************************************************
		def sup_ontext(node, outobj='')
			outobj << "(("+generateChild(node)+"))"
		end
		def font_ontext(node, outobj='')
			attr = node.getAttributeHash
			text = generateChild(node)
			if attr.key?('color') then
				outobj << @my::FM_COLOR % [attr['color'], text]
			elsif attr.key?('size') then
				size = attr['size'].to_i.to_s
				outobj << @my::FM_SIZE % [size, text]
			else
				outobj << text
			end
			return outobj
		end
		def br_ontext(node, outobj='')
			outobj << "~"
		end
		def a_ontext(node, outobj='')
			return outobj if wikiname_ontext(node, outobj)
			attr = node.getAttributeHash
			text = generateInlineText(node).strip
			if attr['name'] then
				if text.empty? then
					text = "&aname(#{attr['name']});"
				else
					text = "&aname(#{attr['name']}){#{text}};"
				end
			else
				text = '[['+text+':'+attr['href']+']]'
			end
			outobj << text
		end
		#*******************************************************************
		# Wiki Text Plugin Generator
		#*******************************************************************
		def plugin_ontext(node, outobj='')
			list = node.childNodes
			prms = Array::new
			0.upto(list.length-1) do |i|
				prms << generateInlineText(list.item(i))
			end
			if node.getAttribute('wiki-node-type') == 'block' then
				outobj << "##{node.getAttribute('name')}"
				outobj << "(#{prms.join(',')})" unless prms.empty?
				outobj << "\n"
			else
				outobj << "&#{node.getAttribute('name')}"
				unless prms.empty? then
					prm = prms.pop if prms.size > 1
					outobj << "(#{prms.join(',')})" unless prms.empty?
					outobj << "{#{prm}}" if prm
				end
				outobj << ';'
			end
			return outobj
		end
	end
end
