module Generator
	class Xml
		ENCODING = {'u' => 'utf-8', 's' => 'Shift_JIS', 'e' => 'euc-jp'}
		ESCAPES = {'<'=>'&lt;','>'=>'&gt;','&'=>'&amp;','"'=>'&quot;','\''=>'&apos;'}
		def initialize
			@escapes = self.class::ESCAPES unless defined?(@escapes)
			@re_escapes = /([#{@escapes.keys.join}])/
			@unescapes = @escapes.invert
			@re_unescapes = /(#{@unescapes.keys.join('|')})/
		end
		def generate(node, outobj='')
			case node
			when Element then
				if node.hasChildNodes then
					list = node.childNodes
					outobj << begin_tag(node)
					0.upto(list.length-1) do |i|
						generate(list.item(i), outobj)
					end
					outobj << end_tag(node)
				else
					outobj << empty_tag(node)
				end
			when Text then
				outobj << escape(node.nodeValue)
			when Comment then
				outobj << '<!--'+escape(node.nodeValue)+'-->'
			when Document then
				encode = ENCODING[$KCODE[0].chr.downcase]
				outobj << %Q[<?xml version="1.0" encoding="#{encode}"?>\n]
				generate(node.documentElement, outobj)
			when DocumentFragment then
				list = node.childNodes
				0.upto(list.length-1) do |i|
					generate(list.item(i), outobj)
				end
			end
			return outobj
		end
	private
		def begin_tag(node)
			tag = "<#{node.nodeName}"
			map = node.attributes
			if map then
				0.upto(map.length-1) do |i|
					attr = map.item(i)
					tag << %Q[ #{attr.nodeName}="#{escape(attr.nodeValue)}"]
				end
			end
			tag << '>'
			return tag
		end
		def end_tag(node)
			return "</#{node.nodeName}>"
		end
		def empty_tag(node)
			return begin_tag(node).sub(/>$/, '/>')
		end
		def escape(text)
			text.gsub(@re_escapes){@escapes[$1]}
		end
		def unescape(text)
			text.gsub(@re_unescapes){@unescapes[$1]}
		end
	end
end
