module DOMUtility
	module NodeCreator
		#===================================================================
		# Method Name   : createElementSimply
		# Explanations  : wrapper of createElement method on DOM
		# Parameters    : name - tag name
		#                 attr - attributes (Hash)
		#                 text - text node value
		#                 append - append element to self after create
		# Return vlaues : created Element object
		#===================================================================
		def createElementSimply(name ,attr=nil, text=nil, append=false)
			doc = Document === self ? self : ownerDocument
			elm = doc.createElement(name)
			if attr then
				attr.each do |key, value|
					next unless key and value
					elm.setAttribute(key, value.to_s)
				end
			end
			elm.appendChild(doc.createTextNode(text)) if text
			appendChild(elm) if append
			return elm
		end
		#===================================================================
		# Method Name   : appendTextSimply
		# Explanations  : Append text node
		# Parameters    : text - text
		# Return vlaues : appended Text object
		#===================================================================
		def appendTextSimply(text)
			return if text.nil? or text.empty?
			ntxt = ownerDocument.createTextNode(text.dup)
			appendChild(ntxt)
			return ntxt
		end
		#===================================================================
		# Method Name   : appendCommentSimply
		# Explanations  : Append comment node
		# Parameters    : comment - comment
		# Return vlaues : appended Comment object
		#===================================================================
		def appendCommentSimply(comment)
			return if comment.nil? or comment.empty?
			ncmt = ownerDocument.createComment(comment.dup)
			appendChild(ncmt)
			return ncmt
		end
	end
	module NodeAccessor
		#===================================================================
		# Method Name   : getTextValues
		# Explanations  : get values of text nodes
		# Return vlaues : array of texts
		#===================================================================
		def getTextValues
			texts = Array::new
			list = childNodes
			0.upto(list.length-1) do |i|
				if Text === list.item(i) then
					texts << list.item(i).nodeValue
				else
					texts << list.item(i).getTextValues
				end
			end
			return texts
		end
		#===================================================================
		# Method Name   : getAttributeHash
		# Explanations  : get pair of keys and values of attributes
		# Return vlaues : Hash object
		#===================================================================
		def getAttributeHash
			attrs = Hash::new
			map = attributes
			return attrs unless map
			0.upto(map.length-1) do |i|
				attr = map.item(i)
				next unless /^[a-zA-Z]/ === attr.nodeName
				attrs[attr.nodeName] = attr.nodeValue
			end
			return attrs
		end
		#===================================================================
		# Method Name   : match?
		# Explanations  : The method checks what the node matchs the condition.
		# Parameters    : type - tag name(s)
		#                 attr - attributes (Hash)
		# Return vlaues : boolean
		#===================================================================
		def match?(type, attr=nil)
			if Array === type then
				return false unless type.include?(nodeName)
			else
				return false unless type === nodeName
			end
			myattr = getAttributeHash
			if Hash === attr then
				attr.each do |name, val|
					return false unless myattr.key?(name)
					return false unless val === myattr[name]
				end
			end
			return true
		end
		#===================================================================
		# Method Name   : each_node
		# Explanations  : The method retrieves nodes matched for
		#                 the condition from self and its child nodes.
		# Parameters    : type - tag name(s)
		#                 attr - attributes (Hash)
		#                 child_only - return child nodes only
		# Return vlaues : Hash object
		#===================================================================
		def each_node(type, attr=nil, child_only=false)
			yield self if not child_only and self.match?(type, attr)
			list = childNodes
			0.upto(list.length-1) do |i|
				item = list.item(i)
				if child_only then
					next unless item.respond_to?(:match?)
					yield self if self.match?(type, attr)
				else
					next unless item.respond_to?(:each_node)
					item.each_node(type, attr, child_only) do |child|
						yield child
					end
				end
			end
		end
	end
	module NodeDebug
		def to_s(indent='', res='')
			res << indent
			res << "<#{nodeName}"
			if Element === self then
				map = attributes
				0.upto(map.length-1) do |i|
					attr = map.item(i)
					res << " #{attr.nodeName}=#{attr.nodeValue.inspect}"
				end
			end
			res << ">\n"
			if nodeValue then
				nodeValue.each do |line|
					res << indent + line.chomp + "\n"
				end
			end
			list = childNodes
			0.upto(list.length-1) do |i|
				list.item(i).to_s(indent+"\t", res)
			end
			res << indent
			res << "</#{nodeName}>\n"
		end
	end
end

class Document
	include DOMUtility::NodeCreator
	include DOMUtility::NodeAccessor
end

class DocumentFragment
	include DOMUtility::NodeCreator
	include DOMUtility::NodeAccessor
end

class Element
	include DOMUtility::NodeCreator
	include DOMUtility::NodeAccessor
end

class Node
	include DOMUtility::NodeDebug
end
