require 'generator/xml'
require 'generator/html'
require 'dom-utility'
require 'vikiwikiconst'

module VikiWiki
	class XmlGenerator < Generator::Xml; end
	class HtmlGenerator < Generator::Html
		RE_ACCESSKEY = /\s+\&(\w)$/
		RE_EMBIMG    = /^(\S+#{IMG_EXT})\s+/

		def initialize(sys, callbacks=nil)
			@sys = sys || {}
			@indent_stack = Array::new
			super(callbacks)
		end
		def begin_tag(node, attr={})
			outobj = ''
			if Element === node then
				name = node.nodeName
				indent = get_indent(node)
				outobj << "\n" if /^h\d$/ === name
				outobj << indent if indent
				@indent_stack.last[1] = true if @indent_stack.last and indent
				@indent_stack.push([indent, false])
			end
			w_attr = attr.dup.delete_if{|key,val| /^wiki/===key}
			outobj << super(node, w_attr)
			return outobj
		end
		def end_tag(node)
			outobj = ''
			if Element === node then
				indent, nested = @indent_stack.pop
				outobj << indent if indent and nested
			end
			outobj << super(node)
			return outobj
		end
		def pre_onhtml(node, outobj='')
			outobj << begin_tag(node)
			outobj << "\n"
			outobj << generate(node.childNodes).chomp
			outobj << "\n"
			outobj << end_tag(node)
			return outobj
		end
		def blockquote_onhtml(node, outobj='')
			outobj << begin_tag(node)
			list = node.childNodes
			0.upto(list.length-1) do |i|
				child = list.item(i)
				if Text === child then
					outobj << begin_tag('p')
					generate(child, outobj)
					outobj << end_tag('p')
				else
					generate(child, outobj)
				end
			end
			outobj << end_tag(node)
		end
		def table_onhtml(node, outobj='')
			attr = node.getAttributeHash
			if attr['class'] == 'noborder' then
				attr['border'] = '0'
			else
				attr['border'] = '1'
			end
			attr['summary'] = 'VikiWiki' unless attr.key?('summary')
			outobj << begin_tag('table', attr)
			generate(node.childNodes, outobj)
			outobj << end_tag('table')
			return outobj
		end
		def td_onhtml(node, outobj='')
			attr = node.getAttributeHash
			return outobj unless attr['span'].nil? or attr['span'].empty?
			colspan = 1
			cur = node
			while cur = cur.nextSibling do
				break if cur.getAttribute('span') != 'column'
				colspan += 1
			end
			i = 0
			cur = node
			i += 1 while cur = cur.previousSibling
			rowspan = 1
			tr = node.parentNode
			while tr = tr.nextSibling do
				list = tr.childNodes
				break if list.item(i).nil? or list.item(i).getAttribute('span') != 'row'
				rowspan += 1
			end
			attr['colspan'] = colspan if colspan > 1
			attr['rowspan'] = rowspan if rowspan > 1
			outobj << begin_tag('td', attr)
			generate(node.childNodes, outobj)
			outobj << end_tag('td')
			return outobj
		end
		def p_onhtml(node, outobj='')
			indent = get_indent(node)
			text = generate(node.childNodes)
			outobj << begin_tag(node)
			outobj << text.gsub(/\n/, indent)
			outobj << end_tag(node)
			return outobj
		end
		def _comment_onhtml(node, outobj='')
			outobj << "\n<!--"+escape(node.nodeValue)+"-->"
			@indent_stack.last[1] = true if @indent_stack.last
			return outobj
		end
		def wikiname_onhtml(node, outobj='')
			name = node.getAttribute('wri')
			text = node.getTextValues.to_s
			text = nil if name == text
			wri = WRI::new(@sys, name)
			href = wri.uri
			text = wri.alias || name unless text
			if wri.exist? then
				aelm = node.createElementSimply('a', {'href'=>href}, text)
				a_onhtml(aelm, outobj)
			else
				href = @sys.pages[InterWikiName].uri if wri.inter
				outobj << escape(text)
				aelm = node.createElementSimply('a', {'href'=>href}, '?')
				a_onhtml(aelm, outobj) if not @sys.static? or @sys['STATICWIKI']
			end
			return outobj
		end
		def plugin_onhtml(node, outobj='')
			return outobj unless @sys and @sys.respond_to?(:plugin)
			name   = node.getAttribute('name')
			id     = node.getAttribute('id')
			plain  = node.getAttribute('plain')
			inline = node.getAttribute('wiki-node-type') == 'inline'
			prms   = node.getTextValues.flatten
			indent = get_indent(node)
			res    = @sys.plugin_onview(name, id, prms, plain, inline).chomp
			unless @sys['PLUGINENCLOSE'] then
				outobj << indent if indent
				outobj << res
			else
				tag = inline ? 'span' : 'div'
				attr = {'class' => "plugin-#{name}"}
				if inline then
					attr['wiki-node-type'] = 'inline'
				else
					attr['wiki-indent-level'] = 0
				end
				outobj << begin_tag(tag, attr)
				outobj << indent if indent
				outobj << res
				outobj << end_tag(tag)
			end
			return outobj
		end
		def escape_onhtml(node, outobj='')
			generate(node.childNodes, outobj)
			return outobj
		end
		def _text_onhtml(node, outobj='')
			return outobj unless text = node.nodeValue
			return outobj if text.empty?
			return outobj << autolink(text) if @sys['AUTOLINK']
			return outobj << autohtml(text) if @sys['HTMLTAG']
			return outobj << escape(text)
		end
		def a_onhtml(node, outobj='')
			doc = node.ownerDocument
			text = node.getTextValues.to_s
			attr = node.getAttributeHash
			uri = node.getAttribute('href')
			attr['href'] = convert_uri(uri) if uri
			attr['id'] = attr['name'] if attr.key?('name') and not attr.key?('id')
			attr['accesskey'], text = $1, $` if RE_ACCESSKEY === text
			if RE_EMBIMG === text then
				outobj << begin_tag('a', attr)
				outobj << empty_tag('img', {'src'=>$1, 'alt'=>$'})
				outobj << end_tag('a')
			else
				text << "(#{attr['accesskey']})" if attr['accesskey']
				outobj << begin_tag('a', attr)
				outobj << escape(text)
				outobj << end_tag('a')
			end
			return outobj
		end
	private
		def convert_uri(uri)
			return uri unless @sys.static?
			return uri if File::fullpath?(uri)
			dir, sub = uri.split('/', 2)
			if dir == IMGNAM and @sys['IMGURI'] and
				( @sys['STATICIMG'] == 'dynamic' or \
				  @sys['STATICIMG'] == 'static' ) then
				uri = [@sys['IMGURI'].first, sub].compact.join('/')
				uri = uri.sub(@sys['STATICURI']+'/', '')
			elsif dir == THEMENAM and @sys['THEMEURI'] and
				( @sys['STATICTHEME'] == 'dynamic' or \
				  @sys['STATICTHEME'] == 'static' ) then
				uri = [@sys['THEMEURI'].first, sub].compact.join('/')
				uri = uri.sub(@sys['STATICURI']+'/', '')
			end
			return File::relative(uri, File::dirname(@sys.page.name))
		end
		def autolink(text)
			return text if @autolink_flag
			begin
				@autolink_flag = true
				unless @autolink_reg then
					names = @sys.pages.names.sort.reverse
					aliases = Array::new
					@sys.aliases.each do |name, als|
						next unless String === name
						next if /^\/.+\/$/ === name
						next if name.empty?
						next if als.empty?
						aliases << als
					end
					names += aliases.sort.reverse
					reg = names.map{|x|
						next if x.nil? or x.empty?
						Regexp::escape(x)
					}.compact.join('|')
					@autolink_reg = Regexp::new(reg)
				end
				return autolink_split(text)
			ensure
				@autolink_flag = false
			end
		end
		def autolink_split(text)
			if @autolink_reg === text then
				fore, name, back = $`, $&, $'
				begin
					node = @sys.xml
					if not @sys.pages.names.include?(name) and \
						key = @sys.aliases.datas.index(name) and not key.empty? \
					then
						welm = node.createElementSimply('wikiname', {'wri'=>key}, name)
					else
						welm = node.createElementSimply('wikiname', {'wri'=>name}, @sys.aliases.get(name))
					end
					link = wikiname_onhtml(welm)
				rescue
					link = escape(name)
				end
				autolink_split(fore) + link + autolink_split(back)
			elsif @sys['HTMLTAG'] then
				autohtml(text)
			else
				escape(text)
			end
		end
		def autohtml(text)
			return text unless /<\/?[A-Za-z]+[^<>]*>/ === text
			autohtml($`) + $& + autohtml($')
		end
		def get_indent(node)
			return nil unless node.respond_to?(:getAttribute)
			return nil if node.getAttribute('wiki-node-type') == 'inline'
			size = node.getAttribute('wiki-indent-level')
			return nil unless size
			return "\n" if node.nodeName == 'pre'
			indent = ' ' * size.to_i if size
			return "\n" + indent
		end
	end
end
