#
# vikiwiki_style.rb: Wiki style for tDiary 2.x format. $Revision: 1.11 $
#
# if you want to use the Wiki style, add @style into tdiary.conf below:
#
#    @style = 'xxxx'
#      xxxx is the following wiki style name.
#      VikiWiki, AsWiki, FreeStyleWiki, Hiki, PukiWiki, Tiki, YukiWiki
#
# and rename this file.
#
#    vikiwiki_style.rb => xxxx_style.rb
#
# and change the style name in the conf file 'vikiwikiconf.rb'.
#
#    'STYLE_NAME'  => 'VikiWiki',
#
# Copyright (C) 2003,2004, ASAI Etsuhisa <mopia@aqua.livedoor.com>
# You can distribute this under GPL.
#
require 'tdiary/vikiwikiconf'
require 'vikiwikiparser'

module TDiary

	class WikiBaseSection
		attr_reader :subtitle, :author
		attr_reader :categories, :stripped_subtitle

		def initialize( subtitle, fragment, author = nil )
			@author = author
			@subtitle = subtitle.dup if subtitle
			@body = fragment.gsub( /\r/, '' ).sub( /\n+\Z/, '' )+"\n\n"

			@categories = get_categories
			@stripped_subtitle = strip_subtitle
		end

		def body
			@body.dup
		end

		def body_to_html
			do_html4( nil, nil, nil )
		end

		def subtitle_to_html
			nil
		end

		def to_src
			r = ''
			r << @subtitle if @subtitle
			r << @body
		end

		def html4( date, idx, opt )
			do_html4( date, idx, opt )
		end

		def do_html4( date, idx, opt, in_stripped_subtitle = nil )
			begin
				scode = $KCODE;
				$KCODE = VIKIWIKICONF['ENCODING'][0].chr.downcase
				sys = VikiWiki::Sys::new(
					VIKIWIKICONF['LOCAL_FILE'],
					VIKIWIKICONF )
				sys.setup
				sty = VIKIWIKICONF['STYLE_NAME']+'Style'
				body = @body
				body = VIKIWIKICONV['STORE'].call( body ) if VIKIWIKICONV['STORE']
				body = VikiWiki::Parser::trans_html( sys, body, sty )
				body = VIKIWIKICONV['RESTORE'].call( body ) if VIKIWIKICONV['RESTORE']
				return body
			ensure
				$KCODE = scode
			end
		end

		def chtml( date, idx, opt )
		end

		def to_s
			to_src
		end

	private
		def get_categories
			return []
		end

		def strip_subtitle
			return nil
		end
	end

	class WikiBaseDiary
		include DiaryBase
		include CategorizableDiary

		def initialize( date, title, body, modified = Time::now )
			init_diary
			replace( date, title, body )
			@last_modified = modified
		end

		def style
			VIKIWIKICONF['STYLE_NAME']
		end

		def replace( date, title, body )
			set_date( date )
			set_title( title )
			@sections = []
			append( body )
		end

		def append( body, author = nil )
			section = TDiary::const_get( style.capitalize + 'Section' )
			@sections << section.new( nil, body, author )
			@last_modified = Time::now
			self
		end

		def each_section
			@sections.each do |section|
				yield section
			end
		end

		def to_src
			r = ''
			each_section do |section|
				r << section.to_src
			end
			r
		end

		def to_html( opt, mode = :HTML )
			case mode
			when :CHTML
				to_chtml( opt )
			else
				to_html4( opt )
			end
		end

		def to_html4( opt )
			r = ''
			idx = 1
			each_section do |section|
				r << section.html4( date, idx, opt ).gsub( /\<h3[^\>]*\>/ ){
					h3 = $&
					rr = "<a "
					if opt['anchor'] then
						rr << %Q[name="p#{'%02d' % idx}" ]
					end
					rr << %Q[href="#{opt['index']}<%=anchor "#{date.strftime( '%Y%m%d' )}#p#{'%02d' % idx}" %>">#{opt['section_anchor']}</a> ]
					if opt['multi_user'] and @author then
						rr << %Q|[#{@author}]|
					end
					idx += 1
					h3+rr
				}
			end
			r
		end

		def to_chtml( opt )
			r = ''
			idx = 1
			each_section do |section|
				r << section.chtml( date, idx, opt )
				idx += 1
			end
			r
		end

		def to_s
			"date=#{date.strftime('%Y%m%d')}, title=#{title}, body=[#{@sections.join('][')}]"
		end

		def categorizable?
			true
		end
	end
end

module TDiary
	class VikiwikiSection < WikiBaseSection; end
	class VikiwikiDiary < WikiBaseDiary; end
end
