package jp.sourceforge.masasa.architecture.framework.validation;

import java.util.ArrayList;
import java.util.List;

import jp.sourceforge.masasa.architecture.framework.validation.annotation.CheckEmail;
import junit.framework.TestCase;

import org.apache.commons.validator.ValidatorException;

/**
 * @see org.apache.commons.validator.EmailTest
 */
public class CheckEmailTest extends TestCase {

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmail() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with numeric domains.
	 */
	public void testEmailWithNumericAddress1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("someone@[216.109.118.76]"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with numeric domains.
	 */
	public void testEmailWithNumericAddress2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("someone@yahoo.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension3() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.net"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension4() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.info"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension5() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache."));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension6() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("jsmith@apache.c"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension7() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("someone@yahoo.museum"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation.
	 */
	public void testEmailExtension8() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("someone@yahoo.mu-seum"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * <p>
	 * Tests the e-mail validation with a dash in the address.
	 * </p>
	 */
	public void testEmailWithDash1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy.noble@data-workshop.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * <p>
	 * Tests the e-mail validation with a dash in the address.
	 * </p>
	 */
	public void testEmailWithDash2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy-noble@data-workshop.-com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * <p>
	 * Tests the e-mail validation with a dash in the address.
	 * </p>
	 */
	public void testEmailWithDash3() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy-noble@data-workshop.c-om"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * <p>
	 * Tests the e-mail validation with a dash in the address.
	 * </p>
	 */
	public void testEmailWithDash4() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy-noble@data-workshop.co-m"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation with a dot at the end of the address.
	 */
	public void testEmailWithDotEnd() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy.noble@data-workshop.com."));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation with an RCS-noncompliant character in the
	 * address.
	 */
	public void testEmailWithBogusCharacter1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy.noble@\u008fdata-workshop.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation with an RCS-noncompliant character in the
	 * address.
	 */
	public void testEmailWithBogusCharacter2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy.o'reilly@data-workshop.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation with an RCS-noncompliant character in the
	 * address.
	 */
	public void testEmailWithBogusCharacter3() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("andy@o'reilly.data-workshop.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the e-mail validation with an RCS-noncompliant character in the
	 * address.
	 */
	public void testEmailWithBogusCharacter4() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("foo+bar@i.am.not.in.us.example.com"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with commas.
	 */
	public void testEmailWithCommas1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@apa,che.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with commas.
	 */
	public void testEmailWithCommas2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@apache.o,rg"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with commas.
	 */
	public void testEmailWithCommas3() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@apache,org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow @apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces2() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@ apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces3() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data(" joeblow@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces4() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@apache.org "));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces5() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joe blow@apache.org "));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with spaces.
	 */
	public void testEmailWithSpaces6() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joeblow@apa che.org "));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Tests the email validation with ascii control characters. (i.e. Ascii
	 * chars 0 - 31 and 127)
	 */
	public void testEmailWithControlChars() throws ValidatorException {
		List<ITestData> list;
		Validator validator;
		for (char c = 0; c < 32; c++) {
			list = new ArrayList<ITestData>();
			list.add(new Data("foo" + c + "bar@domain.com"));
			validator = new Validator();
			validator.validate(list);
			assertFalse(validator.getErrors().isEmpty());
		}
		list = new ArrayList<ITestData>();
		list.add(new Data("foo" + ((char) 127) + "bar@domain.com"));
		validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Write this test according to parts of RFC, as opposed to the type of
	 * character that is being tested.
	 *
	 * <p>
	 * <b>FIXME</b>: This test fails so disable it with a leading _ for 1.1.4
	 * release. The real solution is to fix the email parsing.
	 *
	 * @throws ValidatorException
	 */
	public void _testEmailUserName1() throws ValidatorException {
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joe1blow@apache.org"));
		list.add(new Data("joe$blow@apache.org"));
		list.add(new Data("joe-@apache.org"));
		list.add(new Data("joe_@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	/**
	 * Write this test according to parts of RFC, as opposed to the type of
	 * character that is being tested.
	 *
	 * <p>
	 * <b>FIXME</b>: This test fails so disable it with a leading _ for 1.1.4
	 * release. The real solution is to fix the email parsing.
	 *
	 * @throws ValidatorException
	 */
	public void _testEmailUserName2() throws ValidatorException {
		// UnQuoted Special characters are invalid
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("joe.@apache.org"));
		list.add(new Data("joe+@apache.org"));
		list.add(new Data("joe!@apache.org"));
		list.add(new Data("joe*@apache.org"));
		list.add(new Data("joe'@apache.org"));
		list.add(new Data("joe(@apache.org"));
		list.add(new Data("joe)@apache.org"));
		list.add(new Data("joe,@apache.org"));
		list.add(new Data("joe%45@apache.org"));
		list.add(new Data("joe;@apache.org"));
		list.add(new Data("joe?@apache.org"));
		list.add(new Data("joe&@apache.org"));
		list.add(new Data("joe=@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertFalse(validator.getErrors().isEmpty());
	}

	/**
	 * Write this test according to parts of RFC, as opposed to the type of
	 * character that is being tested.
	 *
	 * <p>
	 * <b>FIXME</b>: This test fails so disable it with a leading _ for 1.1.4
	 * release. The real solution is to fix the email parsing.
	 *
	 * @throws ValidatorException
	 */
	public void _testEmailUserName3() throws ValidatorException {
		// Quoted Special characters are valid
		List<ITestData> list = new ArrayList<ITestData>();
		list.add(new Data("\"joe.\"@apache.org"));
		list.add(new Data("\"joe+\"@apache.org"));
		list.add(new Data("\"joe!\"@apache.org"));
		list.add(new Data("\"joe*\"@apache.org"));
		list.add(new Data("\"joe'\"@apache.org"));
		list.add(new Data("\"joe(\"@apache.org"));
		list.add(new Data("\"joe)\"@apache.org"));
		list.add(new Data("\"joe,\"@apache.org"));
		list.add(new Data("\"joe%45\"@apache.org"));
		list.add(new Data("\"joe;\"@apache.org"));
		list.add(new Data("\"joe?\"@apache.org"));
		list.add(new Data("\"joe&\"@apache.org"));
		list.add(new Data("\"joe=\"@apache.org"));
		Validator validator = new Validator();
		validator.validate(list);
		assertTrue(validator.getErrors().isEmpty());
	}

	public class  Data implements ITestData {
		private static final long serialVersionUID = -7966549989771835759L;
		@CheckEmail
		private String data;

		public Data() {
		}

		public Data(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}

	public class  DataVerifyWhenNull implements ITestData {
		private static final long serialVersionUID = 7210939508079923660L;
		@CheckEmail(verifyWhenNull = true)
		private String data;

		public DataVerifyWhenNull() {
		}

		public DataVerifyWhenNull(String data) {
			this.data = data;
		}

		public String getData() {
			return data;
		}

		public void setData(String data) {
			this.data = data;
		}
	}
}
