package jp.sourceforge.masasa.architecture.framework.validation.validator;

import java.lang.annotation.Annotation;

import jp.sourceforge.masasa.architecture.framework.validation.util.MethodAccessUtil;

/**
 * バリデーション機能のインタフェースを提供します.
 * <p>
 * <ol>
 * 相関チェックなどで<code>lineObject</code>からフィールド値を取得する場合の実装例.
 * <li>{@link IValidator}を実装するクラス内で、 {@link MethodAccessUtil#doInvoke(Object, String)}を使い、相関対象の値を取得する. <br>
 * <ul>
 * <li>第一引数 : invoke対象のオブジェクト.</li>
 * <li>第二引数 : invoke対象のフィールド名.</li>
 * </ul>
 * <br>
 * ヒント．第二引数はアノテーションで渡せるようにすると良い.
 * 
 * <pre>
 * &#064;Target(ElementType.FIELD)
 * &#064;Retention(RetentionPolicy.RUNTIME)
 * &#064;ValidateResolver(CheckRulesEnum.XXXCheck)
 * public @interface CheckXXX {
 *     // 相関フィールドを返却する.
 *     String[] correlationFields();
 * }
 * </pre>
 * 
 * </li>
 * <li>{@link IValidator}を実装するクラス内で、相関チェックを行う.
 * 
 * <pre>
 * // targetValueがアノテーションで渡されるどのフィールド値よりも小さいことをチェックする例.
 * public boolean validate(final Annotation anno, final String targetValue, final Object lineObject) {
 *     CheckXXX resolver = (CheckXXX) anno;
 *     if (!resolver.verifyWhenNull() &amp;&amp; targetValue == null) { return true; }
 *     String[] fieldNames = resolver.correlationFields();
 *     String compareValue = null;
 *     for (int i = 0; i &lt; fieldNames.length(); i++) {
 *         compareValue = MethodAccessUtil.doInvoke(lineObject, fieldNames[i]);
 *         if (targetValue.compareTo(compareValue) &gt; 0) { return false; }
 *     }
 *     return true;
 * }
 * </pre>
 * 
 * </li>
 * </ol>
 * 
 * @author masasa.
 */
public interface IValidator {
    /**
     * バリデーションを実行します.
     * 
     * @param anno
     *            バリデーションに対応するアノテーション.
     * @param targetValue
     *            検証値.
     * @param lineObject
     *            Beanが所属するオブジェクト.
     * @return バリデーション結果.
     *         <ul>
     *         <li><code>true</code>:エラーなし.</li>
     *         <li><code>false</code>:エラーあり.</li>
     *         </ul>
     */
    boolean validate(final Annotation anno, final String targetValue, final Object lineObject);
}
