package net.osdn.util.rest.client;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.Proxy;
import java.net.URLEncoder;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.YearMonth;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.osdn.util.json.jackson.JacksonDatetimeModule;
import okhttp3.Authenticator;
import okhttp3.ConnectionPool;
import okhttp3.Credentials;
import okhttp3.HttpUrl;
import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.Route;

public class Server {
	
	private static ObjectMapper mapper;
	private static JsonFactory factory;
	private static final MediaType APPLICATION_JSON = MediaType.parse("application/json; charset=utf-8");
	
	private static String url;
	private static String username;
	private static String password;
	private static Proxy  proxy;

	private static long     connectTimeoutValue    = 10;
	private static TimeUnit connectTimeoutUnit     = TimeUnit.SECONDS;
	private static long     readTimeoutValue       = 10;
	private static TimeUnit readTimeoutUnit        = TimeUnit.SECONDS;
	private static long     writeTimeoutValue      = 10;
	private static TimeUnit writeTimeoutUnit       = TimeUnit.SECONDS;
	
	private static int      maxIdleConnections     = 5;
	private static long     keepAliveDurationValue = 5;
	private static TimeUnit keepAliveDurationUnit  = TimeUnit.MINUTES;
	
	private static ConnectionPool connectionPool;
	
	private static ThreadLocal<Instance> instances = new ThreadLocal<Instance>() {
		@Override
		protected Instance initialValue() {
			Instance instance = new Instance();
			synchronized (Server.class) {
				instance.setUrl(url);
				instance.setUsername(username);
				instance.setPassword(password);
				instance.setProxy(proxy);
				
				instance.setConnectTimeout(connectTimeoutValue, connectTimeoutUnit);
				instance.setReadTimeout(readTimeoutValue, readTimeoutUnit);
				instance.setWriteTimeout(writeTimeoutValue, writeTimeoutUnit);
				
				if(connectionPool == null) {
					connectionPool = new ConnectionPool(
						maxIdleConnections,
						keepAliveDurationValue,
						keepAliveDurationUnit
					);
				}
				instance.setConnectionPool(connectionPool);
			}
			return instance;
		}
	};
	
	static {
		mapper = new ObjectMapper();
		mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
		
		JacksonDatetimeModule datetimeModule = new JacksonDatetimeModule();
		mapper.registerModule(datetimeModule);
		
		factory = mapper.getFactory();
	}
	
	public static Server.Instance newInstance() {
		return new Instance();
	}
	
	public static synchronized void setUrl(String url, String username, String password) {
		Server.url = url;
		Server.username = username;
		Server.password = password;
	}
	
	public static synchronized void setUrl(String url, String username, String password, Proxy proxy) {
		Server.url = url;
		Server.username = username;
		Server.password = password;
		Server.proxy = proxy;
	}
	
	public static synchronized void setUrl(String url) {
		Server.url = url;
	}
	
	public static synchronized void setUsername(String username) {
		Server.username = username;
	}
	
	public static synchronized void setPassword(String password) {
		Server.password = password;
	}
	
	public static synchronized void setConnectTimeout(long timeout, TimeUnit unit) {
		Server.connectTimeoutValue = timeout;
		Server.connectTimeoutUnit = unit;
	}
	
	public static synchronized void setReadTimeout(long timeout, TimeUnit unit) {
		Server.readTimeoutValue = timeout;
		Server.readTimeoutUnit = unit;
	}
	
	public static synchronized void setWriteTimeout(long timeout, TimeUnit unit) {
		Server.writeTimeoutValue = timeout;
		Server.writeTimeoutUnit = unit;
	}
	
	public static synchronized void setConnectionPool(int maxIdleConnections, long keepAliveDuration, TimeUnit unit) {
		Server.maxIdleConnections = maxIdleConnections;
		Server.keepAliveDurationValue = keepAliveDuration;
		Server.keepAliveDurationUnit = unit;
	}
	
	public static Instance path(String segment) {
		return instances.get().reset().path(segment);
	}
	
	public static Instance path(String... segments) {
		return instances.get().reset().path(segments);
	}
	
	public static Instance param(String name, String value) {
		return instances.get().reset().param(name, value);
	}

	public static Instance param(String name, Number value) {
		return instances.get().reset().param(name, value);
	}
	
	public static Instance param(String name, Date value) {
		return instances.get().reset().param(name, value);
	}
	
	public static Instance param(String name, LocalDateTime value) {
		return instances.get().reset().param(name, value);
	}
	
	public static Instance param(String name, LocalDate value) {
		return instances.get().reset().param(name, value);
	}
	
	public static Instance param(String name, LocalTime value) {
		return instances.get().reset().param(name, value);
	}

	public static Instance param(String name, OffsetDateTime value) {
		return instances.get().reset().param(name, value);
	}

	public static Instance param(String name, Object value) {
		return instances.get().reset().param(name, value);
	}

	public static Instance param(String name, OffsetTime value) {
		return instances.get().reset().param(name, value);
	}
	
	public static Instance param(String name, YearMonth value) {
		return instances.get().reset().param(name, value);
	}

	public static Instance body(Object body) throws JsonProcessingException {
		return instances.get().reset().body(body);
	}
	
	public static Instance body(RequestBody body) {
		return instances.get().reset().body(body);
	}
	
	public static Instance multipart() {
		return instances.get().reset().multipart();
	}
	
	public static Instance form(String name, Object value) throws JsonProcessingException {
		return instances.get().reset().form(name, value);
	}
	
	protected static String getString(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getString(parser);
	}
	
	protected static String getString(JsonParser parser) throws JsonParseException, IOException {
		String value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = parser.getValueAsString();
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = parser.getValueAsString();
				break LOOP;
			case VALUE_STRING:
				value = parser.getValueAsString();
				break LOOP;
			case VALUE_TRUE:
				value = parser.getValueAsString();
				break LOOP;
			case VALUE_FALSE:
				value = parser.getValueAsString();
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static Boolean getBoolean(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getBoolean(parser);
	}
	
	protected static Boolean getBoolean(JsonParser parser) throws JsonParseException, IOException {
		Boolean value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = parser.getValueAsInt() != 0;
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = parser.getValueAsInt() != 0;
				break LOOP;
			case VALUE_STRING:
				String s = parser.getValueAsString();
				if(s != null) {
					s = s.toLowerCase().trim();
					if(s.equals("true")) {
						value = Boolean.TRUE;
					} else if(s.equals("false")) {
						value = Boolean.FALSE;
					}
				}
				break LOOP;
			case VALUE_TRUE:
				value = Boolean.TRUE;
				break LOOP;
			case VALUE_FALSE:
				value = Boolean.FALSE;
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static Integer getInteger(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getInteger(parser);
	}
	
	protected static Integer getInteger(JsonParser parser) throws JsonParseException, IOException {
		Integer value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = parser.getValueAsInt();
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = parser.getValueAsInt();
				break LOOP;
			case VALUE_STRING:
				value = parser.getValueAsInt();
				break LOOP;
			case VALUE_TRUE:
				value = 1;
				break LOOP;
			case VALUE_FALSE:
				value = 0;
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static Long getLong(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getLong(parser);
	}
	
	protected static Long getLong(JsonParser parser) throws JsonParseException, IOException {
		Long value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = parser.getValueAsLong();
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = parser.getValueAsLong();
				break LOOP;
			case VALUE_STRING:
				value = parser.getValueAsLong();
				break LOOP;
			case VALUE_TRUE:
				value = 1L;
				break LOOP;
			case VALUE_FALSE:
				value = 0L;
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static Float getFloat(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getFloat(parser);
	}
	
	protected static Float getFloat(JsonParser parser) throws JsonParseException, IOException {
		Float value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = (float)parser.getValueAsInt();
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = (float)parser.getValueAsDouble();
				break LOOP;
			case VALUE_STRING:
				value = (float)parser.getValueAsDouble();
				break LOOP;
			case VALUE_TRUE:
				value = 1f;
				break LOOP;
			case VALUE_FALSE:
				value = 0f;
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static Double getDouble(Response response) throws JsonParseException, IOException {
		InputStream in = response.body().byteStream();
		JsonParser parser = factory.createParser(in);
		return getDouble(parser);
	}
	
	protected static Double getDouble(JsonParser parser) throws JsonParseException, IOException {
		Double value = null;
		
		JsonToken token;
		LOOP:
		while((token = parser.nextToken()) != null) {
			switch(token) {
			case VALUE_NUMBER_INT:
				value = (double)parser.getValueAsInt();
				break LOOP;
			case VALUE_NUMBER_FLOAT:
				value = parser.getValueAsDouble();
				break LOOP;
			case VALUE_STRING:
				value = parser.getValueAsDouble();
				break LOOP;
			case VALUE_TRUE:
				value = 1.0;
				break LOOP;
			case VALUE_FALSE:
				value = 0.0;
				break LOOP;
			case VALUE_NULL:
				value = null;
				break LOOP;
			default:
			}
		}
		return value;
	}
	
	protected static int getRowCount(Response response) {
		try {
			String s = response.header("X-RowCount");
			if(s != null && !s.isEmpty()) {
				return Integer.parseInt(s);
			}
		} catch(Exception e) {}
		return -1;
	}
	
	protected static int getColumnCount(Response response) {
		try {
			String s = response.header("X-ColumnCount");
			if(s != null && !s.isEmpty()) {
				return Integer.parseInt(s);
			}
			
		} catch(Exception e) {}
		return -1;
	}
	
	
	private static String toJSON(Object value) {
		try {
			return mapper.writeValueAsString(value);
		} catch (JsonProcessingException e) {
			throw new RuntimeException(e);
		}
	}
	
	public static class Instance {
		private String url;
		private String username;
		private String password;
		private Proxy proxy;
		
		private long     connectTimeoutValue;
		private TimeUnit connectTimeoutUnit;
		private long     readTimeoutValue;
		private TimeUnit readTimeoutUnit;
		private long     writeTimeoutValue;
		private TimeUnit writeTimeoutUnit;
		
		private ConnectionPool connectionPool;
		
		private OkHttpClient client;
		private HttpUrl httpUrl;
		private String authorization;
		private List<String> pathSegments = new ArrayList<String>();
		private Map<String, String> queryParameters = new LinkedHashMap<String, String>();
		private RequestBody requestBody;
		private boolean useMultipart;
		private Map<String, Object> formData;
		
		protected Instance() {
		}
		
		public void setUrl(String url) {
			this.url = url;

			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setUsername(String username) {
			this.username = username;
			
			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setPassword(String password) {
			this.password = password;

			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setProxy(Proxy proxy) {
			this.proxy = proxy;
			
			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setConnectTimeout(long timeout, TimeUnit unit) {
			this.connectTimeoutValue = timeout;
			this.connectTimeoutUnit = unit;
			
			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setReadTimeout(long timeout, TimeUnit unit) {
			this.readTimeoutValue = timeout;
			this.readTimeoutUnit = unit;

			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setWriteTimeout(long timeout, TimeUnit unit) {
			this.writeTimeoutValue = timeout;
			this.writeTimeoutUnit = unit;

			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		public void setConnectionPool(int maxIdleConnections, long keepAliveDuration, TimeUnit unit) {
			setConnectionPool(new ConnectionPool(maxIdleConnections, keepAliveDuration, unit));
		}
		
		public void setConnectionPool(ConnectionPool connectionPool) {
			this.connectionPool = connectionPool;
			
			this.client = null;
			this.httpUrl = null;
			this.authorization = null;
		}
		
		protected OkHttpClient getClient() {
			if(client == null) {
				if(connectionPool == null) {
					this.connectionPool = new ConnectionPool(
						Server.maxIdleConnections,
						Server.keepAliveDurationValue,
						Server.keepAliveDurationUnit
					);
				}
				client = new OkHttpClient().newBuilder()
						.connectTimeout(connectTimeoutValue, connectTimeoutUnit)
						.readTimeout(readTimeoutValue, readTimeoutUnit)
						.writeTimeout(writeTimeoutValue, writeTimeoutUnit)
						.connectionPool(connectionPool)
						.proxy(proxy)
						.build();
			}
			return client;
		}
		
		protected HttpUrl getHttpUrl() {
			if(httpUrl == null) {
				httpUrl = HttpUrl.parse(url);
			}
			return httpUrl;
		}
		
		public Instance reset() {
			pathSegments.clear();
			queryParameters.clear();
			requestBody = null;
			useMultipart = false;
			formData = null;
			return this;
		}
		
		public Instance path(String segment) {
			if(segment.indexOf('/') < 0) {
				pathSegments.add(segment);
			} else {
				for(String s : 	segment.split("/")) {
					if(s != null && s.length() > 0) {
						pathSegments.add(s);
					}
				}
			}
			return this;
		}
		
		public Instance path(String... segments) {
			for(String segment : segments) {
				pathSegments.add(segment);
			}
			return this;
		}
		
		public Instance param(String name, Object value) {
			if(value == null) {
				queryParameters.put(name, "null");
			} else {
				queryParameters.put(name, toJSON(value));
			}
			return this;
		}
		
		public Instance body(Object body) throws JsonProcessingException {
			String json = mapper.writeValueAsString(body);
			RequestBody requestBody = RequestBody.create(APPLICATION_JSON, json);
			body(requestBody);
			return this;
		}
		
		public Instance body(RequestBody body) {
			formData = null;
			requestBody = body;
			return this;
		}
		
		public Instance multipart() {
			useMultipart = true;
			return this;
		}
		
		public Instance form(String name, Object value) throws JsonProcessingException {
			requestBody = null;

			if(formData == null) {
				formData = new LinkedHashMap<String, Object>();
			}
			String json = mapper.writeValueAsString(value);
			formData.put(name, json);
			return this;
		}

		protected Request createRequest() throws JsonProcessingException {
			return createRequest((Map<String, String>)null);
		}
		
		protected Request createRequest(Map<String, String> headers) throws JsonProcessingException {
			HttpUrl.Builder urlBuilder = getHttpUrl().newBuilder();
			for(String pathSegment : pathSegments) {
				urlBuilder.addPathSegment(pathSegment);
			}
			for(Entry<String, String> param : queryParameters.entrySet()) {
				urlBuilder.addQueryParameter(param.getKey(), param.getValue());
			}
			HttpUrl url = urlBuilder.build();
			
			Request.Builder requestBuilder = new Request.Builder().url(url);
			
			if(username != null || password != null) {
				if(authorization == null) {
					authorization = Credentials.basic(username, password);
				}
				requestBuilder.header("Authorization", authorization);
			}
			
			if(headers != null) {
				for(Entry<String, String> header : headers.entrySet()) {
					requestBuilder.addHeader(header.getKey(), header.getValue());
				}
			}
			
			// POST
			if(formData != null) {
				if(useMultipart) {
					MultipartBody.Builder builder = new MultipartBody.Builder().setType(MultipartBody.FORM);
					for(Entry<String, Object> e : formData.entrySet()) {
						String name = e.getKey();
						String json = mapper.writeValueAsString(e.getValue());
						builder.addFormDataPart(name, json);
					}
					requestBody = builder.build();
				} else {
					StringBuilder sb = new StringBuilder();
					for(Entry<String, Object> e : formData.entrySet()) {
						try {
							String name = URLEncoder.encode(e.getKey(), "UTF-8");
							String json = mapper.writeValueAsString(e.getValue());
							String value = URLEncoder.encode(json, "UTF-8");
							sb.append("&");
							sb.append(name);
							sb.append("=");
							sb.append(value);
						} catch(UnsupportedEncodingException ex) {}
					}
					if(sb.length() > 0) {
						sb.deleteCharAt(0);
					}
					requestBody = RequestBody.create(MediaType.parse("application/x-www-form-urlencoded"), sb.toString());
				}
			}
			if(requestBody != null) {
				return requestBuilder.post(requestBody).build();
			}
			
			// GET
			return requestBuilder.get().build();
		}
		
		public String get() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				String value = response.body().string();
				return value;
			}
		}
		
		public String getString() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getString(response);
			}
		}
		
		public String getString(String defaultValue) throws IOException, HttpException {
			String value = getString();
			if(value == null) {
				return defaultValue;
			}
			return value;
		}
		
		public Boolean getBoolean() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getBoolean(response);
			}
		}
		
		public boolean getBoolean(boolean defaultValue) throws IOException, HttpException {
			Boolean value = getBoolean();
			if(value == null) {
				return defaultValue;
			}
			return value.booleanValue();
		}
		
		public Integer getInt() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getInteger(response);
			}
		}
		
		public int getInt(int defaultValue) throws IOException, HttpException {
			Integer value = getInt();
			if(value == null) {
				return defaultValue;
			}
			return value.intValue();
		}
		
		public Long getLong() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getLong(response);
			}
		}
		
		public long getLong(long defaultValue) throws IOException, HttpException {
			Long value = getLong();
			if(value == null) {
				return defaultValue;
			}
			return value.longValue();
		}
		
		public Float getFloat() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getFloat(response);
			}
		}
		
		public float getFloat(float defaultValue) throws IOException, HttpException {
			Float value = getFloat();
			if(value == null) {
				return defaultValue;
			}
			return value.floatValue();
		}
		
		public Double getDouble() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				return Server.getDouble(response);
			}
		}
		
		public double getDouble(double defaultValue) throws IOException, HttpException {
			Double value = getDouble();
			if(value == null) {
				return defaultValue;
			}
			return value.doubleValue();
		}
		
		public <T> T get(Class<T> returnType) throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				T value = null;
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				JsonObjectIterator<T> iterator = new JsonObjectIterator<T>(returnType, mapper, parser);
				if(iterator.hasNext()) {
					value = iterator.next();
				}
				return value;
			}
		}
		
		public <T> List<T> getList(Class<T> returnClass) throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				JsonObjectIterator<T> iterator = new JsonObjectIterator<T>(returnClass, mapper, parser);
				List<T> list = new ArrayList<T>();
				while(iterator.hasNext()) {
					T item = iterator.next();
					list.add(item);
				}
				return list;
			}
		}
		
		public <T> JsonObjectIterable<T> getIterable(Class<T> returnClass) throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				JsonObjectIterator<T> iterator = new JsonObjectIterator<T>(returnClass, mapper, parser, getRowCount(response));
				JsonObjectIterable<T> iterable = new JsonObjectIterable<T>(iterator);
				return iterable;
			}
		}
		
		public <T> JsonObjectIterator<T> getIterator(Class<T> returnClass) throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				JsonObjectIterator<T> iterator = new JsonObjectIterator<T>(returnClass, mapper, parser, getRowCount(response));
				return iterator;
			}
		}
		
		/*
		 * 
		 * Map<String, String>
		 * 
		 */
		public Map<String, String> getMap() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				Map<String, String> value = null;
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				
				@SuppressWarnings({ "rawtypes", "unchecked" })
				JsonObjectIterator<Map<String, String>> iterator = new JsonObjectIterator(LinkedHashMap.class, mapper, parser);
				if(iterator.hasNext()) {
					value = iterator.next();
				}
				return value;
			}
		}
		
		public List<Map<String, String>> getMapList() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				
				@SuppressWarnings({ "rawtypes", "unchecked" })
				JsonObjectIterator<Map<String, String>> iterator = new JsonObjectIterator(LinkedHashMap.class, mapper, parser);
				List<Map<String, String>> list = new ArrayList<Map<String, String>>();
				while(iterator.hasNext()) {
					Map<String, String> item = iterator.next();
					list.add(item);
				}
				return list;
			}
		}
		
		public JsonObjectIterable<Map<String, String>> getMapIterable() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);

				@SuppressWarnings({ "rawtypes", "unchecked" })
				JsonObjectIterator<Map<String, String>> iterator = new JsonObjectIterator(LinkedHashMap.class, mapper, parser, getRowCount(response));
				JsonObjectIterable<Map<String, String>> iterable = new JsonObjectIterable<Map<String, String>>(iterator);
				return iterable;
			}
		}
		
		public JsonObjectIterator<Map<String, String>> getMapIterator() throws IOException, HttpException {
			Request request = createRequest();
			try(Response response = getClient().newCall(request).execute()) {
				if(!response.isSuccessful()) {
					throw HttpException.byStatus(response.code(), response.message(), response.body().string());
				}
				
				InputStream in = response.body().byteStream();
				JsonParser parser = factory.createParser(in);
				
				@SuppressWarnings({ "rawtypes", "unchecked" })
				JsonObjectIterator<Map<String, String>> iterator = new JsonObjectIterator(LinkedHashMap.class, mapper, parser, getRowCount(response));
				return iterator;
			}
		}
	}
	
	public static class BasicAuthenticator implements Authenticator {
		
		private String username;
		private String password;
		
		public BasicAuthenticator(String username, String password) {
			this.username = username;
			this.password = password;
		}
		
		@Override
		public Request authenticate(Route route, Response response) throws IOException {
			String credential = Credentials.basic(username, password);
			return response.request().newBuilder().header("Authorization", credential).build();
		}
	}
}
