package net.osdn.util.sql;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.sql.rowset.CachedRowSet;

/** SQL実行ハンドラーのコールバック時に引数として渡されるSQL情報を保持しています。
 * 
 */
public class Sql {
	
	private static class EmptyParameter {}
	private static final EmptyParameter EMPTY = new EmptyParameter();
	
	private static final Pattern PARAMETER_PATTERN = Pattern.compile(
			"'(?:''|[^'])*'|--.*?$|\\/\\*.*?\\*\\/|(\\?)",
			Pattern.MULTILINE | Pattern.DOTALL);
	
	private String   original;
	private Object[] parameters;
	private String   sql;
	private CachedRowSet result;
	private int affectedRows = -1;
	
	protected Sql(String sql) {
		this(sql, new Object[]{});
	}
	
	protected Sql(String sql, Object[] parameters) {
		this.original = sql;
		this.parameters = parameters;
		
		if(this.parameters == null) {
			this.parameters = new Object[]{};
		}
		
		if(this.parameters.length == 0) {
			this.sql = sql;
		}
	}
	
	protected Sql(NamedParameterStatement statement) {
		this.original = statement.getSql();
		int maxIndex = 0;
		Map<Integer, Object> map = new HashMap<Integer, Object>();
		for(NamedParameter np : statement.getParameters()) {
			if(np.getIndex() > maxIndex) {
				maxIndex = np.getIndex();
			}
			map.put(np.getIndex(), np.getValue());
		}
		this.parameters = new Object[maxIndex];
		for(int i = 1; i <= maxIndex; i++) {
			if(map.containsKey(i)) {
				this.parameters[i - 1] = map.get(i);
			} else {
				this.parameters[i - 1] = EMPTY;
			}
		}
	}
	
	protected void setResult(CachedRowSet rowSet) {
		this.result = rowSet;
	}
	
	/** SQL実行の結果をキャッシュされた行セットとして返します。
	 * このメソッドが null 以外の有意な値を返すのは SELECT文など結果セットを返すSQLを実行して、 かつ、onAfterReturning が呼び出された後です。
	 * onBeforeExecution, onAfterThrowing が呼び出された時点では結果セットがないため、null を返します。
	 * 
	 * @return キャッシュされた行セット。結果セットがない場合は null
	 */
	public CachedRowSet getResult() {
		return this.result;
	}
	
	protected void setAffectedRows(int rows) {
		this.affectedRows = rows;
	}
	
	/** SQL実行によって影響をつけた行数を返します。
	 * このメソッドが有意な値を返すのは onAfterReturning が呼び出された後です。
	 * onBeforeExecution, onAfterThrowing が呼び出された時点ではSQL実行が正常終了していないので -1 を返します。
	 * 
	 * @return 影響を受けた行数。影響を受けた行数がない場合は -1
	 */
	public int getAffectedRows() {
		return this.affectedRows;
	}
	
	/** パラメーター置換がおこなわれていないSQL文を取得します。
	 * このメソッドが返すSQL文はパラメーター・プレースホルダー {@literal ?} を含みます。
	 * 
	 * @return パラメーター置換がおこなわれていないSQL文
	 */
	public String getOriginalSql() {
		return original;
	}
	
	/** パラメーター置換をおこなったSQL文を返します。
	 * 
	 * 通常、JDBCドライバーはパラメーターを文字列置換することなくSQLとパラメーターをデータベースに送信します。
	 * このメソッドが返すSQL文はデータベースで実行されるSQLとは厳密に一致しないことに注意してください。
	 * 
	 * @return パラメーター置換をおこなったSQL
	 */
	public String replaceParameterWithValue() {
		if(sql != null) {
			return sql;
		}
		
		try {
			int i = 0;
			StringBuffer sb = new StringBuffer();
			Matcher m = PARAMETER_PATTERN.matcher(original);
			while(m.find()) {
				if(m.group(1) != null) {
					String value;
					if(i < parameters.length) {
						value = convert(parameters[i++]);
					} else {
						value = "?";
					}
					if("?".equals(value) == false) {
						value = "(" + value + ")";
					}
					m.appendReplacement(sb, value);
				} else {
					// quoted string or comment
				}
			}
			m.appendTail(sb);
			sql = sb.toString();
		} catch(Throwable e) {
			// ignore
			sql = original;
		}
		return sql;
	}
	
	/** パラメーター置換をおこなったSQL文から改行を取り除いた文字列を返します。
	 * 
	 * @return パラメーター置換をおこなったSQLから改行を取り除いた文字列
	 */
	public String dump() {
		StringBuilder sb = new StringBuilder();
		BufferedReader r = null;
		try {
			String sql = replaceParameterWithValue();
			r = new BufferedReader(new StringReader(sql));
			String line;
			while((line = r.readLine()) != null) {
				sb.append(' ');
				sb.append(line.trim());
			}
		} catch(IOException e) {
			// ignore
		}
		if(sb.length() >= 1) {
			sb.deleteCharAt(0);
		}
		return sb.toString();
	}
	
	/** オブジェクトの文字列表現を返します。
	 * 
	 */
	@Override
	public String toString() {
		return dump();
	}
	
	protected static String convert(Object parameter) {
		if(parameter == null) {
			return "null";
		} else if(parameter instanceof String) {
			String s = (String)parameter;
			return "'" + s.replaceAll("'", "''") + "'";
		} else if(parameter instanceof Number) {
			return parameter.toString();
		} else if(parameter instanceof java.sql.Date) {
			String s = DateTimeFormatter.ISO_LOCAL_DATE.format(((java.sql.Date)parameter).toLocalDate());
			return "'" + s + "'";
		} else if(parameter instanceof java.sql.Time) {
			String s = DateTimeFormatter.ISO_LOCAL_TIME.format(((java.sql.Time)parameter).toLocalTime());
			return "'" + s + "'";
		} else if(parameter instanceof java.sql.Timestamp) {
			String s = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(((java.sql.Timestamp)parameter).toLocalDateTime());
			return "'" + s.replace('T', ' ') + "'";
		} else if(parameter instanceof java.util.Date) {
			LocalDateTime ldt = LocalDateTime.ofInstant(((java.util.Date)parameter).toInstant(), ZoneId.systemDefault());
			String s = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(ldt);
			return "'" + s.replace('T', ' ') + "'";
		} else if(parameter instanceof LocalDate) {
			String s = DateTimeFormatter.ISO_LOCAL_DATE.format((LocalDate)parameter);
			return "'" + s + "'";
		} else if(parameter instanceof LocalTime) {
			String s = DateTimeFormatter.ISO_LOCAL_TIME.format((LocalTime)parameter);
			return "'" + s + "'";
		} else if(parameter instanceof LocalDateTime) {
			String s = DateTimeFormatter.ISO_LOCAL_DATE_TIME.format((LocalDateTime)parameter);
			return "'" + s.replace('T', ' ') + "'";
		} else if(parameter instanceof OffsetTime) {
			String s = DateTimeFormatter.ISO_OFFSET_TIME.format((OffsetTime)parameter);
			return "'" + s + "'";
		} else if(parameter instanceof OffsetDateTime) {
			String s = DateTimeFormatter.ISO_OFFSET_DATE_TIME.format((OffsetDateTime)parameter);
			return "'" + s.replace('T', ' ') + "'";
		} else if(parameter instanceof EmptyParameter) {
			return "?";
		} else {
			String s = parameter.toString();
			return "'" + s.replaceAll("'", "''") + "'";
		}
	}
}
