package net.osdn.util.sql;

import java.sql.SQLException;

/** 楽観的同時実行制御例外です。
 * 
 * <p>{@link DataSource}の{@link DataSource#update(Object)}メソッド、
 * {@link DataSource#delete(Object)}メソッド、{@link DataSource#merge(Object)}メソッドでは
 * SQL文に行バージョン列が含まれている場合、自動的に楽観的同時実行制御をおこないます。
 * 楽観的同時実行制御違反が検出された場合、このOptimisticConcurrencyExceptionがスローされます。</p>
 * 
 */
public class OptimisticConcurrencyException extends SQLException {
	
	public enum Op {
		Update,
		Delete
	}
	
	private static final long serialVersionUID = 1L;

	private String message;
	private String localizedMessage;
	
	public OptimisticConcurrencyException() {
		this((Op)null);
	}
	
	public OptimisticConcurrencyException(Op op) {
		switch(op) {
		case Update:
			message = "Update statement affected an unexpected number of rows (0). Row may have been modified or deleted since row was loaded.";
			localizedMessage = "更新ステートメントが予期しない数 (0) の行に影響しました。行が読み込まれた以降に、行が変更されたか、削除されています。";
			break;
			
		case Delete:
			message = "Delete statement affected an unexpected number of rows (0). Row may have been modified or deleted since row was loaded.";
			localizedMessage = "削除ステートメントが予期しない数 (0) の行に影響しました。行が読み込まれた以降に、行が変更されたか、削除されています。";
			break;
			
		default:
			message = "Update, or delete statement affected an unexpected number of rows (0). Row may have been modified or deleted since row was loaded.";
			localizedMessage = "更新、または削除ステートメントが予期しない数 (0) の行に影響しました。行が読み込まれた以降に、行が変更されたか、削除されています。";
		}
	}
	
	public OptimisticConcurrencyException(String message) {
		super(message);
	}
	
	@Override
	public String getMessage() {
		if(message != null) {
			return message;
		}
		return super.getMessage();
	}
	
	@Override
	public String getLocalizedMessage() {
		if(localizedMessage != null) {
			return localizedMessage;
		}
		return super.getLocalizedMessage();
	}
}
