package net.osdn.util.sql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.Calendar;

/** 名前付きパラメータークラスです。
 * 
 * <p>通常、このクラスのインスタンスを直接生成することはありません。
 * NamedParameterのインスタンスは{@link NamedParameterStatement}の各メソッドによって生成され、
 * {@link NamedParameterStatement}内部に保持されます。</p>
 * 
 */
public class NamedParameter {
	
	public static enum Method {
		SET_ARRAY,
		SET_ASCII_STREAM,
		SET_BIG_DECIMAL,
		SET_BINARY_STREAM,
		SET_BLOB,
		SET_BOOLEAN,
		SET_BYTE,
		SET_BYTES,
		SET_CHARACTER_STREAM,
		SET_CLOB,
		SET_DATE,
		SET_DOUBLE,
		SET_FLOAT,
		SET_INT,
		SET_LONG,
		SET_NCHARACTER_STREAM,
		SET_NCLOB,
		SET_NSTRING,
		SET_NULL,
		SET_OBJECT,
		SET_REF,
		SET_ROWID,
		SET_SHORT,
		SET_SQLXML,
		SET_STRING,
		SET_TIME,
		SET_TIMESTAMP,
		SET_URL
	}

	private Method method;
	private int index;
	private String name;
	private Object value;
	private long length;
	private Calendar calendar;
	private int type;
	private String typeName;
	
	/** 名前付きパラメーターを作成します。
	 * 
	 * @param method {@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に呼び出されるメソッドを指定します。
	 * @param index PreparedStatementにパラメーターを設定する際のパラメーター {@literal ?}の出現順を指定します。
	 * @param name パラメーターの名前
	 * @param value パラメーターの値
	 */
	public NamedParameter(Method method, int index, String name, Object value) {
		this.method = method;
		this.index = index;
		this.name = name;
		this.value = value;
	}
	
	/** {@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に呼び出されるメソッドを設定します。
	 * 
	 * @param method メソッドを表す{@link Method}の値
	 */
	public void setMethod(Method method) {
		this.method = method;
	}
	
	/** {@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に呼び出されるメソッドを取得します。
	 * 
	 * @return メソッドを表す{@link Method}の値
	 */
	public Method getMethod() {
		return this.method;
	}
	
	/** パラメーターの出現順を設定します。
	 * 
	 * @param index パラメーターの出現順。0以下の値を指定するとIllegalArgumentExceptionがスローされます。
	 */
	public void setIndex(int index) {
		if(index <= 0) {
			throw new IllegalArgumentException();
		}
		this.index = index;
	}
	
	/** パラメーターの出現順を取得します。
	 *  
	 * @return パラメーターの出現順
	 */
	public int getIndex() {
		return this.index;
	}
	
	/** パラメーターの名前を設定します。
	 * 
	 * @param name パラメーターの名前
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/** パラメーターの名前を取得します。
	 * 
	 * @return パラメーターの名前
	 */
	public String getName() {
		return this.name;
	}

	/** パラメーターの値を設定します。
	 * 
	 * @param value パラメーターの値
	 */
	public void setValue(Object value) {
		this.value = value;
	}

	/** パラメーターの値を取得します。
	 * 
	 * @return パラメーターの値
	 */
	public Object getValue() {
		return this.value;
	}
	
	/** パラメーターの長さを設定します。
	 * 
	 * <p>この値は{@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に使われます。
	 * ただし、{@link #setMethod(Method)}メソッドで次のいずれかの値が設定されている場合に限ります:
	 * SET_ASCII_STREAM、SET_BINARY_STREAM、SET_BLOB、SET_CHARACTER_STREAM、SET_CLOB、SET_NCHARACTER_STREAM、SET_NCLOB、SET_OBJECT</p>
	 * 
	 * <p>上記のメソッド以外が設定されている場合はパラメーターの長さは無視されます。</p>
	 * 
	 * @param length パラメーターの長さ。0以下の値を指定するとパラメーターの長さが設定されていなものとして扱われます。
	 */
	public void setLength(long length) {
		this.length = length;
	}

	/** パラメーターの長さを取得します。
	 * 
	 * @return パラメーターの長さ。0以下の値はパラメーターの長さが設定されていないことを表しています。
	 */
	public long getLength() {
		return this.length;
	}
	
	/** このパラメーターを設定する際に使用するCalendarを設定します。
	 * 
	 * <p>この値は{@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に使われます。
	 * ただし、{@link #setMethod(Method)}メソッドで次のいずれかの値が設定されている場合に限ります:
	 * SET_DATE、SET_TIME、SET_TIMESTAMP</p>
	 * 
	 * <p>上記のメソッド以外が設定されている場合はCalendarは無視されます。</p>
	 * 
	 * @param calendar パラメーターを設定する際に使用するCalendar。既定のタイムゾーンを使用する場合はnullを指定します。
	 */
	public void setCalendar(Calendar calendar) {
		this.calendar = calendar;
	}
	
	/** このパラメーターを設定する際に使用するCalendarを取得します。
	 * 
	 * @return パラメーターを設定する際に使用するCalendar。設定されていない場合は null を返します。
	 */
	public Calendar getCalendar() {
		return this.calendar;
	}
	
	/** このパラメーターを設定する際に使用するSQL型コードを設定します。
	 * 
	 * <p>この値は{@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に使われます。
	 * ただし、{@link #setMethod(Method)}メソッドで次のいずれかの値が設定されている場合に限ります:
	 * SET_NULL、SET_OBJECT</p>
	 * 
	 * <p>上記のメソッド以外が設定されている場合はtypeは無視されます。</p>
	 * 
	 * @param type java.sql.Typesで定義されるSQL型コード
	 */
	public void setType(int type) {
		this.type = type;
	}
	
	/** このパラメーターを設定する際に使用されるSQL型コードを取得します。
	 * 
	 * @return java.sql.Typesで定義されるSQL型コード。設定されていない場合は0を返します。
	 */
	public int getType() {
		return this.type;
	}
	
	/** このパラメーターを設定する際に使用するSQLユーザー定義型の完全指定の名前を設定します。
	 * 
	 * <p>この値は{@link #applyTo(PreparedStatement)}メソッドを呼び出してPreparedStatementにパラメーターを設定する際に使われます。
	 * ただし、{@link #setMethod(Method)}メソッドで次のいずれかの値が設定されている場合に限ります:
	 * SET_NULL</p>
	 * 
	 * <p>上記のメソッド以外が設定されている場合はtypeNameは無視されます</p>
	 * 
	 * @param typeName SQLユーザー定義型の完全指定の名前。パラメーターがユーザー定義型でもREFでもない場合は無視されます。
	 */
	public void setTypeName(String typeName) {
		this.typeName = typeName;
	}

	/** このパラメーターを設定する際に使用するSQLユーザー定義型の完全指定の名前を取得します。
	 * 
	 * @return SQLユーザー定義型の完全指定の名前。
	 */
	public String getTypeName() {
		return this.typeName;
	}
	
	/** 指定したPreparedStatementにこのパラメーターを設定します。
	 * 
	 * @param st PreparedStatement
	 * @throws SQLException PreparedStatementの各パラメーター設定メソッドがSQLExceptionをスローした場合
	 */
	public void applyTo(PreparedStatement st) throws SQLException {
		switch(method) {
		case SET_ARRAY:
			st.setArray(index, (java.sql.Array)value);
			break;
		case SET_ASCII_STREAM:
			if(length > 0) {
				st.setAsciiStream(index, (InputStream)value, length);
			} else {
				st.setAsciiStream(index, (InputStream)value);
			}
			break;
		case SET_BIG_DECIMAL:
			st.setBigDecimal(index, (BigDecimal)value);
			break;
		case SET_BINARY_STREAM:
			if(length > 0) {
				st.setBinaryStream(index, (InputStream)value, length);
			} else {
				st.setBinaryStream(index, (InputStream)value);
			}
			break;
		case SET_BLOB:
			if(length > 0) {
				st.setBlob(index, (InputStream)value, length);
			} else if(value instanceof InputStream) {
				st.setBlob(index, (InputStream)value);
			} else {
				st.setBlob(index, (java.sql.Blob)value);
			}
			break;
		case SET_BOOLEAN:
			st.setBoolean(index, (boolean)value);
			break;
		case SET_BYTE:
			st.setByte(index, (byte)value);
			break;
		case SET_BYTES:
			st.setBytes(index, (byte[])value);
			break;
		case SET_CHARACTER_STREAM:
			if(length > 0) {
				st.setCharacterStream(index, (Reader)value, length);
			} else {
				st.setCharacterStream(index, (Reader)value);
			}
			break;
		case SET_CLOB:
			if(length > 0) {
				st.setClob(index, (Reader)value, length);
			} else if(value instanceof Reader) {
				st.setClob(index, (Reader)value);
			} else {
				st.setClob(index, (java.sql.Clob)value);
			}
			break;
		case SET_DATE:
			if(calendar != null) {
				st.setDate(index, (java.sql.Date)value, calendar);
			} else {
				st.setDate(index, (java.sql.Date)value);
			}
			break;
		case SET_DOUBLE:
			st.setDouble(index, (double)value);
			break;
		case SET_FLOAT:
			st.setFloat(index, (float)value);
			break;
		case SET_INT:
			st.setInt(index, (int)value);
			break;
		case SET_LONG:
			st.setLong(index, (long)value);
			break;
		case SET_NCHARACTER_STREAM:
			if(length > 0) {
				st.setNCharacterStream(index, (Reader)value, length);
			} else {
				st.setNCharacterStream(index, (Reader)value);
			}
			break;
		case SET_NCLOB:
			if(length > 0) {
				st.setNClob(index, (Reader)value, length);
			} else if(value instanceof Reader) {
				st.setNClob(index, (Reader)value);
			} else {
				st.setNClob(index, (java.sql.NClob)value);
			}
			break;
		case SET_NSTRING:
			st.setNString(index, (String)value);
			break;
		case SET_NULL:
			if(typeName != null) {
				st.setNull(index, type, typeName);
			} else {
				st.setNull(index, type);
			}
			break;
		case SET_OBJECT:
			if(type != 0) {
				if(length > 0) {
					st.setObject(index, value, type, (int)length);
				} else {
					st.setObject(index, value, type);
				}
			} else {
				st.setObject(index, value);
			}
			break;
		case SET_REF:
			st.setRef(index, (java.sql.Ref)value);
			break;
		case SET_ROWID:
			st.setRowId(index, (java.sql.RowId)value);
			break;
		case SET_SHORT:
			st.setShort(index, (short)value);
			break;
		case SET_SQLXML:
			st.setSQLXML(index, (java.sql.SQLXML)value);
			break;
		case SET_STRING:
			st.setString(index, (String)value);
			break;
		case SET_TIME:
			if(calendar != null) {
				st.setTime(index, (java.sql.Time)value, calendar);
			} else {
				st.setTime(index, (java.sql.Time)value);
			}
			break;
		case SET_TIMESTAMP:
			if(calendar != null) {
				st.setTimestamp(index, (java.sql.Timestamp)value, calendar);
			} else {
				st.setTimestamp(index, (java.sql.Timestamp)value);
			}
			break;
		case SET_URL:
			st.setURL(index, (URL)value);
			break;
		default:
			st.setObject(index, value);
			break;
		}
	}
}
