package net.osdn.util.sql.h2;

import java.lang.reflect.Field;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.osdn.util.sql.NamedParameterStatement;
import net.osdn.util.sql.StatementBuilder;

/** H2データベース用のSQL文構築ビルダーです。
 * 
 * <p>このクラスは{@link StatementBuilder#createMergeStatement(DatabaseMetaData, Class, String)}を
 * オーバーライドして、H2データベース用のMERGE構文を作成します。</p>
 * 
 */
public class H2StatementBuilder extends StatementBuilder {

	/** 指定されたデータベース・メタデータが、この構文ビルダーに適合するかどうかチェックします。
	 * 
	 * <p>DatabaseMetaData.getDatabaseProductName()メソッドが
	 * 返すデータベース製品名に"H2"が含まれる場合、この構文ビルダーに適合すると見なして true を返します。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @return データベース製品名に "H2" が含まれている場合は true、そうでなければ false
	 */
	@Override
	public boolean isAcceptable(DatabaseMetaData md) throws SQLException {
		String productName = md.getDatabaseProductName();
		
		if(productName != null && productName.contains("H2")) {
			return true;
		}
		
		return false;
	}
	
	/** 指定されたデータベース・メタデータ、クラス、テーブル名から
	 * 名前付きパラメーターをサポートするMERGE文を構築します。
	 * 
	 * <p>tableNameにnull以外の値が指定された場合、これをテーブル名として扱います。
	 * tableNameがnullの場合にはclsをテーブル名として扱います。</p>
	 * 
	 * <p>clsのフィールドはMERGE文を構成する列名として使用されます。</p>
	 * 
	 * <p>このメソッドが返す名前付きパラメーター・ステートメントはパラメーターの値が設定されていない状態です。</p>
	 * 
	 * @param md データベース・メタデータ
	 * @param cls テーブル名を表すクラス。フィールドの一部は列名として扱われます。
	 * @param tableName テーブル名。nullを指定した場合はclsをテーブル名として扱います。
	 * @return 名前付きパラメーターを持つMERGE文
	 * @throws SQLException データベースアクセスエラーが発生した場合
	 */
	@Override
	public NamedParameterStatement createMergeStatement(DatabaseMetaData md, Class<?> cls, String tableName) throws SQLException {
		if(tableName == null) {
			tableName = getTableName(md, cls);
		}
		if(tableName == null) {
			throw new IllegalArgumentException();
		}

		Set<String> fieldNames = new HashSet<String>();
		Field[] fields = getFields(cls);
		for(int i = 0; i < fields.length; i++) {
			fieldNames.add(fields[i].getName().toLowerCase());
		}
		
		List<Column> columns = getColumns(md, tableName, fieldNames);
		if(columns.size() == 0) {
			throw new IllegalStateException();
		}
		
		List<Column> whereClauseColumns = new ArrayList<Column>();
		List<Column> valuesClauseColumns = new ArrayList<Column>();
		for(Column column : columns) {
			if(column.isRowIdentifier) {
				whereClauseColumns.add(column);
			}
			if(!column.isAutoincrement) {
				valuesClauseColumns.add(column);
			}
		}
		
		if(whereClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		if(valuesClauseColumns.size() == 0) {
			throw new IllegalArgumentException();
		}
		
		StringBuilder sb = new StringBuilder();
		sb.append("MERGE INTO \"");
		sb.append(tableName);
		sb.append("\" (");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append('"');
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(") KEY (");
		for(int i = 0; i < whereClauseColumns.size(); i++) {
			Column column = whereClauseColumns.get(i);
			sb.append('"');
			sb.append(column.name);
			sb.append('"');
			if(i + 1 < whereClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(") VALUES (");
		for(int i = 0; i < valuesClauseColumns.size(); i++) {
			Column column = valuesClauseColumns.get(i);
			sb.append(':');
			sb.append(column.name);
			if(i + 1 < valuesClauseColumns.size()) {
				sb.append(", ");
			}
		}
		sb.append(')');
		
		return new NamedParameterStatement(sb.toString());
	}
}
