' table/FieldProvider/FieldProvider.vb
'
' Copyright (c) 2008-2009, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Imports System.Drawing

Public Interface IFieldProvider
    Sub FieldInitialize(ByVal field As UTable.CField)
    Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor)
    Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
    Sub SetBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean)
    Sub ToggleValue(ByVal field As UTable.CField, ByVal value As Object)
    Function Setting() As UTable.CSetting
    Function CreateField() As UTable.CField
    Function CreateEditor() As IEditor
    Function Focusable() As Boolean
    Function UndoEnabled() As Boolean
    Function ImeMode() As ImeMode
    Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
    Function ValueRegularize(ByVal value As Object, ByVal editor As IEditor) As Object
    Property Caption() As String
    Property TabOrder() As Integer
    Property Clipboard(ByVal field As UTable.CField) As String
    Property UndoBuffer(ByVal field As UTable.CField) As Object
End Interface

Public Class CFieldProvider
    Implements IFieldProvider

    Private _Caption As String
    Private _TabOrder As Integer = 0

    Public BorderLine As Integer = UTable.CBorder._T Or _
                                   UTable.CBorder._B Or _
                                   UTable.CBorder._L Or _
                                   UTable.CBorder._R

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        Me.Caption = caption
    End Sub

    Public Property Caption() As String Implements IFieldProvider.Caption
        Get
            Return Me._Caption
        End Get
        Set(ByVal value As String)
            Me._Caption = value
        End Set
    End Property

    Public Property TabOrder() As Integer Implements IFieldProvider.TabOrder
        Get
            Return Me._TabOrder
        End Get
        Set(ByVal value As Integer)
            Me._TabOrder = value
        End Set
    End Property

    Public Overridable Function CreateField() As UTable.CField Implements IFieldProvider.CreateField
        Return New UTable.CField()
    End Function

    Public Overridable Sub FieldInitialize(ByVal field As UTable.CField) Implements IFieldProvider.FieldInitialize
    End Sub

    Public Overridable Function CreateEditor() As IEditor Implements IFieldProvider.CreateEditor
        Return Nothing
    End Function

    Public Overridable Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor) Implements IFieldProvider.EditorInitialize
    End Sub

    Public Overridable Function Setting() As UTable.CSetting Implements IFieldProvider.Setting
        Return Nothing
    End Function

    Public Overridable Function Focusable() As Boolean Implements IFieldProvider.Focusable
        Return True
    End Function

    Public Overridable Function UndoEnabled() As Boolean Implements IFieldProvider.UndoEnabled
        Return True
    End Function

    Public Overridable Function ImeMode() As ImeMode Implements IFieldProvider.ImeMode
        Return Windows.Forms.ImeMode.Disable
    End Function

    Public Overridable Function ValueRegularize(ByVal value As Object, ByVal editor As IEditor) As Object Implements IFieldProvider.ValueRegularize
        Return value
    End Function

    Public Overridable Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean) Implements IFieldProvider.Render
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        Me.renderBackground(g, field, s, rect, alter)
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderBackground(g, field, rect)
        End If
        If field.Table.Focused AndAlso field.Focused AndAlso field.Table.FocusFieldDecorator IsNot Nothing Then
            field.Table.FocusFieldDecorator.RenderBackground(g, field, rect)
        End If
        Me.renderForeground(g, field, s, rect, alter)
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderForeground(g, field, rect)
        End If
        If field.Table.Focused AndAlso field.Focused AndAlso field.Table.FocusFieldDecorator IsNot Nothing Then
            field.Table.FocusFieldDecorator.RenderForeground(g, field, rect)
        End If
    End Sub

    Protected Overridable Sub renderBackground(ByVal g As Graphics, _
                                               ByVal field As UTable.CField, _
                                               ByVal s As UTable.CDynamicSetting, _
                                               ByVal rect As Rectangle, _
                                               ByVal alter As Boolean)
        RenderBackgroudRect(g, rect, BackColor(field, s, alter))
    End Sub

    Protected Overridable Sub renderForeground(ByVal g As Graphics, _
                                               ByVal field As UTable.CField, _
                                               ByVal s As UTable.CDynamicSetting, _
                                               ByVal rect As Rectangle, _
                                               ByVal alter As Boolean)
        If field.Editor Is Nothing Then
            RenderValue(g, rect, Me.formatValue(field.Value), ForeColor(field, s, alter), s.Font, s.GetStringFormat)
        End If
    End Sub

    Public Overridable Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size Implements IFieldProvider.GetAdjustSize
        With field.DynamicSetting
            Return g.MeasureString(Me.formatValue(field.Value), .Font, 100000, .GetStringFormat).ToSize
        End With
    End Function

    Protected Overridable Function formatValue(ByVal v As Object) As String
        If v IsNot Nothing Then
            Return v.ToString
        Else
            Return Nothing
        End If
    End Function

    Public Shared Function BackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusBackColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusBackColor
        ElseIf field.Table.SelectRange IsNot Nothing AndAlso _
               field.Table.SelectRange.IsInclude(field) AndAlso _
               Not setting.RangedBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.RangedBackColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordBackColor
        ElseIf alter AndAlso Not setting.AlterBackColor.Equals(Color.Transparent) Then
            Return setting.AlterBackColor
        Else
            Return setting.BackColor
        End If
    End Function

    Public Shared Function ForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusForeColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusForeColor
        ElseIf field.Table.SelectRange IsNot Nothing AndAlso _
           field.Table.SelectRange.IsInclude(field) AndAlso _
           Not setting.RangedForeColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.RangedForeColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordForeColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordForeColor
        Else
            Return setting.ForeColor
        End If
    End Function

    Public Shared Function CaptionBackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If drag AndAlso _
           Not setting.DraggingBackColor.Equals(Color.Transparent) Then
            Return setting.DraggingBackColor
        ElseIf field.Table.FocusField IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionBackColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionBackColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordCaptionBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
               field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordCaptionBackColor
        Else
            Return setting.CaptionBackColor
        End If
    End Function

    Public Shared Function CaptionForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If drag AndAlso _
           Not setting.DraggingForeColor.Equals(Color.Transparent) Then
            Return setting.DraggingForeColor
        ElseIf field.Table.FocusField IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionForeColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionForeColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordCaptionForeColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordCaptionForeColor
        Else
            Return setting.CaptionForeColor
        End If
    End Function

    Public Shared Sub RenderBackgroudRect(ByVal g As Graphics, _
                                          ByVal rect As Rectangle, _
                                          ByVal color As Color)
        Using brush As New SolidBrush(color)
            g.FillRectangle(brush, rect)
        End Using
    End Sub

    Public Shared Sub RenderValue(ByVal g As Graphics, _
                                  ByVal rect As Rectangle, _
                                  ByVal formattedValue As String, _
                                  ByVal color As Color, _
                                  ByVal font As Font, _
                                  ByVal stringFormat As StringFormat)
        If Not String.IsNullOrEmpty(formattedValue) Then
            Using brush As New SolidBrush(color)
                g.DrawString(ValueSanitize(formattedValue), font, brush, New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), stringFormat)
            End Using
        End If
    End Sub

    Public Shared Sub RenderCaptionBackgroudRect(ByVal g As Graphics, _
                                                 ByVal rect As Rectangle, _
                                                 ByVal color As Color, _
                                                 ByVal style As UTable.ECaptionStyle)
        Select Case style
            Case UTable.ECaptionStyle.METALLIC
                Using b As New Drawing2D.LinearGradientBrush(rect, color, GradientColor(color, 32), Drawing2D.LinearGradientMode.Vertical)
                    g.FillRectangle(b, rect)
                End Using
                Using p As New Pen(GradientColor(color, 64))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.Right - 1, rect.Y + 1))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.X + 1, rect.Bottom - 1))
                End Using
            Case UTable.ECaptionStyle.FLAT
                Using b As New SolidBrush(color)
                    g.FillRectangle(b, rect)
                End Using
        End Select
    End Sub

    Public Shared Sub RenderCaptionValue(ByVal g As Graphics, _
                                         ByVal field As UTable.CField, _
                                         ByVal sortState As UTable.CSortState, _
                                         ByVal rect As Rectangle, _
                                         ByVal formattedValue As String, _
                                         ByVal color As Color, _
                                         ByVal font As Font, _
                                         ByVal stringFormat As StringFormat)
        Using brush As New SolidBrush(color)
            If rect.Width > 15 And field.Table.SortState.Field Is field Then
                g.DrawString(ValueSanitize(formattedValue), font, brush, _
                             New RectangleF(rect.X, rect.Y + 2, rect.Width - 15, rect.Height - 2), stringFormat)
                Dim points As New List(Of PointF)
                Select Case field.Table.SortState.Order
                    Case UTable.CSortState.EOrder.ASCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 14))
                    Case UTable.CSortState.EOrder.DESCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 6))
                End Select
                g.FillPolygon(brush, points.ToArray)
            Else
                g.DrawString(ValueSanitize(formattedValue), font, brush, _
                             New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), stringFormat)
            End If
        End Using
    End Sub

    Public Shared Sub RenderCheckBox(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean)
        Dim p As New Point(rect.Left + rect.Width / 2 - 6, rect.Top + rect.Height / 2 - 6)
        If value Then
            CheckBoxRenderer.DrawCheckBox(g, p, VisualStyles.CheckBoxState.CheckedNormal)
        Else
            CheckBoxRenderer.DrawCheckBox(g, p, VisualStyles.CheckBoxState.UncheckedNormal)
        End If
    End Sub

    Public Shared Sub RenderNarrowChild(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean)
        Dim w As Integer = NARROWCHILD_SIZE
        Dim r As New Rectangle(rect.Left + rect.Width / 2 - w / 2, rect.Top + rect.Height / 2 - w / 2, w, w)
        g.DrawRectangle(Pens.Gray, r)
        g.DrawLine(Pens.Gray, New Point(r.Left, r.Top + r.Height / 2), New Point(r.Right, r.Top + r.Height / 2))
        If Not value Then
            g.DrawLine(Pens.Gray, New Point(r.Left + r.Width / 2, r.Top), New Point(r.Left + r.Width / 2, r.Top + r.Height))
        End If
    End Sub

    Protected Shared Sub renderButton(ByVal g As Graphics, ByVal down As Boolean, ByVal formattedValue As String, ByVal s As UTable.CDynamicSetting, ByVal rect As System.Drawing.Rectangle)
        If s.Editable = UTable.EAllow.ALLOW Then
            Dim d As Integer = 0
            If down Then
                _renderButton_aux(g, rect, s, True)
                d = 2
            Else
                _renderButton_aux(g, rect, s, False)
            End If
            If Not String.IsNullOrEmpty(formattedValue) Then
                Dim t As String = ValueSanitize(formattedValue)
                Using brush As New SolidBrush(s.ForeColor)
                    g.DrawString(t, s.Font, brush, New RectangleF(rect.X, rect.Y + 2 + d, rect.Width, rect.Height - 2), s.GetStringFormat)
                End Using
            End If
        Else
            _renderButton_aux(g, rect, s, False)
            Dim t As String = ValueSanitize(formattedValue)
            If Not String.IsNullOrEmpty(t) Then
                ControlPaint.DrawStringDisabled(g, t, s.Font, Color.White, New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), s.GetStringFormat)
            End If
        End If
    End Sub

    Private Shared Sub _renderButton_aux(ByVal g As Graphics, ByVal rect As Rectangle, ByVal s As UTable.CDynamicSetting, ByVal down As Boolean)
        Using b As New Drawing2D.LinearGradientBrush(New RectangleF(rect.X, rect.Y, rect.Width, rect.Height), s.ButtonBackColor, GradientColor(s.ButtonBackColor, 32), Drawing2D.LinearGradientMode.Vertical)
            g.FillRectangle(b, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 3, rect.Height - 3))
        End Using
        Using p As New Pen(GradientColor(s.ButtonBackColor, -128))
            g.DrawRectangle(p, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 4, rect.Height - 4))
        End Using
        If down Then
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        Else
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        End If
    End Sub

    Public Shared Function ValueSanitize(ByVal v As Object) As String
        If v Is Nothing Then
            Return ""
        Else
            Return System.Text.RegularExpressions.Regex.Replace(v.ToString, "]", "-")
        End If
    End Function

    Public Shared Function GradientColor(ByVal c As Color, ByVal d As Integer) As Color
        Dim r As Integer = c.R + d
        Dim g As Integer = c.G + d
        Dim b As Integer = c.B + d
        If r > 255 Then r = 255
        If r < 0 Then r = 0
        If g > 255 Then g = 255
        If g < 0 Then g = 0
        If b > 255 Then b = 255
        If b < 0 Then b = 0
        Return Color.FromArgb(r, g, b)
    End Function

    Public Overridable Sub SetBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean) Implements IFieldProvider.SetBorder
        SetFieldBorder(field, border, merged, Me.BorderLine)
    End Sub

    Public Shared Sub SetFieldBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean, ByVal borderLine As Integer)
        If field.Desc.Layout IsNot Nothing Then
            Dim b As Integer = borderLine
            If merged Then
                b = b And (UTable.CBorder._T Or UTable.CBorder._L Or UTable.CBorder._R)
            End If
            border.Fill(field.Desc.Layout, UTable.CBorder.EBorderType.FIELD, b)
        End If
    End Sub

    Public Shared Sub SetCaptionBorder(ByVal field As UTable.CField, ByVal border As UTable.CBorder, ByVal merged As Boolean, ByVal borderLine As Integer)
        If field.Desc.Layout IsNot Nothing Then
            Dim b As Integer = borderLine
            If merged Then
                b = b And (UTable.CBorder._T Or UTable.CBorder._L Or UTable.CBorder._R)
            End If
            border.Fill(field.Desc.Layout, UTable.CBorder.EBorderType.CAPTION, b)
        End If
    End Sub

    Public Shared Sub FlatBorder(ByVal descs As Dictionary(Of Object, UTable.CRecordProvider.CFieldDesc), ByVal key1 As Object, ByVal key2 As Object)
        Dim l As UTable.CGrid.CRegion = descs(key1).Layout.Union(descs(key2).Layout)
        For Each d As UTable.CRecordProvider.CFieldDesc In descs.Values
            If TypeOf d.Provider Is CFieldProvider AndAlso d.Layout IsNot Nothing Then
                If d.Layout.Row < (l.Row + l.Rows) And (d.Layout.Row + d.Layout.Rows) > l.Row And _
                   d.Layout.Col < (l.Col + l.Cols) And (d.Layout.Col + d.Layout.Cols) > l.Col Then
                    With CType(d.Provider, CFieldProvider)
                        .BorderLine = 0
                        If d.Layout.Row = l.Row Then
                            .BorderLine = .BorderLine Or UTable.CBorder._T
                        End If
                        If d.Layout.Row + d.Layout.Rows = l.Row + l.Rows Then
                            .BorderLine = .BorderLine Or UTable.CBorder._B
                        End If
                        If d.Layout.Col = l.Col Then
                            .BorderLine = .BorderLine Or UTable.CBorder._L
                        End If
                        If d.Layout.Col + d.Layout.Cols = l.Col + l.Cols Then
                            .BorderLine = .BorderLine Or UTable.CBorder._R
                        End If
                    End With
                End If
            End If
        Next
    End Sub

    Public Shared ADJUSTSIZE_TOGGLE As New Size(18, 18)
    Public Shared CLICKWIDTH_TOGGLE As Integer = 9
    Public Shared NARROWCHILD_SIZE As Integer = 10
    Public Toggle As Boolean = False

    Public Function SetToggle(ByVal field As UTable.CField) As UTable.CField
        AddHandler field.MouseDown, AddressOf CFieldProvider._toggle_mousedown
        AddHandler field.KeyPress, AddressOf CFieldProvider._toggle_keyPress
        Me.Toggle = True
        Return field
    End Function

    Private Shared Sub _toggle_mousedown(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs)
        If TypeOf field.Desc.Provider Is CFieldProvider Then
            If e.Button = MouseButtons.Left AndAlso field.Table.FocusField Is field Then
                Dim f As UTable.CRenderCache.CField = field.Table.RenderCache.FindField(field)
                If f IsNot Nothing AndAlso _
                   (CLICKWIDTH_TOGGLE = 0 OrElse _
                   (System.Math.Abs((f.Rect.Width / 2) - location.X) < CLICKWIDTH_TOGGLE And _
                    System.Math.Abs((f.Rect.Height / 2) - location.Y) < CLICKWIDTH_TOGGLE)) AndAlso _
                    f.Field.Editable = UTable.EAllow.ALLOW Then
                    field.Desc.Provider.ToggleValue(field, Not CType(field.Value, Boolean))
                End If
            End If
        End If

    End Sub

    Private Shared Sub _toggle_keyPress(ByVal field As UTable.CField, ByVal e As System.Windows.Forms.KeyPressEventArgs)
        If TypeOf field.Desc.Provider Is CFieldProvider Then
            Using field.Table.RenderBlock
                If e.KeyChar.Equals(" "c) AndAlso field.Editable = UTable.EAllow.ALLOW Then
                    field.Desc.Provider.ToggleValue(field, Not CType(field.Value, Boolean))
                    If field.Table.SelectRange IsNot Nothing AndAlso field.Table.Setting.RangeToggleValue Then
                        For Each l As List(Of UTable.CField) In field.Table.SelectRange.FieldMatrix
                            For Each f As UTable.CField In l
                                If f IsNot Nothing AndAlso _
                                   TypeOf f.Desc.Provider Is CFieldProvider AndAlso _
                                   CType(f.Desc.Provider, CFieldProvider).Toggle AndAlso _
                                   f IsNot field AndAlso _
                                   f.Editable = UTable.EAllow.ALLOW Then
                                    f.Desc.Provider.ToggleValue(f, field.Value)
                                End If
                            Next
                        Next
                    End If
                End If
            End Using
        End If
    End Sub

    Public Overridable Sub ToggleValue(ByVal field As UTable.CField, ByVal value As Object) Implements IFieldProvider.ToggleValue
        field.SetValueIfValidated(value)
    End Sub

    Public Overridable Property Clipboard(ByVal field As UTable.CField) As String Implements IFieldProvider.Clipboard
        Get
            If field.Value IsNot Nothing Then
                Return field.Value.ToString
            Else
                Return ""
            End If
        End Get
        Set(ByVal value As String)
            field.SetValueIfValidated(value)
        End Set
    End Property

    Public Overridable Property UndoBuffer(ByVal field As UTable.CField) As Object Implements IFieldProvider.UndoBuffer
        Get
            Return field.CommittedValue
        End Get
        Set(ByVal value As Object)
            field.ValueCommit(value)
        End Set
    End Property

End Class
