' table/FieldProvider.vb
'
' Copyright (c) 2009, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Imports System.Drawing

Public Interface IFieldProvider
    Sub FieldInitialize(ByVal field As UTable.CField)
    Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor)    
    Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
    Function Setting() As UTable.CSetting
    Function CreateField() As UTable.CField
    Function CreateEditor() As IEditor    
    Function Focusable() As Boolean
    Function ImeMode() As ImeMode
    Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
    Function RenderOrder() As Integer
    Property Caption() As String
    Property TabOrder() As Integer
End Interface

Public Class CFieldProvider
    Implements IFieldProvider

    Public Const _T As Integer = 1
    Public Const _B As Integer = 2
    Public Const _L As Integer = 4
    Public Const _R As Integer = 8

    Private _Caption As String
    Private _TabOrder As Integer = 0

    Public BorderLine As Integer = _T Or _B Or _L Or _R

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        Me.Caption = caption
    End Sub

    Public Property Caption() As String Implements IFieldProvider.Caption
        Get
            Return Me._Caption
        End Get
        Set(ByVal value As String)
            Me._Caption = value
        End Set
    End Property

    Public Property TabOrder() As Integer Implements IFieldProvider.TabOrder
        Get
            Return Me._TabOrder
        End Get
        Set(ByVal value As Integer)
            Me._TabOrder = value
        End Set
    End Property

    Public Overridable Function CreateField() As UTable.CField Implements IFieldProvider.CreateField
        Return New UTable.CField()
    End Function

    Public Overridable Sub FieldInitialize(ByVal field As UTable.CField) Implements IFieldProvider.FieldInitialize
    End Sub

    Public Overridable Function CreateEditor() As IEditor Implements IFieldProvider.CreateEditor
        Return Nothing
    End Function

    Public Overridable Sub EditorInitialize(ByVal field As UTable.CField, ByVal editor As IEditor) Implements IFieldProvider.EditorInitialize
    End Sub

    Public Overridable Function Setting() As UTable.CSetting Implements IFieldProvider.Setting
        Return Nothing
    End Function

    Public Overridable Function Focusable() As Boolean Implements IFieldProvider.Focusable
        Return True
    End Function

    Public Overridable Function ImeMode() As ImeMode Implements IFieldProvider.ImeMode
        Return Windows.Forms.ImeMode.Disable
    End Function

    Public Overridable Function RenderOrder() As Integer Implements IFieldProvider.RenderOrder
        Return 0
    End Function

    Public Overridable Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean) Implements IFieldProvider.Render
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        RenderBackgroud(g, field, rect, BackColor(field, s, alter))
        RenderBorder(g, rect, Me.BorderLine, field.Table.Setting.BorderColor, field.Table.Setting.BorderStyle)
        If field.Editor Is Nothing Then
            RenderValue(g, field, rect, Me.formatValue(field.Value), ForeColor(field, s, alter), s.Font, s.GetStringFormat)
        End If
    End Sub

    Public Overridable Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size Implements IFieldProvider.GetAdjustSize
        With field.DynamicSetting
            Return g.MeasureString(Me.formatValue(field.Value), .Font, 100000, .GetStringFormat).ToSize
        End With
    End Function

    Protected Overridable Function formatValue(ByVal v As Object) As String
        If v IsNot Nothing Then
            Return v.ToString
        Else
            Return Nothing
        End If
    End Function

    Public Shared Function BackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusBackColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
            field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusBackColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordBackColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Editor IsNot Nothing Or _
                field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordBackColor
        ElseIf alter AndAlso Not setting.AlterBackColor.Equals(Color.Transparent) Then
            Return setting.AlterBackColor
        Else
            Return setting.BackColor
        End If
    End Function

    Public Shared Function ForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal alter As Boolean) As Color
        If field Is field.Table.FocusField AndAlso _
           Not setting.FocusForeColor.Equals(Color.Transparent) AndAlso _
           (field.Table.Focused Or field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusForeColor
        ElseIf field.Record Is field.Table.FocusRecord AndAlso _
               Not setting.FocusRecordForeColor.Equals(Color.Transparent) AndAlso _
               (field.Table.Focused Or field.Table.Setting.FocusColorAlways) Then
            Return setting.FocusRecordForeColor
        Else
            Return setting.ForeColor
        End If
    End Function

    Public Shared Function CaptionBackColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If field.Table.FocusField IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionBackColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionBackColor
        ElseIf drag AndAlso _
            Not setting.DraggingBackColor.Equals(Color.Transparent) Then
            Return setting.DraggingBackColor
        Else
            Return setting.CaptionBackColor
        End If
    End Function

    Public Shared Function CaptionForeColor(ByVal field As UTable.CField, ByVal setting As UTable.CDynamicSetting, ByVal drag As Boolean) As Color
        If field.Table.FocusField IsNot Nothing AndAlso _
           (field.Table.Focused Or field.Table.Editor IsNot Nothing Or field.Table.Setting.FocusColorAlways) AndAlso _
           field.Content.TopLevelContent Is field.Table.HeaderContent AndAlso _
           Not setting.FocusCaptionForeColor.Equals(Color.Transparent) AndAlso _
           field.Content.Level.Equals(field.Table.FocusField.Content.Level) AndAlso _
           field Is field.Record.FindField(field.Table.FocusField.Desc.Layout.Point) Then
            Return setting.FocusCaptionForeColor
        ElseIf drag AndAlso _
            Not setting.DraggingForeColor.Equals(Color.Transparent) Then
            Return setting.DraggingForeColor
        Else
            Return setting.CaptionForeColor
        End If
    End Function

    Public Shared Sub RenderBackgroud(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal color As Color)
        Using brush As New SolidBrush(color)
            g.FillRectangle(brush, rect)
        End Using
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderOverBackground(g, field, rect)
        End If
    End Sub

    Public Shared Sub RenderBorder(ByVal g As Graphics, ByVal rect As Rectangle, ByVal borderLine As Integer, ByVal color As Color, ByVal style As Drawing2D.DashStyle)
        Using pen As New Pen(color)
            pen.DashStyle = style
            If borderLine And _T Then g.DrawLine(pen, rect.Left, rect.Top, rect.Right, rect.Top)
            If borderLine And _B Then g.DrawLine(pen, rect.Left, rect.Bottom, rect.Right, rect.Bottom)
            If borderLine And _L Then g.DrawLine(pen, rect.Left, rect.Top, rect.Left, rect.Bottom)
            If borderLine And _R Then g.DrawLine(pen, rect.Right, rect.Top, rect.Right, rect.Bottom)
        End Using
    End Sub

    Public Shared Sub RenderValue(ByVal g As Graphics, _
                                  ByVal field As UTable.CField, _
                                  ByVal rect As Rectangle, _
                                  ByVal formattedValue As String, _
                                  ByVal color As Color, _
                                  ByVal font As Font, _
                                  ByVal stringFormat As StringFormat)
        If field.Editor IsNot Nothing Then
            Exit Sub
        End If
        If Not String.IsNullOrEmpty(formattedValue) Then
            Using brush As New SolidBrush(color)
                g.DrawString(ValueSanitize(formattedValue), font, brush, New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), stringFormat)
            End Using
        End If
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderOverValue(g, field, rect)
        End If
    End Sub

    Public Shared Sub RenderCaptionBackgroud(ByVal g As Graphics, _
                                             ByVal field As UTable.CField, _
                                             ByVal rect As Rectangle, _
                                             ByVal color As Color, _
                                             ByVal style As UTable.ECaptionStyle)
        Select Case style
            Case UTable.ECaptionStyle.METALISH
                Using b As New Drawing2D.LinearGradientBrush(rect, color, GradientColor(color, 32), Drawing2D.LinearGradientMode.Vertical)
                    g.FillRectangle(b, rect)
                End Using
                Using p As New Pen(GradientColor(color, 64))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.Right - 1, rect.Y + 1))
                    g.DrawLine(p, New Point(rect.X + 1, rect.Y + 1), New Point(rect.X + 1, rect.Bottom - 1))
                End Using
            Case UTable.ECaptionStyle.FLAT
                Using b As New SolidBrush(color)
                    g.FillRectangle(b, rect)
                End Using
        End Select
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderOverBackground(g, field, rect)
        End If
    End Sub

    Public Shared Sub RenderCaptionBorder(ByVal g As Graphics, ByVal rect As Rectangle, ByVal borderLine As Integer, ByVal color As Color)
        Using pen As New Pen(color)
            If borderLine And _T Then g.DrawLine(pen, rect.Left, rect.Top, rect.Right, rect.Top)
            If borderLine And _B Then g.DrawLine(pen, rect.Left, rect.Bottom, rect.Right, rect.Bottom)
            If borderLine And _L Then g.DrawLine(pen, rect.Left, rect.Top, rect.Left, rect.Bottom)
            If borderLine And _R Then g.DrawLine(pen, rect.Right, rect.Top, rect.Right, rect.Bottom)
        End Using
    End Sub

    Public Shared Sub RenderCaptionValue(ByVal g As Graphics, _
                                         ByVal field As UTable.CField, _
                                         ByVal sortState As UTable.CSortState, _
                                         ByVal rect As Rectangle, _
                                         ByVal formattedValue As String, _
                                         ByVal color As Color, _
                                         ByVal font As Font, _
                                         ByVal stringFormat As StringFormat)
        Using brush As New SolidBrush(color)
            If rect.Width > 15 And field.Table.SortState.Field Is field Then
                g.DrawString(ValueSanitize(formattedValue), font, brush, New RectangleF(rect.X, rect.Y + 2, rect.Width - 15, rect.Height - 2), stringFormat)
                Dim points As New List(Of PointF)
                Select Case field.Table.SortState.Order
                    Case UTable.CSortState.EOrder.ASCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 7))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 14))
                    Case UTable.CSortState.EOrder.DESCEND
                        points.Add(New PointF(rect.Right - 14, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 2, rect.Bottom - 12))
                        points.Add(New PointF(rect.Right - 8, rect.Bottom - 6))
                End Select
                g.FillPolygon(brush, points.ToArray)
            Else
                g.DrawString(field.Value, font, brush, New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), stringFormat)
            End If
        End Using
        If field.Decorator IsNot Nothing Then
            field.Decorator.RenderOverValue(g, field, rect)
        End If
    End Sub

    Public Shared Sub RenderCheckBox(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean)
        Dim p As New Point(rect.Left + rect.Width / 2 - 6, rect.Top + rect.Height / 2 - 6)
        If value Then
            CheckBoxRenderer.DrawCheckBox(g, p, VisualStyles.CheckBoxState.CheckedNormal)
        Else
            CheckBoxRenderer.DrawCheckBox(g, p, VisualStyles.CheckBoxState.UncheckedNormal)
        End If
    End Sub

    Public Shared Sub RenderNarrowChild(ByVal g As Graphics, ByVal rect As Rectangle, ByVal value As Boolean)
        Dim w As Integer = NARROWCHILD_SIZE
        Dim r As New Rectangle(rect.Left + rect.Width / 2 - w / 2, rect.Top + rect.Height / 2 - w / 2, w, w)
        g.DrawRectangle(Pens.Gray, r)
        g.DrawLine(Pens.Gray, New Point(r.Left, r.Top + r.Height / 2), New Point(r.Right, r.Top + r.Height / 2))
        If Not value Then
            g.DrawLine(Pens.Gray, New Point(r.Left + r.Width / 2, r.Top), New Point(r.Left + r.Width / 2, r.Top + r.Height))
        End If
    End Sub

    Public Shared Function ValueSanitize(ByVal v As Object) As String
        If v Is Nothing Then
            Return ""
        Else
            Return System.Text.RegularExpressions.Regex.Replace(v.ToString, "]", "-")
        End If
    End Function

    Public Shared Function GradientColor(ByVal c As Color, ByVal d As Integer) As Color
        Dim r As Integer = c.R + d
        Dim g As Integer = c.G + d
        Dim b As Integer = c.B + d
        If r > 255 Then r = 255
        If r < 0 Then r = 0
        If g > 255 Then g = 255
        If g < 0 Then g = 0
        If b > 255 Then b = 255
        If b < 0 Then b = 0
        Return Color.FromArgb(r, g, b)
    End Function

    Public Shared Sub FlatBorder(ByVal descs As Dictionary(Of Object, UTable.CRecordProvider.CFieldDesc), ByVal key1 As Object, ByVal key2 As Object)
        Dim l As UTable.CGrid.CRegion = descs(key1).Layout.Union(descs(key2).Layout)
        For Each d As UTable.CRecordProvider.CFieldDesc In descs.Values
            If TypeOf d.Provider Is CFieldProvider AndAlso d.Layout IsNot Nothing Then
                If d.Layout.Row < (l.Row + l.Rows) And (d.Layout.Row + d.Layout.Rows) > l.Row And _
                   d.Layout.Col < (l.Col + l.Cols) And (d.Layout.Col + d.Layout.Cols) > l.Col Then
                    With CType(d.Provider, CFieldProvider)
                        .BorderLine = 0
                        If d.Layout.Row = l.Row Then
                            .BorderLine = .BorderLine Or _T
                        End If
                        If d.Layout.Row + d.Layout.Rows = l.Row + l.Rows Then
                            .BorderLine = .BorderLine Or _B
                        End If
                        If d.Layout.Col = l.Col Then
                            .BorderLine = .BorderLine Or _L
                        End If
                        If d.Layout.Col + d.Layout.Cols = l.Col + l.Cols Then
                            .BorderLine = .BorderLine Or _R
                        End If
                    End With
                End If
            End If
        Next
    End Sub

    Public Shared ADJUSTSIZE_TOGGLE As New Size(18, 18)
    Public Shared CLICKWIDTH_TOGGLE As Integer = 9
    Public Shared NARROWCHILD_SIZE As Integer = 10

    Public Shared Function SetToggleHandler(ByVal field As UTable.CField) As UTable.CField
        AddHandler field.MouseDown, AddressOf CFieldProvider._toggle_mousedown
        AddHandler field.KeyDown, AddressOf CFieldProvider._toggle_keyDown
        Return field
    End Function

    Private Shared Sub _toggle_mousedown(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs)
        If e.Button = MouseButtons.Left AndAlso field.Table.FocusField Is field Then
            Dim f As UTable.CRenderCache.CField = field.Table.RenderCache.FindField(field)
            If f IsNot Nothing AndAlso _
               (CLICKWIDTH_TOGGLE = 0 OrElse _
               (System.Math.Abs((f.Rect.Width / 2) - location.X) < CLICKWIDTH_TOGGLE And _
                System.Math.Abs((f.Rect.Height / 2) - location.Y) < CLICKWIDTH_TOGGLE)) Then
                CFieldProvider._toggle_changeValue(field)
            End If
        End If
    End Sub

    Private Shared Sub _toggle_keyDown(ByVal field As UTable.CField, ByVal e As System.Windows.Forms.KeyEventArgs)
        Select Case e.KeyData
            Case Keys.Space
                CFieldProvider._toggle_changeValue(field)
        End Select
    End Sub

    Private Shared Sub _toggle_changeValue(ByVal field As UTable.CField)
        If field.Editable = UTable.EAllow.ALLOW Then
            field.SetValueIfValidated(Not CType(field.Value, Boolean))
        End If
    End Sub

End Class

Public Class CConstFieldProvider
    Inherits CFieldProvider

    Public Value As Object

    Public Sub New(ByVal value As String)
        Me.New(Nothing, value)
    End Sub

    Public Sub New(ByVal caption As String, ByVal value As Object)
        MyBase.New(caption)
        Me.Value = value
    End Sub

    Public Overrides Function CreateField() As UTable.CField
        Return New CConstField()
    End Function

    Public Class CConstField
        Inherits UTable.CField
        Public Overrides Property Value() As Object
            Get
                Return CType(Me.Desc.Provider, CConstFieldProvider).Value
            End Get
            Set(ByVal value As Object)
            End Set
        End Property
        Public Overrides Sub Clear()
        End Sub
    End Class

End Class

Public Class CTextFieldProvider
    Inherits CFieldProvider

    Private _imeMode As ImeMode

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        Me.New(caption, Windows.Forms.ImeMode.NoControl)
    End Sub

    Public Sub New(ByVal caption As String, ByVal imeMode As ImeMode)
        MyBase.New(caption)
        Me._imeMode = imeMode
    End Sub

    Public Overrides Function CreateEditor() As IEditor
        Return New CTextBoxEditor
    End Function

    Public Overrides Function ImeMode() As System.Windows.Forms.ImeMode
        Return Me._imeMode
    End Function

End Class

Public Class CNarrowChildFieldProvider
    Inherits CFieldProvider

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        MyBase.New(caption)
    End Sub

    Public Overrides Function CreateField() As UTable.CField
        Return New CNarrowChildField()
    End Function

    Public Overrides Sub FieldInitialize(ByVal field As UTable.CField)
        CFieldProvider.SetToggleHandler(field)
    End Sub

    Public Overrides Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        RenderBackgroud(g, field, rect, BackColor(field, s, alter))
        RenderBorder(g, rect, Me.BorderLine, field.Table.Setting.BorderColor, field.Table.Setting.BorderStyle)
        RenderNarrowChild(g, rect, field.Value)
    End Sub

    Public Overrides Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
        Return CFieldProvider.ADJUSTSIZE_TOGGLE
    End Function

    Public Class CNarrowChildField
        Inherits UTable.CField

        Public Overrides Property Value() As Object
            Get
                If Me.Record.Child IsNot Nothing AndAlso Me.Record.Child.Records.Count > 0 Then
                    Return Me.Record.Child.Visible
                Else
                    Return True
                End If
            End Get
            Set(ByVal value As Object)
                MyBase.Value = value
            End Set
        End Property

        Public Overrides Sub ValueCommit(ByVal v As Object)
            If Me.Record.Child IsNot Nothing Then
                Me.Record.Child.Visible = v
            End If
        End Sub

        Public Overrides Sub Clear()
        End Sub

    End Class

End Class

Public Class CCheckFieldProvider
    Inherits CFieldProvider

    Public Sub New()
        Me.New(Nothing)
    End Sub

    Public Sub New(ByVal caption As String)
        MyBase.New(caption)
    End Sub

    Public Overrides Sub FieldInitialize(ByVal field As UTable.CField)
        field.Value = False
        CFieldProvider.SetToggleHandler(field)
    End Sub

    Public Overrides Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        RenderBackgroud(g, field, rect, BackColor(field, s, alter))
        RenderBorder(g, rect, Me.BorderLine, field.Table.Setting.BorderColor, field.Table.Setting.BorderStyle)
        RenderCheckBox(g, rect, field.Value)
    End Sub

    Public Overrides Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
        Return CFieldProvider.ADJUSTSIZE_TOGGLE
    End Function

End Class

Public Class CButtonFieldProvider
    Inherits CFieldProvider

    Public DefaultValue As Object

    Public Sub New()
        Me.New(Nothing, Nothing)
    End Sub

    Public Sub New(ByVal defaultValue As Object)
        Me.New(Nothing, defaultValue)
    End Sub

    Public Sub New(ByVal caption As String, ByVal defaultValue As Object)
        MyBase.New(caption)
        Me.DefaultValue = defaultValue
    End Sub

    Public Overrides Function CreateField() As UTable.CField
        Return New CButtonField()
    End Function

    Public Overrides Sub FieldInitialize(ByVal field As UTable.CField)
        field.Value = Me.DefaultValue
    End Sub

    Public Overrides Function Setting() As UTable.CSetting
        Dim s As New UTable.CSetting
        s.HorizontalAlignment = UTable.EHAlign.MIDDLE
        Return s
    End Function

    Public Overrides Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        RenderBackgroud(g, field, rect, BackColor(field, s, alter))
        RenderBorder(g, rect, Me.BorderLine, field.Table.Setting.BorderColor, field.Table.Setting.BorderStyle)
        If s.Editable = UTable.EAllow.ALLOW Then
            Dim d As Integer = 0
            If CType(field, CButtonField).Down Then
                Me._renderButton_aux(g, field, rect, s, alter, True)
                d = 2
            Else
                Me._renderButton_aux(g, field, rect, s, alter, False)
            End If
            Dim t As String = Me.formatValue(field.Value)
            If Not String.IsNullOrEmpty(t) Then
                Using brush As New SolidBrush(s.ForeColor)
                    g.DrawString(ValueSanitize(t), s.Font, brush, New RectangleF(rect.X, rect.Y + 2 + d, rect.Width, rect.Height - 2), s.GetStringFormat)
                End Using
            End If
        Else
            Me._renderButton_aux(g, field, rect, s, alter, False)
            Dim t As String = Me.formatValue(field.Value)
            If Not String.IsNullOrEmpty(t) Then
                ControlPaint.DrawStringDisabled(g, ValueSanitize(t), s.Font, Color.White, New RectangleF(rect.X, rect.Y + 2, rect.Width, rect.Height - 2), s.GetStringFormat)
            End If
        End If
    End Sub

    Private Sub _renderButton_aux(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal s As UTable.CDynamicSetting, ByVal alter As Boolean, ByVal push As Boolean)
        Using b As New Drawing2D.LinearGradientBrush(New RectangleF(rect.X, rect.Y, rect.Width, rect.Height), s.ButtonBackColor, GradientColor(s.ButtonBackColor, 32), Drawing2D.LinearGradientMode.Vertical)
            g.FillRectangle(b, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 3, rect.Height - 3))
        End Using
        Using p As New Pen(GradientColor(s.ButtonBackColor, -128))
            g.DrawRectangle(p, New Rectangle(rect.X + 2, rect.Y + 2, rect.Width - 4, rect.Height - 4))
        End Using
        If push Then
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        Else
            Using p As New Pen(GradientColor(s.ButtonBackColor, -64))
                g.DrawLine(p, rect.Right - 3, rect.Y + 3, rect.Right - 3, rect.Bottom - 3)
                g.DrawLine(p, rect.X + 3, rect.Bottom - 3, rect.Right - 3, rect.Bottom - 3)
            End Using
            Using p As New Pen(GradientColor(s.ButtonBackColor, 64))
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.Right - 3, rect.Y + 3)
                g.DrawLine(p, rect.X + 3, rect.Y + 3, rect.X + 3, rect.Bottom - 3)
            End Using
        End If
    End Sub

    Public Overrides Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
        With MyBase.GetAdjustSize(g, field)
            Return New Size(.Width + 6, .Height)
        End With
    End Function

    Public Class CButtonField
        Inherits UTable.CField
        Public Down As Boolean = False
        Private Sub CButtonField_Leave(ByVal field As UTable.CField) Handles Me.Leave
            Me.Down = False
        End Sub
        Private Sub CButtonField_KeyDown(ByVal field As UTable.CField, ByVal e As System.Windows.Forms.KeyEventArgs) Handles Me.KeyDown
            If e.KeyCode = Keys.Space AndAlso Me.DynamicSetting.Editable = UTable.EAllow.ALLOW Then
                Me.Down = True
                Me.Table.Render()
            End If
        End Sub
        Private Sub CButtonField_KeyUp(ByVal field As UTable.CField, ByVal e As System.Windows.Forms.KeyEventArgs) Handles Me.KeyUp
            If e.KeyCode = Keys.Space AndAlso Me.Down Then
                Me.Down = False
                Me.Table.Render()
                Me.raiseButtonClick()
                Me.Table.raiseFieldButtonClick(Me)
            End If
        End Sub
        Private Sub CButtonField_MouseDown(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseDown
            If e.Button = MouseButtons.Left AndAlso Me.DynamicSetting.Editable = UTable.EAllow.ALLOW Then
                Me.Down = True
                Me.Table.Render()
            End If
        End Sub
        Private Sub CButtonField_MouseUp(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs) Handles Me.MouseUp
            If Me.Down Then
                Me.Down = False
                Me.Table.Render()
                If location.X >= 0 And location.Y >= 0 Then
                    Me.raiseButtonClick()
                    Me.Table.raiseFieldButtonClick(Me)
                End If
            End If
        End Sub
        Public Overrides Sub Clear()
        End Sub
    End Class

End Class

Public Class CCaptionFieldProvider
    Inherits CFieldProvider

    Private _Setting As UTable.CSetting = Nothing
    Private draggingRecord As UTable.CRecord = Nothing
    Private WithEvents table As UTable

    Public UserSortable As UTable.EAllow = UTable.EAllow.DEFAULT
    Public Draggable As Boolean

    Public Sub New()
        Me.New(Nothing, False)
    End Sub

    Public Sub New(ByVal caption As String)
        Me.New(caption, False)
    End Sub

    Public Sub New(ByVal caption As String, ByVal draggable As Boolean)
        MyBase.New(caption)
        Me.Draggable = draggable
    End Sub

    Friend Sub New(ByVal fieldDesc As UTable.CRecordProvider.CFieldDesc)
        Me.New(fieldDesc.Provider.Caption)
        Me.UserSortable = fieldDesc.UserSortable
        If TypeOf fieldDesc.Provider Is CFieldProvider Then
            Me.BorderLine = CType(fieldDesc.Provider, CFieldProvider).BorderLine
        End If
        If fieldDesc.HasSetting Then
            Me._Setting = fieldDesc.Setting.Clone
        End If
    End Sub

    Public Overrides Function Setting() As UTable.CSetting
        Return Me._Setting
    End Function

    Public Overrides Sub FieldInitialize(ByVal field As UTable.CField)
        Me.table = field.Table
        field.Value = Me.Caption
        AddHandler field.DoubleClick, AddressOf Me._DoubleClick
    End Sub

    Public Overrides Function CreateEditor() As IEditor
        Return Nothing
    End Function

    Public Overrides Sub Render(ByVal g As Graphics, ByVal field As UTable.CField, ByVal rect As Rectangle, ByVal alter As Boolean)
        Dim s As UTable.CDynamicSetting = field.DynamicSetting
        Dim drag As Boolean = field.Record Is Me.draggingRecord
        RenderCaptionBackgroud(g, field, rect, CaptionBackColor(field, s, drag), s.CaptionStyle)
        RenderCaptionBorder(g, rect, Me.BorderLine, field.Table.Setting.CaptionBorderColor)
        RenderCaptionValue(g, field, field.Table.SortState, rect, formatValue(field.Value), s.CaptionForeColor, s.CaptionFont, s.GetStringFormat)
    End Sub

    Private Sub _DoubleClick(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs)
        If field.TopLevelContent Is field.Table.HeaderContent Then
            Dim _sortable As UTable.EAllow = field.Table.Setting.UserSortable
            With CType(field.Desc.Provider, CCaptionFieldProvider)
                If .UserSortable <> UTable.EAllow.DEFAULT Then
                    _sortable = .UserSortable
                End If
            End With
            If _sortable = UTable.EAllow.ALLOW Then
                Using field.Table.RenderBlock
                    Dim _order As UTable.CSortState.EOrder = UTable.CSortState.EOrder.ASCEND
                    If field.Table.SortState.Field Is field And field.Table.SortState.Order = UTable.CSortState.EOrder.ASCEND Then
                        _order = UTable.CSortState.EOrder.DESCEND
                    End If
                    field.Table.Content.Sort(field.Key, _order)
                    With field.Table.SortState
                        .Field = field
                        .Order = _order
                    End With
                End Using
            End If
        End If
    End Sub

    Public Overrides Function Focusable() As Boolean
        Return False
    End Function

    Public Overrides Function GetAdjustSize(ByVal g As Graphics, ByVal field As UTable.CField) As Size
        With field.DynamicSetting
            Return g.MeasureString(Me.formatValue(field.Value), .CaptionFont, 100000, .GetStringFormat).ToSize
        End With
    End Function

    Private Sub table_FieldMouseDown(ByVal field As UTable.CField, ByVal location As Point, ByVal e As System.Windows.Forms.MouseEventArgs) Handles table.FieldMouseDown
        If field.TopLevelContent Is field.Table.Content AndAlso _
           Me.Draggable AndAlso _
           e.Button = Windows.Forms.MouseButtons.Left Then
            If field.Desc.Provider Is Me Then
                Me.draggingRecord = field.Record
                field.Table.Render()
            End If
        End If
    End Sub

    Private Sub table_MouseMove(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles table.MouseMove
        If Me.draggingRecord IsNot Nothing Then
            Dim f As UTable.CRenderCache.CField = Me.table.RenderCache.FindField(e.Location)
            If f IsNot Nothing Then
                Me.recordMove(f.Field)
            ElseIf e.Location.Y < 0 Then
                table.SetVScrollValue(table.VScrollBar.Value - table.VScrollBar.SmallChange)
            ElseIf e.Location.Y > table.Height Then
                table.SetVScrollValue(table.VScrollBar.Value + table.VScrollBar.SmallChange)
            End If
        End If
    End Sub

    Private Sub recordMove(ByVal field As UTable.CField)
        If Me.draggingRecord.Content Is field.Content AndAlso Me.draggingRecord IsNot field.Record Then
            Using Me.table.RenderBlock
                With Me.draggingRecord.Content
                    Dim i1 As Integer = .Records.IndexOf(Me.draggingRecord)
                    Dim i2 As Integer = .Records.IndexOf(field.Record)
                    If i2 >= 0 AndAlso i1 <> i2 Then
                        .MoveRecord(Me.draggingRecord, i2)
                    End If
                End With
                Me.draggingRecord.TopLevelContent.LayoutCacheInvalid = True
            End Using
        End If
    End Sub

    Private Sub UTable_MouseUp(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles table.MouseUp
        If e.Button = Windows.Forms.MouseButtons.Left Then
            If Me.draggingRecord IsNot Nothing Then
                Me.draggingRecord = Nothing
                CType(sender, UTable).Render()
            End If
        End If
    End Sub

    Public Overrides Function RenderOrder() As Integer
        Return 1
    End Function

End Class