' table/KeyboardOperation.vb
'
' Copyright (c) 2008, SystemBase Co.,Ltd.
' All rights reserved.
'
' Redistribution and use in source and binary forms, with or without 
' modification, are permitted provided that the following conditions are met:
'
'    1. Redistributions of source code must retain the above copyright 
'       notice, this list of conditions and the following disclaimer.
'    2. Redistributions in binary form must reproduce the above copyright 
'       notice, this list of conditions and the following disclaimer in the
'       documentation and/or other materials provided with the distribution.
'
' THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
' IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
' ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
' LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
' CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
' SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
' INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
' CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
' ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
' POSSIBILITY OF SUCH DAMAGE.

Public Interface IKeyboardOperation
    Function ProcessDialogKey(ByVal key As Keys, ByVal table As UTable) As Boolean
    Function EditorProcessDialogKey(ByVal key As Keys, ByVal table As UTable) As Boolean
    Sub LeaveEdit(ByVal direction As String, ByVal table As UTable)
End Interface

Public Class CDefaultKeyboardOperation
    Implements IKeyboardOperation

    Public Enum EDirection
        COL
        ROW
    End Enum

    Public Direction As EDirection = EDirection.COL
    Public NextAsTab As Boolean = False
    Public LeaveOnLastField As Boolean = False

    Public Overridable Function ProcessDialogKey(ByVal key As System.Windows.Forms.Keys, ByVal table As UTable) As Boolean Implements IKeyboardOperation.ProcessDialogKey
        Select Case key
            Case Keys.Up, Keys.Down, Keys.Left, Keys.Right
                If table.FocusField Is Nothing Then
                    table.EntryFocus()
                Else
                    Select Case key
                        Case Keys.Up
                            Me.prevRow(table)
                        Case Keys.Down
                            Me.nextRow(table)
                        Case Keys.Left
                            Me.prevCol(table)
                        Case Keys.Right
                            Me.nextCol(table)
                    End Select
                End If
                Return True
            Case Keys.Home, Keys.End
                If table.FocusField IsNot Nothing Then
                    Select Case key
                        Case Keys.Home
                            Me.startCol(table)
                        Case Keys.End
                            Me.endCol(table)
                    End Select
                End If
                Return True
            Case Keys.PageDown
                Me.pageDown(table, table.VScrollBar)
                Return True
            Case Keys.PageUp
                Me.pageUp(table, table.VScrollBar)
                Return True
            Case Keys.Enter
                If table.FocusField Is Nothing Then
                    table.EntryFocus()
                    If table.FocusField Is Nothing Then
                        table.TopLevelControl.SelectNextControl(table, True, True, True, True)
                    End If
                Else
                    Dim handled As Boolean = False
                    table.RaiseFieldSelected(table.FocusField, handled)
                    If Not handled AndAlso table.StartEditing() Is Nothing Then
                        Me.next(table)
                    End If
                End If
                Return True
            Case Keys.Enter Or Keys.Shift
                If table.FocusField IsNot Nothing Then
                    Me.prev(table)
                    If table.FocusField Is Nothing Then
                        table.TopLevelControl.SelectNextControl(table, False, True, True, True)
                    End If
                    Return True
                End If
            Case Keys.Tab
                If Me.NextAsTab Then
                    If table.FocusField Is Nothing Then
                        table.EntryFocus()
                        If table.FocusField Is Nothing Then
                            table.TopLevelControl.SelectNextControl(table, True, True, True, True)
                        End If
                    Else
                        Me.next(table)
                    End If
                    Return True
                End If
            Case Keys.Tab Or Keys.Shift
                If Me.NextAsTab Then
                    If table.FocusField IsNot Nothing Then
                        Me.prev(table)
                        If table.FocusField Is Nothing Then
                            table.TopLevelControl.SelectNextControl(table, False, True, True, True)
                        End If
                        Return True
                    End If
                End If
            Case Keys.Delete
                If table.FocusField IsNot Nothing Then
                    If table.RaiseEditStarting(table.FocusField) = UTable.CSetting.EEditable.ALLOW Then
                        table.FocusField.Clear()
                    End If                    
                End If
        End Select
        Return False
    End Function

    Public Function EditorProcessDialogKey(ByVal key As System.Windows.Forms.Keys, ByVal table As UTable) As Boolean Implements IKeyboardOperation.EditorProcessDialogKey
        Select Case key
            Case Keys.Enter
                table.LeaveEdit("NEXT")
                Return True
            Case (Keys.Enter Or Keys.Shift)
                table.LeaveEdit("PREV")
                Return True
            Case Keys.Tab
                If NextAsTab Then
                    table.LeaveEdit("NEXT")                    
                    Return True
                End If
            Case (Keys.Tab Or Keys.Shift)
                If NextAsTab Then
                    table.LeaveEdit("PREV")                    
                    Return True
                End If
        End Select
        Return False
    End Function

    Public Overridable Sub LeaveEdit(ByVal direction As String, ByVal table As UTable) Implements IKeyboardOperation.LeaveEdit
        Select Case direction
            Case "UP"
                Me.prevRow(table)
            Case "DOWN"
                Me.nextRow(table)
            Case "LEFT"
                Me.prevCol(table)
            Case "RIGHT"
                Me.nextCol(table)
            Case "NEXT"
                Me.next(table)
            Case "PREV"
                Me.prev(table)
        End Select
    End Sub

    Protected Overridable Sub nextRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.EndRow
        Dim field As UTable.CField = UTable.GetNextRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevRow(ByVal table As UTable)
        table.FocusGrid.Row = table.FocusField.Desc.Layout.Row
        Dim field As UTable.CField = UTable.GetPrevRowField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub nextCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.EndCol
        Dim field As UTable.CField = UTable.GetNextColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub prevCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = UTable.GetPrevColField(table.FocusGrid, table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub startCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = UTable.GetNextColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, -1), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub endCol(ByVal table As UTable)
        table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
        Dim field As UTable.CField = UTable.GetPrevColField(New UTable.CGrid.CPoint(table.FocusGrid.Row, table.Cols.Count), table.FocusRecord)
        If field IsNot Nothing Then
            table.MoveFocusField(field)
        End If
    End Sub

    Protected Overridable Sub pageDown(ByVal table As UTable, ByVal scrollBar As ScrollBar)
        table.SetScrollValue(scrollBar, scrollBar.Value + scrollBar.LargeChange)
    End Sub

    Protected Overridable Sub pageUp(ByVal table As UTable, ByVal scrollBar As ScrollBar)
        table.SetScrollValue(scrollBar, scrollBar.Value - scrollBar.LargeChange)
    End Sub

    Protected Overridable Sub [next](ByVal table As UTable)
        Select Case Me.Direction
            Case EDirection.COL
                table.FocusGrid.Col = table.FocusField.Desc.Layout.Col + table.FocusField.Desc.Layout.Cols - 1
                Dim p As New UTable.CGrid.CPoint(table.FocusGrid.Row, table.FocusGrid.Col)
                Dim r As UTable.CRecord = table.FocusRecord
                Do
                    p.Col += 1
                    If p.Col >= r.Table.Cols.Count Then
                        p.Col = 0
                        p.Row += 1
                    End If
                    If p.Row >= r.Rows.Count Then
                        p.Col = 0
                        p.Row = 0
                        r = r.LayoutCache.NextRecord
                    End If
                    If r Is Nothing Then
                        Exit Do
                    End If
                    Dim field As UTable.CField = UTable.FindEditableField(p, r)
                    If field IsNot Nothing Then
                        table.FocusField = field
                        Exit Sub
                    End If
                Loop
            Case EDirection.ROW
                table.FocusGrid.Row = table.FocusField.Desc.Layout.Row + table.FocusField.Desc.Layout.Rows - 1
                Dim p As New UTable.CGrid.CPoint(table.FocusGrid.Row, table.FocusGrid.Col)
                Dim r As UTable.CRecord = table.FocusRecord
                Do
                    p.Row += 1
                    If p.Row >= r.Rows.Count Then
                        p.Col += 1
                        p.Row = 0
                    End If
                    If p.Col >= r.Table.Cols.Count Then
                        p.Col = 0
                        p.Row = 0
                        r = r.LayoutCache.NextRecord
                    End If
                    If r Is Nothing Then
                        Exit Do
                    End If
                    Dim field As UTable.CField = UTable.FindEditableField(p, r)
                    If field IsNot Nothing Then
                        table.FocusField = field
                        Exit Sub
                    End If
                Loop
        End Select
        If table.TopLevelControl IsNot Nothing AndAlso LeaveOnLastField Then
            table.TopLevelControl.SelectNextControl(table, True, True, True, True)
        End If
    End Sub

    Protected Overridable Sub [prev](ByVal table As UTable)
        Select Case Me.Direction
            Case EDirection.COL
                table.FocusGrid.Col = table.FocusField.Desc.Layout.Col
                Dim p As New UTable.CGrid.CPoint(table.FocusGrid.Row, table.FocusGrid.Col)
                Dim r As UTable.CRecord = table.FocusRecord
                Do
                    p.Col -= 1
                    If p.Col < 0 Then
                        p.Col = table.Cols.Count - 1
                        p.Row -= 1
                    End If
                    If p.Row < 0 Then
                        r = r.LayoutCache.PrevRecord
                        If r Is Nothing Then
                            Exit Do
                        End If
                        p.Col = table.Cols.Count - 1
                        p.Row = r.Rows.Count - 1
                    End If
                    Dim field As UTable.CField = UTable.FindEditableField(p, r)
                    If field IsNot Nothing Then
                        table.FocusField = field
                        Exit Sub
                    End If
                Loop
            Case EDirection.ROW
                table.FocusGrid.Row = table.FocusField.Desc.Layout.Row
                Dim p As New UTable.CGrid.CPoint(table.FocusGrid.Row, table.FocusGrid.Col)
                Dim r As UTable.CRecord = table.FocusRecord
                Do
                    p.Row -= 1
                    If p.Row < 0 Then
                        p.Col -= 1
                        p.Row = r.Rows.Count - 1
                    End If
                    If p.Col < 0 Then
                        r = r.LayoutCache.PrevRecord
                        If r Is Nothing Then
                            Exit Do
                        End If
                        p.Col = table.Cols.Count - 1
                        p.Row = r.Rows.Count - 1
                    End If
                    Dim field As UTable.CField = UTable.FindEditableField(p, r)
                    If field IsNot Nothing Then
                        table.FocusField = field
                        Exit Sub
                    End If
                Loop
        End Select
        If LeaveOnLastField Then
            table.TopLevelControl.SelectNextControl(table, False, True, True, True)
        End If
    End Sub

End Class
