﻿// Program.cs
//
// Author:
// tsntsumi <tsntsumi at tsntsumi.com>
//
// Copyright (c) 2015 tsntsumi
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
//	You should have received a copy of the GNU Lesser General Public License
//	along with this program.  If not, see <http://www.gnu.org/licenses/>.

/// @file
/// <summary>
/// コンソールベースの簡単なチャットサーバ。
/// </summary>
/// @since 2015.8.14
using System;
using System.Text;
using System.Net;
using SocketNet;
using SampleChatClient;

namespace SampleChatServer
{
    /// <summary>
    /// SocketNet ライブラリのサンプルプログラム (コンソールベースの簡単なチャットサーバ)。
    /// </summary>
	class MainClass
	{
		/// <summary>
		/// チャットサーバプログラムのエントリポイント。
		/// </summary>
		/// <remarks>
		/// チャットサーバオブジェクトを生成し、動作の設定を行ってから開始します。
		/// 何かキーを押されたら、チャットサーバを停止して終了します。
		/// </remarks>
		public static void Main()
		{
			var chatServer = new ChatServer(IPAddress.Any, 50001);
			chatServer.Connected += OnConnected;
			chatServer.Disconnected += OnDisconnected;
			chatServer.ChatMessageReceived += OnChatMessageReceived;
			chatServer.Start();
			Console.WriteLine("Press any key to exit.");
			Console.ReadKey();
			chatServer.Stop();
		}

        /// <summary>
        /// チャットクライアントから接続があったときのイベントを処理します。
        /// </summary>
        /// <param name="sender">イベントのセンダ。</param>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		private static void OnConnected(object sender, TcpConnectionEventArgs e)
		{
			Console.WriteLine("{0} から接続がありました。", e.Connection.Client.RemoteEndPoint.ToString());
		}

		/// <summary>
		/// チャットクライアントが接続を切ったときのイベントを処理します。
		/// </summary>
		/// <param name="sender">イベントのセンダ。</param>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		private static void OnDisconnected(object sender, TcpConnectionEventArgs e)
		{
			Console.WriteLine("{0} が切断しました。", e.Connection.Client.RemoteEndPoint.ToString());
		}

		/// <summary>
		/// チャットクライアントからのメッセージを受信したときのイベントを処理します。
		/// </summary>
		/// <param name="sender">イベントのセンダ。</param>
		/// <param name="e">イベントデータを格納するオブジェクト。</param>
		private static void OnChatMessageReceived(object sender, ChatMessageReceivedEventArgs e)
		{
			Console.WriteLine("<< {0}", e.ChatMessage.Message);
			switch (e.ChatMessage.Message.ToUpper())
			{
			case "/LIST":
				{
					ChatServer chatServer = (ChatServer)sender;
					StringBuilder sb = new StringBuilder();

					foreach (var connection in chatServer.ActiveConnections)
					{
						Console.WriteLine(">> {0}", connection.Client.RemoteEndPoint.ToString());
						sb.AppendLine(connection.Client.RemoteEndPoint.ToString());
					}
					var message = new ChatMessage(sb.ToString(), e.ChatMessage.Connection);
					e.ChatMessage.Connection.Writer.Write(message.ToByteArray());
					e.ChatMessage.Connection.Writer.Flush();
				}
				break;
			default:
				{
					ChatServer chatServer = (ChatServer)sender;

					foreach (var connection in chatServer.ActiveConnections)
					{
						connection.Writer.Write(e.ChatMessage.ToByteArray());
						connection.Writer.Flush();
					}
				}
				break;
			}
		}
	}
}
