/*
 * PMX material information handler
 *
 * License : The MIT License
 * Copyright(c) 2015 MikuToga Partners
 */

package jp.sfjp.mikutoga.pmx.parser;


import jp.sfjp.mikutoga.bin.parser.LoopHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;

/**
 * PMDモデルの各種材質情報の通知用ハンドラ。
 * 色空間はsRGB?
 */
public interface PmxMaterialHandler extends LoopHandler {

    /** 材質抽出ループ。 */
    ParseStage TEXTURE_LIST = new ParseStage();
    /** 材質抽出ループ。 */
    ParseStage MATERIAL_LIST = new ParseStage();

    /**
     * テクスチャーの通知を受け取る.
     * <p>{@link #TEXTURE_LIST}ループの構成要素。
     * @param textureFile テクスチャーファイル名
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxTexture(String textureFile)
            throws MmdFormatException;

    /**
     * 材質の名前の通知を受け取る.
     * <p>{@link #TEXTURE_LIST}ループの構成要素。
     * @param name 材質名
     * @param nameE name of the material.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxMaterialNames(String name, String nameE)
            throws MmdFormatException;

    /**
     * 材質の拡散光成分の通知を受け取る.
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMaterialDiffuse(float red, float green, float blue,
                               float alpha )
            throws MmdFormatException;

    /**
     * 材質の反射光成分の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param shininess 光沢強度(1～15ぐらい)
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMaterialSpecular(float red, float green, float blue,
                                float shininess)
            throws MmdFormatException;

    /**
     * 材質の環境色成分の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMaterialAmbient(float red, float green, float blue)
            throws MmdFormatException;


    /**
     * 材質のエッジ成分の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param flags 描画フラグ(8bit).
     * - 各bit 0:OFF 1:ON
     * 0x01:両面描画, 0x02:地面影, 0x04:セルフシャドウマップへの描画,
     * 0x08:セルフシャドウの描画, 0x10:エッジ描画
     * @param red 0.0～1.0の範囲の赤成分
     * @param green 0.0～1.0の範囲の緑成分
     * @param blue 0.0～1.0の範囲の青成分
     * @param alpha 0.0(透明)～1.0(不透明)のアルファ値。
     * @param thick 0.0～2.0くらいの厚さ
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    public void pmxMaterialEdges(byte flags, float red, float green, float blue, float alpha, float thick)
            throws MmdFormatException;

    /**
     * 材質のテクスチャー情報の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param textureId テクスチャID. 無ければ-1.
     * @param sphereId スフィアマップID. 無ければ-1.
     * @param mode スフィアモード. 0:無効 1:乗算(sph) 2:加算(spa)
     *     3:サブテクスチャ(追加UV1のx,yをUV参照して通常テクスチャ描画を行う)
     * @throws MmdFormatException 不正フォーマットによる
     *     パース処理の中断をパーサに指示
     */
    void pmxMaterialTexture(int textureId, int sphereId,byte mode)
            throws MmdFormatException;

    /**
     * 材質のトゥーン情報の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param shared 0: トゥーンはテクスチャーファイル. 1: 共有ファイル.
     * @param toonId トゥーンファイル番号。
     * shared:1 では, 0ならtoon01.bmp, 9ならtoon10.bmp. 0xffならtoon0.bmp.
     * shared:0 では, texture ID.
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     */
    void pmxMaterialToon(byte shared, int toonId)
            throws MmdFormatException;
    /**
     * 材質情報の通知を受け取る。
     * <p>{@link #MATERIAL_LIST}ループの構成要素。
     * @param script メモとかエフェクター用のスクリプトとか.
     * @param vertexNum 面頂点数. 3の倍数のはず.
     * 3で割ると、材質に属する面の数を表す.
     * <p>通算した面数を面情報通知順と突き合わせることにより、
     * 材質に属する面の集合を得ることが可能。
     * @throws MmdFormatException 不正フォーマットによる
     * パース処理の中断をパーサに指示
     * @see PmxFaceHandler#pmxFaceTriangle(int, int, int)
     */
    void pmxMaterialInfo(String script,int vertexNum)
            throws MmdFormatException;

}
