
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






/**
 * Var16CharVector implements a vector of variable width values.  Elements in the vector
 * are accessed by position from the logical start of the vector.  A fixed width offsetVector
 * is used to convert an element's position to it's offset from the start of the (0-based)
 * DrillBuf.  Size is inferred by adjacent elements.
 *   The width of each element is 4 byte(s)
 *   The equivalent Java primitive is 'int'
 *
 * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
 */
@SuppressWarnings("unused")
public final class Var16CharVector extends BaseDataValueVector implements VariableWidthVector{
  static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(Var16CharVector.class);

  private static final int INITIAL_BYTE_COUNT = 32768;
  private static final int MIN_BYTE_COUNT = 4096;
  
  private final UInt4Vector offsetVector;
  private final Accessor accessor;
  private final Mutator mutator;
  
  private final UInt4Vector.Accessor oAccessor;
  

  private int allocationTotalByteCount = INITIAL_BYTE_COUNT;
  private int allocationMonitor = 0;

  public Var16CharVector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
    this.offsetVector = new UInt4Vector(null, allocator);
    this.oAccessor = offsetVector.getAccessor();
    this.accessor = new Accessor();
    this.mutator = new Mutator();
  }

  public int getBufferSize(){
    if(valueCount == 0) return 0;
    return offsetVector.getBufferSize() + data.writerIndex();
  }
  
  int getSizeFromCount(int valueCount) {
    return valueCount * 4;
  }
  
  public int getValueCapacity(){
    return offsetVector.getValueCapacity() - 1;
  }
  
  public int getByteCapacity(){
    return data.capacity(); 
  }

  public int getCurrentSizeInBytes() {
    return offsetVector.getAccessor().get(currentValueCount);
  }
  
  /**
   * Return the number of bytes contained in the current var len byte vector.
   * @return
   */
  public int getVarByteLength(){
    if(valueCount == 0) return 0;
    return offsetVector.getAccessor().get(valueCount); 
  }
  
  @Override
  public SerializedField getMetadata() {
    return getMetadataBuilder() //
             .setValueCount(valueCount) //
             .setVarByteLength(getVarByteLength()) //
             .setBufferLength(getBufferSize()) //
             .build();
  }

  public int load(int dataBytes, int valueCount, DrillBuf buf){
    this.valueCount = valueCount;
    if(valueCount == 0){
      allocateNew(0,0);
      return 0;
    }
    int loaded = offsetVector.load(valueCount+1, buf);
    data = buf.slice(loaded, dataBytes - loaded);
    data.retain();
    return  dataBytes;
  }
  
  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    assert this.field.matches(metadata) : String.format("The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int loaded = load(metadata.getBufferLength(), metadata.getValueCount(), buffer);
    assert metadata.getBufferLength() == loaded : String.format("Expected to load %d bytes but actually loaded %d bytes", metadata.getBufferLength(), loaded);
  }
  
  @Override
  public void clear() {
    super.clear();
    offsetVector.clear();
  }

  
  @Override
  public DrillBuf[] getBuffers(boolean clear) {
    DrillBuf[] buffers = ObjectArrays.concat(offsetVector.getBuffers(clear), super.getBuffers(clear), DrillBuf.class);
    if (clear) {
      clear();
    }
    return buffers;
  }
  
  public long getOffsetAddr(){
    return offsetVector.getDataAddr();
  }
  
  public TransferPair getTransferPair(){
    return new TransferImpl(getField());
  }
  public TransferPair getTransferPair(FieldReference ref){
    return new TransferImpl(getField().clone(ref));
  }

  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((Var16CharVector) to);
  }
  
  public void transferTo(Var16CharVector target){
    this.offsetVector.transferTo(target.offsetVector);
    target.data = data;
    target.data.retain();
    target.valueCount = valueCount;
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, Var16CharVector target) {
    int startPoint = this.offsetVector.getAccessor().get(startIndex);
    int sliceLength = this.offsetVector.getAccessor().get(startIndex + length) - startPoint;
    target.offsetVector.clear();
    target.offsetVector.allocateNew(length + 1);
    for (int i = 0; i < length + 1; i++) {
      target.offsetVector.getMutator().set(i, this.offsetVector.getAccessor().get(startIndex + i) - startPoint);
    }
    target.data = this.data.slice(startPoint, sliceLength);
    target.data.retain();
    target.getMutator().setValueCount(length);
}
  
  protected void copyFrom(int fromIndex, int thisIndex, Var16CharVector from){
    int start = from.offsetVector.getAccessor().get(fromIndex);
    int end =   from.offsetVector.getAccessor().get(fromIndex+1);
    int len = end - start;
    
    int outputStart = offsetVector.data.getInt(thisIndex * 4);
    from.data.getBytes(start, data, outputStart, len);
    offsetVector.data.setInt( (thisIndex+1) * 4, outputStart + len);
  }
  
  public boolean copyFromSafe(int fromIndex, int thisIndex, Var16CharVector from){

    int start = from.offsetVector.getAccessor().get(fromIndex);
    int end =   from.offsetVector.getAccessor().get(fromIndex+1);
    int len = end - start;
    
    int outputStart = offsetVector.data.getInt(thisIndex * 4);
    
    if(data.capacity() < outputStart + len) {
        decrementAllocationMonitor();
        return false;
    }

    if (!offsetVector.getMutator().setSafe(thisIndex + 1, outputStart + len)) {
       decrementAllocationMonitor();
       return false;
    }

    from.data.getBytes(start, data, outputStart, len);
    offsetVector.data.setInt( (thisIndex+1) * 4, outputStart + len);

    return true;
  }

  
  private class TransferImpl implements TransferPair{
    Var16CharVector to;
    
    public TransferImpl(MaterializedField field){
      this.to = new Var16CharVector(field, allocator);
    }

    public TransferImpl(Var16CharVector to){
      this.to = to;
    }

    public Var16CharVector getTo(){
      return to;
    }
    
    public void transfer(){
      transferTo(to);
    }

    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }
    
    @Override
    public boolean copyValueSafe(int fromIndex, int toIndex) {
      return to.copyFromSafe(fromIndex, toIndex, Var16CharVector.this);
    }
  }

  public void allocateNew() {
    if(!allocateNewSafe()){
      throw new OutOfMemoryRuntimeException("Failure while allocating buffer.");
    }
  }
  
  @Override
  public boolean allocateNewSafe() {
    clear();
    if (allocationMonitor > 10) {
      allocationTotalByteCount = Math.max(MIN_BYTE_COUNT, (int) (allocationTotalByteCount / 2));
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      allocationTotalByteCount = (int) (allocationTotalByteCount * 2);
      allocationMonitor = 0;
    }

    data = allocator.buffer(allocationTotalByteCount);
    if(data == null){
      return false;
    }
    
    data.readerIndex(0);
    if(!offsetVector.allocateNewSafe()){
      return false;
    }
    offsetVector.zeroVector();
    return true;
  }
  
  public void allocateNew(int totalBytes, int valueCount) {
    clear();
    assert totalBytes >= 0;
    data = allocator.buffer(totalBytes);
    data.readerIndex(0);
    allocationTotalByteCount = totalBytes;
    offsetVector.allocateNew(valueCount+1);
    offsetVector.zeroVector();
  }

    private void decrementAllocationMonitor() {
      if (allocationMonitor > 0) {
        allocationMonitor = 0;
      }
      --allocationMonitor;
    }

    private void incrementAllocationMonitor() {
      ++allocationMonitor;
    }

  public Accessor getAccessor(){
    return accessor;
  }
  
  public Mutator getMutator() {
    return mutator;
  }
  
  public final class Accessor extends BaseValueVector.BaseAccessor implements VariableWidthAccessor {
    final FieldReader reader = new Var16CharReaderImpl(Var16CharVector.this);
    final UInt4Vector.Accessor oAccessor = offsetVector.getAccessor();
    public FieldReader getReader(){
      return reader;
    }
    
    public long getStartEnd(int index){
      return oAccessor.getTwoAsLong(index);
    }
    
    public byte[] get(int index) {
      assert index >= 0;
      int startIdx = oAccessor.get(index);
      int length = oAccessor.get(index + 1) - startIdx;
      assert length >= 0;
      byte[] dst = new byte[length];
      data.getBytes(startIdx, dst, 0, length);
      return dst;
    }

    public int getValueLength(int index) {
      return offsetVector.getAccessor().get(index + 1) - offsetVector.getAccessor().get(index);
    }
    
    public void get(int index, Var16CharHolder holder){
      holder.start = oAccessor.get(index);
      holder.end = oAccessor.get(index + 1);
      holder.buffer = data;
    }
    
    public void get(int index, NullableVar16CharHolder holder){
      holder.isSet = 1;
      holder.start = oAccessor.get(index);
      holder.end = oAccessor.get(index + 1);
      holder.buffer = data;
    }
    

    public String getObject(int index) {
      return new String(get(index), Charsets.UTF_16);
    }
    
    
    
    public int getValueCount() {
      return valueCount;
    }

    public boolean isNull(int index){
      return false;
    }
    
    public UInt4Vector getOffsetVector(){
      return offsetVector;
    }
  }
  
  /**
   * MutableVar16Char implements a vector of variable width values.  Elements in the vector
   * are accessed by position from the logical start of the vector.  A fixed width offsetVector
   * is used to convert an element's position to it's offset from the start of the (0-based)
   * DrillBuf.  Size is inferred by adjacent elements.
   *   The width of each element is 4 byte(s)
   *   The equivalent Java primitive is 'int'
   *
   * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
   */
  public final class Mutator extends BaseValueVector.BaseMutator implements VariableWidthVector.VariableWidthMutator {

    /**
     * Set the variable length element at the specified index to the supplied byte array.
     *
     * @param index   position of the bit to set
     * @param bytes   array of bytes to write
     */
    protected void set(int index, byte[] bytes) {
      assert index >= 0;
      int currentOffset = offsetVector.getAccessor().get(index);
      offsetVector.getMutator().set(index + 1, currentOffset + bytes.length);
      data.setBytes(currentOffset, bytes, 0, bytes.length);
    }

    public boolean setSafe(int index, byte[] bytes) {
      assert index >= 0;

      int currentOffset = offsetVector.getAccessor().get(index);
      if (data.capacity() < currentOffset + bytes.length) {
        decrementAllocationMonitor();
        return false;
      }
      if (!offsetVector.getMutator().setSafe(index + 1, currentOffset + bytes.length)) {
        return false;
      }
      offsetVector.getMutator().set(index + 1, currentOffset + bytes.length);
      data.setBytes(currentOffset, bytes, 0, bytes.length);
      return true;
    }

    /**
     * Set the variable length element at the specified index to the supplied byte array.
     *
     * @param index   position of the bit to set
     * @param bytes   array of bytes to write
     * @param start   start index of bytes to write
     * @param length  length of bytes to write
     */
    protected void set(int index, byte[] bytes, int start, int length) {
      assert index >= 0;
      int currentOffset = offsetVector.getAccessor().get(index);
      offsetVector.getMutator().set(index + 1, currentOffset + length);
      data.setBytes(currentOffset, bytes, start, length);
    }

    public boolean setSafe(int index, byte[] bytes, int start, int length) {
      assert index >= 0;

      int currentOffset = offsetVector.getAccessor().get(index);

      if (data.capacity() < currentOffset + length) {
        decrementAllocationMonitor();
        return false;
      }
      if (!offsetVector.getMutator().setSafe(index + 1, currentOffset + length)) {
        return false;
      }
      data.setBytes(currentOffset, bytes, start, length);
      return true;
    }

    public boolean setValueLengthSafe(int index, int length) {
      int offset = offsetVector.getAccessor().get(index);
      if(data.capacity() < offset + length ) {
        decrementAllocationMonitor();
        return false;
      }
      return offsetVector.getMutator().setSafe(index + 1, offsetVector.getAccessor().get(index) + length);
    }


    public boolean setSafe(int index, int start, int end, DrillBuf buffer){
      int len = end - start;
      
      int outputStart = offsetVector.data.getInt(index * 4);
      
      if(data.capacity() < outputStart + len) {
        decrementAllocationMonitor();
        return false;
      }
      
      if (!offsetVector.getMutator().setSafe( index+1,  outputStart + len)) {
        return false;
      }
      buffer.getBytes(start, data, outputStart, len);

      return true;
    }
    
    
    public boolean setSafe(int index, NullableVar16CharHolder holder){
      assert holder.isSet == 1;

      int start = holder.start;
      int end =   holder.end;
      int len = end - start;
      
      int outputStart = offsetVector.data.getInt(index * 4);
      
      if(data.capacity() < outputStart + len) {
        decrementAllocationMonitor();
        return false;
      }
      
      holder.buffer.getBytes(start, data, outputStart, len);
      if (!offsetVector.getMutator().setSafe( index+1,  outputStart + len)) {
        return false;
      }

      // set(index, holder);

      return true;
    }
    
    public boolean setSafe(int index, Var16CharHolder holder){

      int start = holder.start;
      int end =   holder.end;
      int len = end - start;
      
      int outputStart = offsetVector.data.getInt(index * 4);
      
      if(data.capacity() < outputStart + len) {
        decrementAllocationMonitor();
        return false;
      }
      
      holder.buffer.getBytes(start, data, outputStart, len);
      if (!offsetVector.getMutator().setSafe( index+1,  outputStart + len)) {
        return false;
      }

      // set(index, holder);

      return true;
    }
    
    protected void set(int index, int start, int length, DrillBuf buffer){
      assert index >= 0;
      int currentOffset = offsetVector.getAccessor().get(index);
      offsetVector.getMutator().set(index + 1, currentOffset + length);
      DrillBuf bb = buffer.slice(start, length);
      data.setBytes(currentOffset, bb);
    }

    protected void set(int index, NullableVar16CharHolder holder){
      int length = holder.end - holder.start;
      int currentOffset = offsetVector.getAccessor().get(index);
      offsetVector.getMutator().set(index + 1, currentOffset + length);
      data.setBytes(currentOffset, holder.buffer, holder.start, length);
    }
    
    protected void set(int index, Var16CharHolder holder){
      int length = holder.end - holder.start;
      int currentOffset = offsetVector.getAccessor().get(index);
      offsetVector.getMutator().set(index + 1, currentOffset + length);
      data.setBytes(currentOffset, holder.buffer, holder.start, length);
    }
    
    public void setValueCount(int valueCount) {
      int currentByteCapacity = getByteCapacity();
      Var16CharVector.this.valueCount = valueCount;
      int idx = offsetVector.getAccessor().get(valueCount);
      data.writerIndex(idx);
      if (valueCount > 0 && currentByteCapacity > idx * 2) {
        incrementAllocationMonitor();
      } else if (allocationMonitor > 0) {
        allocationMonitor = 0;
      }
      VectorTrimmer.trim(data, idx);
      offsetVector.getMutator().setValueCount(valueCount == 0 ? 0 : valueCount+1);
    }

    @Override
    public void generateTestData(int size){
      boolean even = true;
      java.nio.charset.Charset charset = Charsets.UTF_16;
      for(int i =0; i < size; i++, even = !even){
        if(even){
          set(i, new String("aaaaa").getBytes(charset));
        }else{
          set(i, new String("bbbbbbbbbb").getBytes(charset));
        }
      }
      setValueCount(size);
    }
  }
  
}


 



 
