
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.vector;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.memory.OutOfMemoryRuntimeException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.hadoop.io.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






/**
 * NullableTinyInt implements a vector of values which could be null.  Elements in the vector
 * are first checked against a fixed length vector of boolean values.  Then the element is retrieved
 * from the base class (if not null).
 *
 * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
 */
@SuppressWarnings("unused")
public final class NullableTinyIntVector extends BaseValueVector implements FixedWidthVector, NullableVector{

  private int valueCount;
  final UInt1Vector bits;
  final TinyIntVector values;
  private final Accessor accessor;
  private final Mutator mutator;

  public NullableTinyIntVector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
    this.bits = new UInt1Vector(null, allocator);
    this.values = new TinyIntVector(field, allocator);
    this.accessor = new Accessor();
    this.mutator = new Mutator();
  }
  
  public int getValueCapacity(){
    return Math.min(bits.getValueCapacity(), values.getValueCapacity());
  }

  public int getCurrentValueCount() {
    return values.getCurrentValueCount();
  }

  public void setCurrentValueCount(int count) {
    values.setCurrentValueCount(count);
  }

  @Override
  public DrillBuf[] getBuffers(boolean clear) {
    DrillBuf[] buffers = ObjectArrays.concat(bits.getBuffers(clear), values.getBuffers(clear), DrillBuf.class);
    if (clear) {
      clear();
    }
    return buffers;
  }
  
  @Override
  public void clear() {
    valueCount = 0;
    bits.clear();
    values.clear();
  }
  
  public int getBufferSize(){
    return values.getBufferSize() + bits.getBufferSize();
  }

  public DrillBuf getData(){
    return values.getData();
  }

  public TinyIntVector getValuesVector() {
    return values;
  }

  @Override
  public SerializedField getMetadata() {
    return getMetadataBuilder()
             .setValueCount(valueCount)
             .setBufferLength(getBufferSize())
             .build();
  }

  @Override
  public void allocateNew() {
    values.allocateNew();
    bits.allocateNew();
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
  }
  

  @Override
  public boolean allocateNewSafe() {
    if(!values.allocateNewSafe()) return false;
    if(!bits.allocateNewSafe()) return false;
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
    return true;
  }

  @Override
  public void allocateNew(int valueCount) {
    values.allocateNew(valueCount);
    bits.allocateNew(valueCount);
    bits.zeroVector();
    mutator.reset();
    accessor.reset();
  }

  /**
   * {@inheritDoc}
   */
  public void zeroVector() {
    this.values.zeroVector();
    this.bits.zeroVector();
  }

  @Override
  public int load(int valueCount, DrillBuf buf){
    clear();
    this.valueCount = valueCount;
    int loaded = bits.load(valueCount, buf);
    
    // remove bits part of buffer.
    buf = buf.slice(loaded, buf.capacity() - loaded);
    loaded += values.load(valueCount, buf);
    return loaded;
  }
  
  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    assert this.field.matches(metadata);
    int loaded = load(metadata.getValueCount(), buffer);
    assert metadata.getBufferLength() == loaded;
  }
  
  
  public TransferPair getTransferPair(){
    return new TransferImpl(getField());
  }
  public TransferPair getTransferPair(FieldReference ref){
    return new TransferImpl(getField().clone(ref));
  }

  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((NullableTinyIntVector) to);
  }

  
  public void transferTo(NullableTinyIntVector target){
    bits.transferTo(target.bits);
    values.transferTo(target.values);
    target.valueCount = valueCount;
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, NullableTinyIntVector target) {
    bits.splitAndTransferTo(startIndex, length, target.bits);
    values.splitAndTransferTo(startIndex, length, target.values);
    target.valueCount = length;
  }
  
  private class TransferImpl implements TransferPair{
    NullableTinyIntVector to;
    
    public TransferImpl(MaterializedField field){
      this.to = new NullableTinyIntVector(field, allocator);
    }

    public TransferImpl(NullableTinyIntVector to){
      this.to = to;
    }

    public NullableTinyIntVector getTo(){
      return to;
    }
    
    public void transfer(){
      transferTo(to);
    }

    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public boolean copyValueSafe(int fromIndex, int toIndex) {
      return to.copyFromSafe(fromIndex, toIndex, NullableTinyIntVector.this);
    }
  }
  
  public Accessor getAccessor(){
    return accessor;
  }
  
  public Mutator getMutator(){
    return mutator;
  }
  
  public TinyIntVector convertToRequiredVector(){
    TinyIntVector v = new TinyIntVector(getField().getOtherNullableVersion(), allocator);
    v.data = values.data;
    v.valueCount = this.valueCount;
    v.data.retain();
    clear();
    return v;
  }

  
  protected void copyFrom(int fromIndex, int thisIndex, NullableTinyIntVector from){
    if (!from.getAccessor().isNull(fromIndex)) {
      mutator.set(thisIndex, from.getAccessor().get(fromIndex));
    }
  }

  
  public boolean copyFromSafe(int fromIndex, int thisIndex, TinyIntVector from){
    boolean success = values.copyFromSafe(fromIndex, thisIndex, from);
    success = success && bits.getMutator().setSafe(thisIndex, 1);
    return success;    
  }
  
  public boolean copyFromSafe(int fromIndex, int thisIndex, NullableTinyIntVector from){
    boolean b1 = bits.copyFromSafe(fromIndex, thisIndex, from.bits);
    boolean b2 = values.copyFromSafe(fromIndex, thisIndex, from.values);
    return b1 && b2;
  }

  public long getDataAddr(){
    return values.getDataAddr();
  }
  
  public long getBitAddr(){
    return bits.getDataAddr();
  }
  
  
  public final class Accessor implements ValueVector.Accessor{

    final FieldReader reader = new NullableTinyIntReaderImpl(NullableTinyIntVector.this);
    final UInt1Vector.Accessor bAccessor = bits.getAccessor();
    final TinyIntVector.Accessor vAccessor = values.getAccessor();
    
    public FieldReader getReader(){
      return reader;
    }
    
    /**
     * Get the element at the specified position.
     *
     * @param   index   position of the value
     * @return  value of the element, if not null
     * @throws  NullValueException if the value is null
     */
    public byte get(int index) {
      assert !isNull(index) : "Tried to get null value";
      return vAccessor.get(index);
    }

    public boolean isNull(int index) {
      return isSet(index) == 0;
    }

    public int isSet(int index){
      return bAccessor.get(index);
    }
    
    

    



    public void get(int index, NullableTinyIntHolder holder){
      vAccessor.get(index, holder);
      holder.isSet = bAccessor.get(index);

    }

    @Override
    public Byte getObject(int index) {
      if (isNull(index)) {
          return null;
      }else{
        return vAccessor.getObject(index);
      }
    }


    public int getValueCount(){
      return valueCount;
    }
    
    public void reset(){}
  }
  
  public final class Mutator implements ValueVector.Mutator, NullableVectorDefinitionSetter {
    
    private int setCount;
    

    private Mutator(){
    }

    public TinyIntVector getVectorWithValues(){
      return values;
    }

    public void setIndexDefined(int index){
      bits.getMutator().set(index, 1);
    }

    /**
     * Set the variable length element at the specified index to the supplied byte array.
     *
     * @param index   position of the bit to set
     * @param bytes   array of bytes to write
     */
    public void set(int index, int value) {
      setCount++;
      bits.getMutator().set(index, 1);
      values.getMutator().set(index, value);
      
    }

    
    public boolean setSafe(int index, byte[] value, int start, int length) {
      throw new UnsupportedOperationException();
    }

    public boolean setNull(int index){
      return bits.getMutator().setSafe(index, 0);
    }
    
    public void setSkipNull(int index, TinyIntHolder holder){
      values.getMutator().set(index, holder);
    }

    public void setSkipNull(int index, NullableTinyIntHolder holder){
      values.getMutator().set(index, holder);
    }
    
    
    public void set(int index, NullableTinyIntHolder holder){
      bits.getMutator().set(index, holder.isSet);
      values.getMutator().set(index, holder);
      
    }

    public void set(int index, TinyIntHolder holder){
      bits.getMutator().set(index, 1);
      values.getMutator().set(index, holder);
      
    }
    
    public boolean isSafe(int outIndex) {
      return outIndex < NullableTinyIntVector.this.getValueCapacity();
    }

    //public boolean setSafe(int index, int isSetint value){

    public void set(int index, int isSet, byte valueField ){
      bits.getMutator().set(index, isSet);
      values.getMutator().set(index, valueField);
      
    }
    
    public boolean setSafe(int index, int isSet, byte valueField ) {
      
      boolean b1 = bits.getMutator().setSafe(index, isSet);
      boolean b2 = values.getMutator().setSafe(index, valueField);
      if(b1 && b2){
        setCount++;
        
        return true;
      }else{
        return false;
      }

    }


    public boolean setSafe(int index, NullableTinyIntHolder value) {

      boolean b1 = bits.getMutator().setSafe(index, value.isSet);
      boolean b2 = values.getMutator().setSafe(index, value);
      if(b1 && b2){
        setCount++;
        
        return true;
      }else{
        return false;
      }

    }

    public boolean setSafe(int index, TinyIntHolder value) {

      boolean b1 = bits.getMutator().setSafe(index, 1);
      boolean b2 = values.getMutator().setSafe(index, value);
      if(b1 && b2){
        setCount++;
        
        return true;
      }else{
        return false;
      }

    }
    
      public boolean setSafe(int index, byte value) {
        boolean b1 = bits.getMutator().setSafe(index, 1);
        boolean b2 = values.getMutator().setSafe(index, value);
        if(b1 && b2){
          setCount++;
          return true;
        }else{
          return false;
        }
      }


    public void setValueCount(int valueCount) {
      assert valueCount >= 0;
      NullableTinyIntVector.this.valueCount = valueCount;
      values.getMutator().setValueCount(valueCount);
      bits.getMutator().setValueCount(valueCount);
    }
    
    public boolean noNulls(){
      return valueCount == setCount;
    }
    
    public void generateTestData(int valueCount){
      bits.getMutator().generateTestDataAlt(valueCount);
      values.getMutator().generateTestData(valueCount);
      
      setValueCount(valueCount);
    }
    
    public void reset(){
      setCount = 0;
      
    }
    
  }
}


