/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.store;

import com.fasterxml.jackson.core.JsonGenerator;
import com.google.common.collect.Lists;
import org.apache.drill.common.types.TypeProtos.MinorType;
import org.apache.drill.exec.expr.TypeHelper;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.record.BatchSchema;
import org.apache.drill.exec.store.EventBasedRecordWriter.FieldConverter;
import org.apache.drill.exec.store.parquet.ParquetTypeHelper;
import org.apache.drill.exec.vector.*;
import org.apache.drill.exec.util.DecimalUtility;
import org.apache.drill.exec.vector.complex.reader.FieldReader;
import parquet.io.api.RecordConsumer;
import parquet.schema.MessageType;
import parquet.io.api.Binary;
import io.netty.buffer.ByteBuf;
import org.apache.drill.exec.memory.TopLevelAllocator;
import org.apache.drill.exec.record.BatchSchema;
import org.apache.drill.exec.record.MaterializedField;


import org.apache.drill.common.types.TypeProtos;

import org.joda.time.DateTimeUtils;

import java.io.IOException;
import java.lang.UnsupportedOperationException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

/**
 * Abstract implementation of RecordWriter interface which exposes interface:
 *    {@link #writeHeader(List)}
 *    {@link #addField(int,String)}
 * to output the data in string format instead of implementing addField for each type holder.
 *
 * This is useful for text format writers such as CSV, TSV etc.
 */
public abstract class JSONOutputRecordWriter extends AbstractRecordWriter implements RecordWriter {

  protected JsonGenerator gen;

  @Override
  public FieldConverter getNewNullableTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableTinyIntJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableTinyIntJsonConverter extends FieldConverter {

    public NullableTinyIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readByte());
    }
  }
  @Override
  public FieldConverter getNewTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new TinyIntJsonConverter(fieldId, fieldName, reader);
  }

  public class TinyIntJsonConverter extends FieldConverter {

    public TinyIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readByte());
    }
  }
  @Override
  public FieldConverter getNewRepeatedTinyIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedTinyIntJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedTinyIntJsonConverter extends FieldConverter {

    public RepeatedTinyIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readByte(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableUInt1JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableUInt1JsonConverter extends FieldConverter {

    public NullableUInt1JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readByte());
    }
  }
  @Override
  public FieldConverter getNewUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    return new UInt1JsonConverter(fieldId, fieldName, reader);
  }

  public class UInt1JsonConverter extends FieldConverter {

    public UInt1JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readByte());
    }
  }
  @Override
  public FieldConverter getNewRepeatedUInt1Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedUInt1JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedUInt1JsonConverter extends FieldConverter {

    public RepeatedUInt1JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readByte(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableUInt2JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableUInt2JsonConverter extends FieldConverter {

    public NullableUInt2JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readCharacter());
    }
  }
  @Override
  public FieldConverter getNewUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    return new UInt2JsonConverter(fieldId, fieldName, reader);
  }

  public class UInt2JsonConverter extends FieldConverter {

    public UInt2JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readCharacter());
    }
  }
  @Override
  public FieldConverter getNewRepeatedUInt2Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedUInt2JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedUInt2JsonConverter extends FieldConverter {

    public RepeatedUInt2JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readCharacter(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableSmallIntJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableSmallIntJsonConverter extends FieldConverter {

    public NullableSmallIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readShort());
    }
  }
  @Override
  public FieldConverter getNewSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new SmallIntJsonConverter(fieldId, fieldName, reader);
  }

  public class SmallIntJsonConverter extends FieldConverter {

    public SmallIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readShort());
    }
  }
  @Override
  public FieldConverter getNewRepeatedSmallIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedSmallIntJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedSmallIntJsonConverter extends FieldConverter {

    public RepeatedSmallIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readShort(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableIntJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableIntJsonConverter extends FieldConverter {

    public NullableIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readInteger());
    }
  }
  @Override
  public FieldConverter getNewIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new IntJsonConverter(fieldId, fieldName, reader);
  }

  public class IntJsonConverter extends FieldConverter {

    public IntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readInteger());
    }
  }
  @Override
  public FieldConverter getNewRepeatedIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedIntJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedIntJsonConverter extends FieldConverter {

    public RepeatedIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readInteger(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableUInt4JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableUInt4JsonConverter extends FieldConverter {

    public NullableUInt4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readInteger());
    }
  }
  @Override
  public FieldConverter getNewUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new UInt4JsonConverter(fieldId, fieldName, reader);
  }

  public class UInt4JsonConverter extends FieldConverter {

    public UInt4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readInteger());
    }
  }
  @Override
  public FieldConverter getNewRepeatedUInt4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedUInt4JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedUInt4JsonConverter extends FieldConverter {

    public RepeatedUInt4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readInteger(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableFloat4JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableFloat4JsonConverter extends FieldConverter {

    public NullableFloat4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readFloat());
    }
  }
  @Override
  public FieldConverter getNewFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new Float4JsonConverter(fieldId, fieldName, reader);
  }

  public class Float4JsonConverter extends FieldConverter {

    public Float4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readFloat());
    }
  }
  @Override
  public FieldConverter getNewRepeatedFloat4Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedFloat4JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedFloat4JsonConverter extends FieldConverter {

    public RepeatedFloat4JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readFloat(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableTimeJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableTimeJsonConverter extends FieldConverter {

    public NullableTimeJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    return new TimeJsonConverter(fieldId, fieldName, reader);
  }

  public class TimeJsonConverter extends FieldConverter {

    public TimeJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedTimeConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedTimeJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedTimeJsonConverter extends FieldConverter {

    public RepeatedTimeJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readDateTime(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableIntervalYearJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableIntervalYearJsonConverter extends FieldConverter {

    public NullableIntervalYearJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    return new IntervalYearJsonConverter(fieldId, fieldName, reader);
  }

  public class IntervalYearJsonConverter extends FieldConverter {

    public IntervalYearJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedIntervalYearConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedIntervalYearJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedIntervalYearJsonConverter extends FieldConverter {

    public RepeatedIntervalYearJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readPeriod(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal9JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal9JsonConverter extends FieldConverter {

    public NullableDecimal9JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal9JsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal9JsonConverter extends FieldConverter {

    public Decimal9JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal9Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal9JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal9JsonConverter extends FieldConverter {

    public RepeatedDecimal9JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableBigIntJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableBigIntJsonConverter extends FieldConverter {

    public NullableBigIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readLong());
    }
  }
  @Override
  public FieldConverter getNewBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new BigIntJsonConverter(fieldId, fieldName, reader);
  }

  public class BigIntJsonConverter extends FieldConverter {

    public BigIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readLong());
    }
  }
  @Override
  public FieldConverter getNewRepeatedBigIntConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedBigIntJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedBigIntJsonConverter extends FieldConverter {

    public RepeatedBigIntJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readLong(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableUInt8JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableUInt8JsonConverter extends FieldConverter {

    public NullableUInt8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readLong());
    }
  }
  @Override
  public FieldConverter getNewUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new UInt8JsonConverter(fieldId, fieldName, reader);
  }

  public class UInt8JsonConverter extends FieldConverter {

    public UInt8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readLong());
    }
  }
  @Override
  public FieldConverter getNewRepeatedUInt8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedUInt8JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedUInt8JsonConverter extends FieldConverter {

    public RepeatedUInt8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readLong(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableFloat8JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableFloat8JsonConverter extends FieldConverter {

    public NullableFloat8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readDouble());
    }
  }
  @Override
  public FieldConverter getNewFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new Float8JsonConverter(fieldId, fieldName, reader);
  }

  public class Float8JsonConverter extends FieldConverter {

    public Float8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readDouble());
    }
  }
  @Override
  public FieldConverter getNewRepeatedFloat8Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedFloat8JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedFloat8JsonConverter extends FieldConverter {

    public RepeatedFloat8JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readDouble(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDateConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDateJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDateJsonConverter extends FieldConverter {

    public NullableDateJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewDateConverter(int fieldId, String fieldName, FieldReader reader) {
    return new DateJsonConverter(fieldId, fieldName, reader);
  }

  public class DateJsonConverter extends FieldConverter {

    public DateJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDateConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDateJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDateJsonConverter extends FieldConverter {

    public RepeatedDateJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readDateTime(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableTimeStampJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableTimeStampJsonConverter extends FieldConverter {

    public NullableTimeStampJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    return new TimeStampJsonConverter(fieldId, fieldName, reader);
  }

  public class TimeStampJsonConverter extends FieldConverter {

    public TimeStampJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedTimeStampConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedTimeStampJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedTimeStampJsonConverter extends FieldConverter {

    public RepeatedTimeStampJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readDateTime(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal18JsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal18JsonConverter extends FieldConverter {

    public NullableDecimal18JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal18JsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal18JsonConverter extends FieldConverter {

    public Decimal18JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal18Converter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal18JsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal18JsonConverter extends FieldConverter {

    public RepeatedDecimal18JsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableTimeStampTZConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableTimeStampTZJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableTimeStampTZJsonConverter extends FieldConverter {

    public NullableTimeStampTZJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewTimeStampTZConverter(int fieldId, String fieldName, FieldReader reader) {
    return new TimeStampTZJsonConverter(fieldId, fieldName, reader);
  }

  public class TimeStampTZJsonConverter extends FieldConverter {

    public TimeStampTZJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readDateTime().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedTimeStampTZConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedTimeStampTZJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedTimeStampTZJsonConverter extends FieldConverter {

    public RepeatedTimeStampTZJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readDateTime(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableIntervalDayJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableIntervalDayJsonConverter extends FieldConverter {

    public NullableIntervalDayJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    return new IntervalDayJsonConverter(fieldId, fieldName, reader);
  }

  public class IntervalDayJsonConverter extends FieldConverter {

    public IntervalDayJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedIntervalDayConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedIntervalDayJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedIntervalDayJsonConverter extends FieldConverter {

    public RepeatedIntervalDayJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readPeriod(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableIntervalJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableIntervalJsonConverter extends FieldConverter {

    public NullableIntervalJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    return new IntervalJsonConverter(fieldId, fieldName, reader);
  }

  public class IntervalJsonConverter extends FieldConverter {

    public IntervalJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readPeriod().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedIntervalConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedIntervalJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedIntervalJsonConverter extends FieldConverter {

    public RepeatedIntervalJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readPeriod(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal28DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal28DenseJsonConverter extends FieldConverter {

    public NullableDecimal28DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal28DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal28DenseJsonConverter extends FieldConverter {

    public Decimal28DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal28DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal28DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal28DenseJsonConverter extends FieldConverter {

    public RepeatedDecimal28DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal38DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal38DenseJsonConverter extends FieldConverter {

    public NullableDecimal38DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal38DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal38DenseJsonConverter extends FieldConverter {

    public Decimal38DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal38DenseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal38DenseJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal38DenseJsonConverter extends FieldConverter {

    public RepeatedDecimal38DenseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal38SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal38SparseJsonConverter extends FieldConverter {

    public NullableDecimal38SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal38SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal38SparseJsonConverter extends FieldConverter {

    public Decimal38SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal38SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal38SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal38SparseJsonConverter extends FieldConverter {

    public RepeatedDecimal38SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableDecimal28SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableDecimal28SparseJsonConverter extends FieldConverter {

    public NullableDecimal28SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new Decimal28SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class Decimal28SparseJsonConverter extends FieldConverter {

    public Decimal28SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeNumber(reader.readBigDecimal());
    }
  }
  @Override
  public FieldConverter getNewRepeatedDecimal28SparseConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedDecimal28SparseJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedDecimal28SparseJsonConverter extends FieldConverter {

    public RepeatedDecimal28SparseJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

      gen.writeNumber(reader.readBigDecimal(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableVarBinaryJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableVarBinaryJsonConverter extends FieldConverter {

    public NullableVarBinaryJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeBinary(reader.readByteArray());
    }
  }
  @Override
  public FieldConverter getNewVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    return new VarBinaryJsonConverter(fieldId, fieldName, reader);
  }

  public class VarBinaryJsonConverter extends FieldConverter {

    public VarBinaryJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeBinary(reader.readByteArray());
    }
  }
  @Override
  public FieldConverter getNewRepeatedVarBinaryConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedVarBinaryJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedVarBinaryJsonConverter extends FieldConverter {

    public RepeatedVarBinaryJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeBinary(reader.readByteArray(i));
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableVarCharJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableVarCharJsonConverter extends FieldConverter {

    public NullableVarCharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readText().toString());
    }
  }
  @Override
  public FieldConverter getNewVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new VarCharJsonConverter(fieldId, fieldName, reader);
  }

  public class VarCharJsonConverter extends FieldConverter {

    public VarCharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readText().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedVarCharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedVarCharJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedVarCharJsonConverter extends FieldConverter {

    public RepeatedVarCharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readText(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableVar16CharJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableVar16CharJsonConverter extends FieldConverter {

    public NullableVar16CharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeString(reader.readString().toString());
    }
  }
  @Override
  public FieldConverter getNewVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new Var16CharJsonConverter(fieldId, fieldName, reader);
  }

  public class Var16CharJsonConverter extends FieldConverter {

    public Var16CharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeString(reader.readString().toString());
    }
  }
  @Override
  public FieldConverter getNewRepeatedVar16CharConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedVar16CharJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedVar16CharJsonConverter extends FieldConverter {

    public RepeatedVar16CharJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeString(reader.readString(i).toString());
    }
      gen.writeEndArray();
    }
  }
  @Override
  public FieldConverter getNewNullableBitConverter(int fieldId, String fieldName, FieldReader reader) {
    return new NullableBitJsonConverter(fieldId, fieldName, reader);
  }

  public class NullableBitJsonConverter extends FieldConverter {

    public NullableBitJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    if (!reader.isSet()) {
      gen.writeNull();
      return;
    }

      gen.writeBoolean(reader.readBoolean());
    }
  }
  @Override
  public FieldConverter getNewBitConverter(int fieldId, String fieldName, FieldReader reader) {
    return new BitJsonConverter(fieldId, fieldName, reader);
  }

  public class BitJsonConverter extends FieldConverter {

    public BitJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {

      gen.writeBoolean(reader.readBoolean());
    }
  }
  @Override
  public FieldConverter getNewRepeatedBitConverter(int fieldId, String fieldName, FieldReader reader) {
    return new RepeatedBitJsonConverter(fieldId, fieldName, reader);
  }

  public class RepeatedBitJsonConverter extends FieldConverter {

    public RepeatedBitJsonConverter(int fieldId, String fieldName, FieldReader reader) {
      super(fieldId, fieldName, reader);
    }

    @Override
    public void startField() throws IOException {
      gen.writeFieldName(fieldName);
    }

    @Override
    public void writeField() throws IOException {
    // empty lists are represented by simply not starting a field, rather than starting one and putting in 0 elements
    if (reader.size() == 0) {
      return;
    }
    gen.writeStartArray();
    for (int i = 0; i < reader.size(); i++) {

              gen.writeBoolean(reader.readBoolean(i));
    }
      gen.writeEndArray();
    }
  }

}
