/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.DroppedSnapshotException;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.regionserver.FlushRequester;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.HRegionServer;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.HasThread;
import org.apache.hadoop.util.StringUtils;
import org.cliffc.high_scale_lib.Counter;

class MemStoreFlusher
extends HasThread
implements FlushRequester {
    static final Log LOG = LogFactory.getLog(MemStoreFlusher.class);
    private final BlockingQueue<FlushQueueEntry> flushQueue = new DelayQueue<FlushQueueEntry>();
    private final Map<HRegion, FlushRegionEntry> regionsInQueue = new HashMap<HRegion, FlushRegionEntry>();
    private AtomicBoolean wakeupPending = new AtomicBoolean();
    private final long threadWakeFrequency;
    private final HRegionServer server;
    private final ReentrantLock lock = new ReentrantLock();
    private final Condition flushOccurred = this.lock.newCondition();
    protected final long globalMemStoreLimit;
    protected final long globalMemStoreLimitLowMark;
    private static final float DEFAULT_UPPER = 0.4f;
    private static final float DEFAULT_LOWER = 0.35f;
    private static final String UPPER_KEY = "hbase.regionserver.global.memstore.upperLimit";
    private static final String LOWER_KEY = "hbase.regionserver.global.memstore.lowerLimit";
    private long blockingWaitTime;
    private final Counter updatesBlockedMsHighWater = new Counter();

    public MemStoreFlusher(Configuration conf, HRegionServer server) {
        this.server = server;
        this.threadWakeFrequency = conf.getLong("hbase.server.thread.wakefrequency", 10000L);
        long max = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage().getMax();
        this.globalMemStoreLimit = MemStoreFlusher.globalMemStoreLimit(max, 0.4f, UPPER_KEY, conf);
        long lower = MemStoreFlusher.globalMemStoreLimit(max, 0.35f, LOWER_KEY, conf);
        if (lower > this.globalMemStoreLimit) {
            lower = this.globalMemStoreLimit;
            LOG.info((Object)"Setting globalMemStoreLimitLowMark == globalMemStoreLimit because supplied hbase.regionserver.global.memstore.lowerLimit was > hbase.regionserver.global.memstore.upperLimit");
        }
        this.globalMemStoreLimitLowMark = lower;
        this.blockingWaitTime = conf.getInt("hbase.hstore.blockingWaitTime", 90000);
        LOG.info((Object)("globalMemStoreLimit=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimit) + ", globalMemStoreLimitLowMark=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimitLowMark) + ", maxHeap=" + StringUtils.humanReadableInt((long)max)));
    }

    static long globalMemStoreLimit(long max, float defaultLimit, String key, Configuration c) {
        float limit = c.getFloat(key, defaultLimit);
        return MemStoreFlusher.getMemStoreLimit(max, limit, defaultLimit);
    }

    static long getMemStoreLimit(long max, float limit, float defaultLimit) {
        float effectiveLimit = limit;
        if (limit >= 0.9f || limit < 0.1f) {
            LOG.warn((Object)("Setting global memstore limit to default of " + defaultLimit + " because supplied value outside allowed range of 0.1 -> 0.9"));
            effectiveLimit = defaultLimit;
        }
        return (long)((float)max * effectiveLimit);
    }

    public Counter getUpdatesBlockedMsHighWater() {
        return this.updatesBlockedMsHighWater;
    }

    private boolean flushOneForGlobalPressure() {
        SortedMap<Long, HRegion> regionsBySize = this.server.getCopyOfOnlineRegionsSortedBySize();
        HashSet<HRegion> excludedRegions = new HashSet<HRegion>();
        boolean flushedOne = false;
        while (!flushedOne) {
            HRegion regionToFlush;
            HRegion bestFlushableRegion = this.getBiggestMemstoreRegion(regionsBySize, excludedRegions, true);
            HRegion bestAnyRegion = this.getBiggestMemstoreRegion(regionsBySize, excludedRegions, false);
            if (bestAnyRegion == null) {
                LOG.error((Object)"Above memory mark but there are no flushable regions!");
                return false;
            }
            if (bestFlushableRegion != null && bestAnyRegion.memstoreSize.get() > 2L * bestFlushableRegion.memstoreSize.get()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Under global heap pressure: Region " + bestAnyRegion.getRegionNameAsString() + " has too many " + "store files, but is " + StringUtils.humanReadableInt((long)bestAnyRegion.memstoreSize.get()) + " vs best flushable region's " + StringUtils.humanReadableInt((long)bestFlushableRegion.memstoreSize.get()) + ". Choosing the bigger."));
                }
                regionToFlush = bestAnyRegion;
            } else {
                regionToFlush = bestFlushableRegion == null ? bestAnyRegion : bestFlushableRegion;
            }
            Preconditions.checkState((regionToFlush.memstoreSize.get() > 0L ? 1 : 0) != 0);
            LOG.info((Object)("Flush of region " + regionToFlush + " due to global heap pressure"));
            flushedOne = this.flushRegion(regionToFlush, true);
            if (flushedOne) continue;
            LOG.info((Object)("Excluding unflushable region " + regionToFlush + " - trying to find a different region to flush."));
            excludedRegions.add(regionToFlush);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        while (!this.server.isStopped()) {
            FlushQueueEntry fqe = null;
            try {
                this.wakeupPending.set(false);
                fqe = this.flushQueue.poll(this.threadWakeFrequency, TimeUnit.MILLISECONDS);
                if (fqe == null || fqe instanceof WakeupFlushThread) {
                    if (!this.isAboveLowWaterMark()) continue;
                    LOG.debug((Object)("Flush thread woke up because memory above low water=" + StringUtils.humanReadableInt((long)this.globalMemStoreLimitLowMark)));
                    if (!this.flushOneForGlobalPressure()) {
                        this.lock.lock();
                        try {
                            Thread.sleep(1000L);
                            this.flushOccurred.signalAll();
                        }
                        finally {
                            this.lock.unlock();
                        }
                    }
                    this.wakeupFlushThread();
                    continue;
                }
                FlushRegionEntry fre = (FlushRegionEntry)fqe;
                if (this.flushRegion(fre)) continue;
                break;
            }
            catch (InterruptedException ex) {
            }
            catch (ConcurrentModificationException ex) {
            }
            catch (Exception ex) {
                LOG.error((Object)("Cache flusher failed for entry " + fqe), (Throwable)ex);
                if (this.server.checkFileSystem()) continue;
                break;
            }
        }
        this.regionsInQueue.clear();
        this.flushQueue.clear();
        this.lock.lock();
        try {
            this.flushOccurred.signalAll();
        }
        finally {
            this.lock.unlock();
        }
        LOG.info((Object)(this.getName() + " exiting"));
    }

    private void wakeupFlushThread() {
        if (this.wakeupPending.compareAndSet(false, true)) {
            this.flushQueue.add(new WakeupFlushThread());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private HRegion getBiggestMemstoreRegion(SortedMap<Long, HRegion> regionsBySize, Set<HRegion> excludedRegions, boolean checkStoreFileCount) {
        Map<HRegion, FlushRegionEntry> map = this.regionsInQueue;
        synchronized (map) {
            for (HRegion region : regionsBySize.values()) {
                if (excludedRegions.contains(region) || checkStoreFileCount && this.isTooManyStoreFiles(region)) continue;
                return region;
            }
        }
        return null;
    }

    private boolean isAboveHighWaterMark() {
        return this.server.getRegionServerAccounting().getGlobalMemstoreSize() >= this.globalMemStoreLimit;
    }

    private boolean isAboveLowWaterMark() {
        return this.server.getRegionServerAccounting().getGlobalMemstoreSize() >= this.globalMemStoreLimitLowMark;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestFlush(HRegion r) {
        Map<HRegion, FlushRegionEntry> map = this.regionsInQueue;
        synchronized (map) {
            if (!this.regionsInQueue.containsKey(r)) {
                FlushRegionEntry fqe = new FlushRegionEntry(r);
                this.regionsInQueue.put(r, fqe);
                this.flushQueue.add(fqe);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void requestDelayedFlush(HRegion r, long delay) {
        Map<HRegion, FlushRegionEntry> map = this.regionsInQueue;
        synchronized (map) {
            if (!this.regionsInQueue.containsKey(r)) {
                FlushRegionEntry fqe = new FlushRegionEntry(r);
                fqe.requeue(delay);
                this.regionsInQueue.put(r, fqe);
                this.flushQueue.add(fqe);
            }
        }
    }

    public int getFlushQueueSize() {
        return this.flushQueue.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void interruptIfNecessary() {
        this.lock.lock();
        try {
            this.interrupt();
        }
        finally {
            this.lock.unlock();
        }
    }

    private boolean flushRegion(FlushRegionEntry fqe) {
        HRegion region = fqe.region;
        if (!fqe.region.getRegionInfo().isMetaRegion() && this.isTooManyStoreFiles(region)) {
            if (fqe.isMaximumWait(this.blockingWaitTime)) {
                LOG.info((Object)("Waited " + (System.currentTimeMillis() - fqe.createTime) + "ms on a compaction to clean up 'too many store files'; waited " + "long enough... proceeding with flush of " + region.getRegionNameAsString()));
            } else {
                if (fqe.getRequeueCount() <= 0) {
                    LOG.warn((Object)("Region " + region.getRegionNameAsString() + " has too many " + "store files; delaying flush up to " + this.blockingWaitTime + "ms"));
                    if (!this.server.compactSplitThread.requestSplit(region)) {
                        try {
                            this.server.compactSplitThread.requestCompaction(region, this.getName());
                        }
                        catch (IOException e) {
                            LOG.error((Object)("Cache flush failed" + (region != null ? " for region " + Bytes.toStringBinary(region.getRegionName()) : "")), (Throwable)RemoteExceptionHandler.checkIOException(e));
                        }
                    }
                }
                this.flushQueue.add(fqe.requeue(this.blockingWaitTime / 100L));
                return true;
            }
        }
        return this.flushRegion(region, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean flushRegion(HRegion region, boolean emergencyFlush) {
        Map<HRegion, FlushRegionEntry> map = this.regionsInQueue;
        synchronized (map) {
            FlushRegionEntry fqe = this.regionsInQueue.remove(region);
            if (fqe != null && emergencyFlush) {
                this.flushQueue.remove(fqe);
            }
            this.lock.lock();
        }
        try {
            boolean shouldSplit;
            boolean shouldCompact = region.flushcache().isCompactionNeeded();
            boolean bl = shouldSplit = region.checkSplit() != null;
            if (shouldSplit) {
                this.server.compactSplitThread.requestSplit(region);
            } else if (shouldCompact) {
                this.server.compactSplitThread.requestCompaction(region, this.getName());
            }
            this.server.getMetrics().addFlush(region.getRecentFlushInfo());
        }
        catch (DroppedSnapshotException ex) {
            this.server.abort("Replay of HLog required. Forcing server shutdown", ex);
            boolean bl = false;
            return bl;
        }
        catch (IOException ex) {
            LOG.error((Object)("Cache flush failed" + (region != null ? " for region " + Bytes.toStringBinary(region.getRegionName()) : "")), (Throwable)RemoteExceptionHandler.checkIOException(ex));
            if (!this.server.checkFileSystem()) {
                boolean bl = false;
                return bl;
            }
        }
        finally {
            this.flushOccurred.signalAll();
            this.lock.unlock();
        }
        return true;
    }

    private boolean isTooManyStoreFiles(HRegion region) {
        for (Store hstore : region.stores.values()) {
            if (!hstore.hasTooManyStoreFiles()) continue;
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void reclaimMemStoreMemory() {
        if (this.isAboveHighWaterMark()) {
            this.lock.lock();
            try {
                boolean blocked = false;
                long startTime = 0L;
                while (this.isAboveHighWaterMark() && !this.server.isStopped()) {
                    if (!blocked) {
                        startTime = EnvironmentEdgeManager.currentTimeMillis();
                        LOG.info((Object)("Blocking updates on " + this.server.toString() + ": the global memstore size " + StringUtils.humanReadableInt((long)this.server.getRegionServerAccounting().getGlobalMemstoreSize()) + " is >= than blocking " + StringUtils.humanReadableInt((long)this.globalMemStoreLimit) + " size"));
                    }
                    blocked = true;
                    this.wakeupFlushThread();
                    try {
                        this.flushOccurred.await(5L, TimeUnit.SECONDS);
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                if (!blocked) return;
                long totalTime = EnvironmentEdgeManager.currentTimeMillis() - startTime;
                if (totalTime > 0L) {
                    this.updatesBlockedMsHighWater.add(totalTime);
                }
                LOG.info((Object)("Unblocking updates for server " + this.server.toString()));
                return;
            }
            finally {
                this.lock.unlock();
            }
        } else {
            if (!this.isAboveLowWaterMark()) return;
            this.wakeupFlushThread();
        }
    }

    public String toString() {
        return "flush_queue=" + this.flushQueue.size();
    }

    public String dumpQueue() {
        StringBuilder queueList = new StringBuilder();
        queueList.append("Flush Queue Queue dump:\n");
        queueList.append("  Flush Queue:\n");
        Iterator it = this.flushQueue.iterator();
        while (it.hasNext()) {
            queueList.append("    " + ((FlushQueueEntry)it.next()).toString());
            queueList.append("\n");
        }
        return queueList.toString();
    }

    static class FlushRegionEntry
    implements FlushQueueEntry {
        private final HRegion region;
        private final long createTime;
        private long whenToExpire;
        private int requeueCount = 0;

        FlushRegionEntry(HRegion r) {
            this.region = r;
            this.whenToExpire = this.createTime = System.currentTimeMillis();
        }

        public boolean isMaximumWait(long maximumWait) {
            return System.currentTimeMillis() - this.createTime > maximumWait;
        }

        public int getRequeueCount() {
            return this.requeueCount;
        }

        public FlushRegionEntry requeue(long when) {
            this.whenToExpire = System.currentTimeMillis() + when;
            ++this.requeueCount;
            return this;
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.whenToExpire - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
        }

        @Override
        public int compareTo(Delayed other) {
            return Long.valueOf(this.getDelay(TimeUnit.MILLISECONDS) - other.getDelay(TimeUnit.MILLISECONDS)).intValue();
        }

        public String toString() {
            return "[flush region " + Bytes.toStringBinary(this.region.getRegionName()) + "]";
        }
    }

    static class WakeupFlushThread
    implements FlushQueueEntry {
        WakeupFlushThread() {
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return 0L;
        }

        @Override
        public int compareTo(Delayed o) {
            return -1;
        }
    }

    static interface FlushQueueEntry
    extends Delayed {
    }
}

