/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.coord.zk;

import com.google.common.base.Function;
import com.google.common.base.Throwables;
import com.google.common.collect.Collections2;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.retry.RetryNTimes;
import org.apache.curator.x.discovery.ServiceCache;
import org.apache.curator.x.discovery.ServiceDiscovery;
import org.apache.curator.x.discovery.ServiceDiscoveryBuilder;
import org.apache.curator.x.discovery.ServiceInstance;
import org.apache.curator.x.discovery.details.ServiceCacheListener;
import org.apache.drill.common.config.DrillConfig;
import org.apache.drill.exec.coord.ClusterCoordinator;
import org.apache.drill.exec.coord.DistributedSemaphore;
import org.apache.drill.exec.coord.DrillServiceInstanceHelper;
import org.apache.drill.exec.coord.zk.ZKRegistrationHandle;
import org.apache.drill.exec.coord.zk.ZkDistributedSemaphore;
import org.apache.drill.exec.proto.CoordinationProtos;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZKClusterCoordinator
extends ClusterCoordinator {
    static final Logger logger = LoggerFactory.getLogger(ZKClusterCoordinator.class);
    private CuratorFramework curator;
    private ServiceDiscovery<CoordinationProtos.DrillbitEndpoint> discovery;
    private ServiceCache<CoordinationProtos.DrillbitEndpoint> serviceCache;
    private volatile Collection<CoordinationProtos.DrillbitEndpoint> endpoints = Collections.emptyList();
    private final String serviceName;
    private final CountDownLatch initialConnection = new CountDownLatch(1);
    private static final Pattern ZK_COMPLEX_STRING = Pattern.compile("(^.*?)/(.*)/([^/]*)$");

    public ZKClusterCoordinator(DrillConfig config) throws IOException {
        this(config, null);
    }

    public ZKClusterCoordinator(DrillConfig config, String connect) throws IOException {
        connect = connect == null || connect.isEmpty() ? config.getString("drill.exec.zk.connect") : connect;
        String clusterId = config.getString("drill.exec.cluster-id");
        String zkRoot = config.getString("drill.exec.zk.root");
        Matcher m = ZK_COMPLEX_STRING.matcher(connect);
        if (m.matches()) {
            connect = m.group(1);
            zkRoot = m.group(2);
            clusterId = m.group(3);
        }
        logger.debug("Connect {}, zkRoot {}, clusterId: " + clusterId, (Object)connect, (Object)zkRoot);
        this.serviceName = clusterId;
        RetryNTimes rp = new RetryNTimes(config.getInt("drill.exec.zk.retry.count"), config.getInt("drill.exec.zk.retry.delay"));
        this.curator = CuratorFrameworkFactory.builder().namespace(zkRoot).connectionTimeoutMs(config.getInt("drill.exec.zk.timeout")).retryPolicy(rp).connectString(connect).build();
        this.curator.getConnectionStateListenable().addListener(new InitialConnectionListener());
        this.curator.start();
        this.discovery = this.getDiscovery();
        this.serviceCache = this.discovery.serviceCacheBuilder().name(this.serviceName).build();
    }

    @Override
    public void start(long millisToWait) throws Exception {
        logger.debug("Starting ZKClusterCoordination.");
        this.discovery.start();
        this.serviceCache.start();
        this.serviceCache.addListener(new ZKListener());
        if (millisToWait != 0L) {
            boolean success = this.initialConnection.await(millisToWait, TimeUnit.MILLISECONDS);
            if (!success) {
                throw new IOException(String.format("Failure to connect to the zookeeper cluster service within the allotted time of %d milliseconds.", millisToWait));
            }
        } else {
            this.initialConnection.await();
        }
        this.updateEndpoints();
    }

    @Override
    public void close() throws IOException {
        this.serviceCache.close();
        this.discovery.close();
        this.curator.close();
    }

    @Override
    public ClusterCoordinator.RegistrationHandle register(CoordinationProtos.DrillbitEndpoint data) {
        try {
            ServiceInstance<CoordinationProtos.DrillbitEndpoint> serviceInstance = this.getServiceInstance(data);
            this.discovery.registerService(serviceInstance);
            return new ZKRegistrationHandle(serviceInstance.getId());
        }
        catch (Exception e) {
            throw Throwables.propagate(e);
        }
    }

    @Override
    public void unregister(ClusterCoordinator.RegistrationHandle handle) {
        if (!(handle instanceof ZKRegistrationHandle)) {
            throw new UnsupportedOperationException("Unknown handle type: " + handle.getClass().getName());
        }
        this.listeners.clear();
        ZKRegistrationHandle h = (ZKRegistrationHandle)handle;
        try {
            ServiceInstance serviceInstance = ServiceInstance.builder().address("").port(0).id(h.id).name(this.serviceName).build();
            this.discovery.unregisterService(serviceInstance);
        }
        catch (Exception e) {
            Throwables.propagate(e);
        }
    }

    @Override
    public Collection<CoordinationProtos.DrillbitEndpoint> getAvailableEndpoints() {
        return this.endpoints;
    }

    @Override
    public DistributedSemaphore getSemaphore(String name, int maximumLeases) {
        return new ZkDistributedSemaphore(this.curator, "/semaphore/" + name, maximumLeases);
    }

    private synchronized void updateEndpoints() {
        try {
            Collection<CoordinationProtos.DrillbitEndpoint> newDrillbitSet = Collections2.transform(this.discovery.queryForInstances(this.serviceName), new Function<ServiceInstance<CoordinationProtos.DrillbitEndpoint>, CoordinationProtos.DrillbitEndpoint>(){

                public CoordinationProtos.DrillbitEndpoint apply(ServiceInstance<CoordinationProtos.DrillbitEndpoint> input) {
                    return (CoordinationProtos.DrillbitEndpoint)input.getPayload();
                }
            });
            HashSet<CoordinationProtos.DrillbitEndpoint> unregisteredBits = new HashSet<CoordinationProtos.DrillbitEndpoint>(this.endpoints);
            unregisteredBits.removeAll(newDrillbitSet);
            this.endpoints = newDrillbitSet;
            if (logger.isDebugEnabled()) {
                StringBuilder builder = new StringBuilder();
                builder.append("Active drillbit set changed.  Now includes ");
                builder.append(newDrillbitSet.size());
                builder.append(" total bits.  New active drillbits: \n");
                for (CoordinationProtos.DrillbitEndpoint bit : newDrillbitSet) {
                    builder.append('\t');
                    builder.append(bit.getAddress());
                    builder.append(':');
                    builder.append(bit.getUserPort());
                    builder.append(':');
                    builder.append(bit.getControlPort());
                    builder.append(':');
                    builder.append(bit.getDataPort());
                    builder.append('\n');
                }
                logger.debug(builder.toString());
            }
            if (!unregisteredBits.isEmpty()) {
                this.drillbitUnregistered(unregisteredBits);
            }
        }
        catch (Exception e) {
            logger.error("Failure while update Drillbit service location cache.", e);
        }
    }

    private ServiceInstance<CoordinationProtos.DrillbitEndpoint> getServiceInstance(CoordinationProtos.DrillbitEndpoint endpoint) throws Exception {
        return ServiceInstance.builder().name(this.serviceName).payload(endpoint).build();
    }

    public ServiceDiscovery<CoordinationProtos.DrillbitEndpoint> getDiscovery() {
        return ServiceDiscoveryBuilder.builder(CoordinationProtos.DrillbitEndpoint.class).basePath("/").client(this.curator).serializer(DrillServiceInstanceHelper.SERIALIZER).build();
    }

    private class ZKListener
    implements ServiceCacheListener {
        private ZKListener() {
        }

        @Override
        public void stateChanged(CuratorFramework client, ConnectionState newState) {
        }

        @Override
        public void cacheChanged() {
            logger.debug("Cache changed, updating.");
            ZKClusterCoordinator.this.updateEndpoints();
        }
    }

    private class InitialConnectionListener
    implements ConnectionStateListener {
        private InitialConnectionListener() {
        }

        @Override
        public void stateChanged(CuratorFramework client, ConnectionState newState) {
            if (newState == ConnectionState.CONNECTED) {
                ZKClusterCoordinator.this.initialConnection.countDown();
                client.getConnectionStateListenable().removeListener(this);
            }
        }
    }
}

