/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.shuffle;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import java.io.DataInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import org.apache.spark.network.buffer.FileSegmentManagedBuffer;
import org.apache.spark.network.buffer.ManagedBuffer;
import org.apache.spark.network.shuffle.protocol.ExecutorShuffleInfo;
import org.apache.spark.network.util.JavaUtils;
import org.apache.spark.network.util.TransportConf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExternalShuffleBlockManager {
    private final Logger logger = LoggerFactory.getLogger(ExternalShuffleBlockManager.class);
    private final ConcurrentMap<AppExecId, ExecutorShuffleInfo> executors;
    private final Executor directoryCleaner;
    private final TransportConf conf;

    public ExternalShuffleBlockManager(TransportConf conf) {
        this(conf, Executors.newSingleThreadExecutor());
    }

    @VisibleForTesting
    ExternalShuffleBlockManager(TransportConf conf, Executor directoryCleaner) {
        this.conf = conf;
        this.executors = Maps.newConcurrentMap();
        this.directoryCleaner = directoryCleaner;
    }

    public void registerExecutor(String appId, String execId, ExecutorShuffleInfo executorInfo) {
        AppExecId fullId = new AppExecId(appId, execId);
        this.logger.info("Registered executor {} with {}", (Object)fullId, (Object)executorInfo);
        this.executors.put(fullId, executorInfo);
    }

    public ManagedBuffer getBlockData(String appId, String execId, String blockId) {
        String[] blockIdParts = blockId.split("_");
        if (blockIdParts.length < 4) {
            throw new IllegalArgumentException("Unexpected block id format: " + blockId);
        }
        if (!blockIdParts[0].equals("shuffle")) {
            throw new IllegalArgumentException("Expected shuffle block id, got: " + blockId);
        }
        int shuffleId = Integer.parseInt(blockIdParts[1]);
        int mapId = Integer.parseInt(blockIdParts[2]);
        int reduceId = Integer.parseInt(blockIdParts[3]);
        ExecutorShuffleInfo executor = (ExecutorShuffleInfo)this.executors.get(new AppExecId(appId, execId));
        if (executor == null) {
            throw new RuntimeException(String.format("Executor is not registered (appId=%s, execId=%s)", appId, execId));
        }
        if ("org.apache.spark.shuffle.hash.HashShuffleManager".equals(executor.shuffleManager)) {
            return this.getHashBasedShuffleBlockData(executor, blockId);
        }
        if ("org.apache.spark.shuffle.sort.SortShuffleManager".equals(executor.shuffleManager)) {
            return this.getSortBasedShuffleBlockData(executor, shuffleId, mapId, reduceId);
        }
        throw new UnsupportedOperationException("Unsupported shuffle manager: " + executor.shuffleManager);
    }

    public void applicationRemoved(String appId, boolean cleanupLocalDirs) {
        this.logger.info("Application {} removed, cleanupLocalDirs = {}", (Object)appId, (Object)cleanupLocalDirs);
        Iterator it = this.executors.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry entry = it.next();
            AppExecId fullId = (AppExecId)entry.getKey();
            final ExecutorShuffleInfo executor = (ExecutorShuffleInfo)entry.getValue();
            if (!appId.equals(fullId.appId)) continue;
            it.remove();
            if (!cleanupLocalDirs) continue;
            this.logger.info("Cleaning up executor {}'s {} local dirs", (Object)fullId, (Object)executor.localDirs.length);
            this.directoryCleaner.execute(new Runnable(){

                @Override
                public void run() {
                    ExternalShuffleBlockManager.this.deleteExecutorDirs(executor.localDirs);
                }
            });
        }
    }

    private void deleteExecutorDirs(String[] dirs) {
        for (String localDir : dirs) {
            try {
                JavaUtils.deleteRecursively(new File(localDir));
                this.logger.debug("Successfully cleaned up directory: " + localDir);
            }
            catch (Exception e) {
                this.logger.error("Failed to delete directory: " + localDir, (Throwable)e);
            }
        }
    }

    private ManagedBuffer getHashBasedShuffleBlockData(ExecutorShuffleInfo executor, String blockId) {
        File shuffleFile = ExternalShuffleBlockManager.getFile(executor.localDirs, executor.subDirsPerLocalDir, blockId);
        return new FileSegmentManagedBuffer(this.conf, shuffleFile, 0L, shuffleFile.length());
    }

    private ManagedBuffer getSortBasedShuffleBlockData(ExecutorShuffleInfo executor, int shuffleId, int mapId, int reduceId) {
        FileSegmentManagedBuffer fileSegmentManagedBuffer;
        block5: {
            File indexFile = ExternalShuffleBlockManager.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.index");
            DataInputStream in = null;
            try {
                in = new DataInputStream(new FileInputStream(indexFile));
                in.skipBytes(reduceId * 8);
                long offset = in.readLong();
                long nextOffset = in.readLong();
                fileSegmentManagedBuffer = new FileSegmentManagedBuffer(this.conf, ExternalShuffleBlockManager.getFile(executor.localDirs, executor.subDirsPerLocalDir, "shuffle_" + shuffleId + "_" + mapId + "_0.data"), offset, nextOffset - offset);
                if (in == null) break block5;
            }
            catch (IOException e) {
                try {
                    throw new RuntimeException("Failed to open file: " + indexFile, e);
                }
                catch (Throwable throwable) {
                    if (in != null) {
                        JavaUtils.closeQuietly(in);
                    }
                    throw throwable;
                }
            }
            JavaUtils.closeQuietly(in);
        }
        return fileSegmentManagedBuffer;
    }

    @VisibleForTesting
    static File getFile(String[] localDirs, int subDirsPerLocalDir, String filename) {
        int hash = JavaUtils.nonNegativeHash(filename);
        String localDir = localDirs[hash % localDirs.length];
        int subDirId = hash / localDirs.length % subDirsPerLocalDir;
        return new File(new File(localDir, String.format("%02x", subDirId)), filename);
    }

    private static class AppExecId {
        final String appId;
        final String execId;

        private AppExecId(String appId, String execId) {
            this.appId = appId;
            this.execId = execId;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AppExecId appExecId = (AppExecId)o;
            return Objects.equal((Object)this.appId, (Object)appExecId.appId) && Objects.equal((Object)this.execId, (Object)appExecId.execId);
        }

        public int hashCode() {
            return Objects.hashCode((Object[])new Object[]{this.appId, this.execId});
        }

        public String toString() {
            return Objects.toStringHelper((Object)this).add("appId", (Object)this.appId).add("execId", (Object)this.execId).toString();
        }
    }
}

