/*
 * Decompiled with CFR 0.152.
 */
package org.apache.spark.network.client;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.util.concurrent.SettableFuture;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import java.io.Closeable;
import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.spark.network.client.ChunkReceivedCallback;
import org.apache.spark.network.client.RpcResponseCallback;
import org.apache.spark.network.client.TransportResponseHandler;
import org.apache.spark.network.protocol.ChunkFetchRequest;
import org.apache.spark.network.protocol.RpcRequest;
import org.apache.spark.network.protocol.StreamChunkId;
import org.apache.spark.network.util.NettyUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TransportClient
implements Closeable {
    private final Logger logger = LoggerFactory.getLogger(TransportClient.class);
    private final Channel channel;
    private final TransportResponseHandler handler;

    public TransportClient(Channel channel, TransportResponseHandler handler) {
        this.channel = (Channel)Preconditions.checkNotNull((Object)channel);
        this.handler = (TransportResponseHandler)Preconditions.checkNotNull((Object)handler);
    }

    public boolean isActive() {
        return this.channel.isOpen() || this.channel.isActive();
    }

    public void fetchChunk(long streamId, final int chunkIndex, final ChunkReceivedCallback callback) {
        final String serverAddr = NettyUtils.getRemoteAddress(this.channel);
        final long startTime = System.currentTimeMillis();
        this.logger.debug("Sending fetch chunk request {} to {}", (Object)chunkIndex, (Object)serverAddr);
        final StreamChunkId streamChunkId = new StreamChunkId(streamId, chunkIndex);
        this.handler.addFetchRequest(streamChunkId, callback);
        this.channel.writeAndFlush(new ChunkFetchRequest(streamChunkId)).addListener(new ChannelFutureListener(){

            @Override
            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    long timeTaken = System.currentTimeMillis() - startTime;
                    TransportClient.this.logger.trace("Sending request {} to {} took {} ms", new Object[]{streamChunkId, serverAddr, timeTaken});
                } else {
                    String errorMsg = String.format("Failed to send request %s to %s: %s", streamChunkId, serverAddr, future.cause());
                    TransportClient.this.logger.error(errorMsg, future.cause());
                    TransportClient.this.handler.removeFetchRequest(streamChunkId);
                    TransportClient.this.channel.close();
                    try {
                        callback.onFailure(chunkIndex, new IOException(errorMsg, future.cause()));
                    }
                    catch (Exception e) {
                        TransportClient.this.logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                    }
                }
            }
        });
    }

    public void sendRpc(byte[] message, final RpcResponseCallback callback) {
        final String serverAddr = NettyUtils.getRemoteAddress(this.channel);
        final long startTime = System.currentTimeMillis();
        this.logger.trace("Sending RPC to {}", (Object)serverAddr);
        final long requestId = Math.abs(UUID.randomUUID().getLeastSignificantBits());
        this.handler.addRpcRequest(requestId, callback);
        this.channel.writeAndFlush(new RpcRequest(requestId, message)).addListener(new ChannelFutureListener(){

            @Override
            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    long timeTaken = System.currentTimeMillis() - startTime;
                    TransportClient.this.logger.trace("Sending request {} to {} took {} ms", new Object[]{requestId, serverAddr, timeTaken});
                } else {
                    String errorMsg = String.format("Failed to send RPC %s to %s: %s", requestId, serverAddr, future.cause());
                    TransportClient.this.logger.error(errorMsg, future.cause());
                    TransportClient.this.handler.removeRpcRequest(requestId);
                    TransportClient.this.channel.close();
                    try {
                        callback.onFailure(new IOException(errorMsg, future.cause()));
                    }
                    catch (Exception e) {
                        TransportClient.this.logger.error("Uncaught exception in RPC response callback handler!", (Throwable)e);
                    }
                }
            }
        });
    }

    public byte[] sendRpcSync(byte[] message, long timeoutMs) {
        final SettableFuture result = SettableFuture.create();
        this.sendRpc(message, new RpcResponseCallback(){

            @Override
            public void onSuccess(byte[] response) {
                result.set((Object)response);
            }

            @Override
            public void onFailure(Throwable e) {
                result.setException(e);
            }
        });
        try {
            return (byte[])result.get(timeoutMs, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            throw Throwables.propagate((Throwable)e.getCause());
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    @Override
    public void close() {
        this.channel.close().awaitUninterruptibly(10L, TimeUnit.SECONDS);
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add("remoteAdress", (Object)this.channel.remoteAddress()).add("isActive", this.isActive()).toString();
    }
}

