/*
	drivedlg.c

	Virtual Floppy Disk drive control panel
	Copyright (C) 2003 Kenji Kato
*/

#include "sysincl.h"
#include "vfdctl.h"
#include "vfdutil.h"
#include "vfdwin.h"
#include "resource.h"

//
//	local definitions
//

//	OPENFILENAME flag values in commdlg.h for WINVER >= 0x0500

//	we don't want *.sys file to be added to recent docs folder
#define OFN_DONTADDTORECENT		0x02000000

//	we want *.sys files to be shown in open dialog box
#define OFN_FORCESHOWHIDDEN		0x10000000

//	fallback text strings for the open file dialog
#define FALLBACK_IMAGE_FILTER	"Known image files (bin,dat,fdd,flp,ima,img,vfd)\0" \
								"*.bin;*.dat;*.fdd;*.flp;*.ima;*.img;*.vfd\0" \
								"All files (*.*)\0*.*\0"
#define FALLBACK_IMAGE_TITLE	"Virtual FD Image"

#define FALLBACK_DRIVER_FILTER	"Virtual FD Driver (vfd.sys)\0vfd.sys\0"
#define FALLBACK_DRIVER_TITLE	"Virtual FD Driver"

#define FALLBACK_FORCE_STOP		"Failed to unmount current image.\r\n" \
								"The driver may not stop properly.  Stop it anyway?"
#define FALLBACK_FORCE_REMOVE	"Failed to stop the Virtual FD driver.\r\n" \
								"The driver may not unload properly. Remove it anyway?"

//
//	Windows message handlers
//
static void OnInitDialog	(HWND hDlg);

static void OnImagePathEdit	(HWND hDlg, HWND hEdit);
static void OnImageBrowse	(HWND hDlg);
static void OnImageMount	(HWND hDlg, HWND hButton);
static void OnImageUnmount	(HWND hDlg, HWND hButton);

static void OnDriverPathEdit(HWND hDlg, HWND hEdit);
static void OnDriverBrowse	(HWND hDlg);
static void OnDriverInstall	(HWND hDlg, HWND hButton);
static void OnDriverRemove	(HWND hDlg, HWND hButton);
static void OnDriverStart	(HWND hDlg, HWND hButton);
static void OnDriverStop	(HWND hDlg, HWND hButton);

//
//	local functions
//
static BOOL ImageMount		(HWND hDlg);
static BOOL DriverInstall	(HWND hDlg);
static BOOL DriverRemove	(HWND hDlg);
static BOOL DriverStart		(HWND hDlg);
static BOOL DriverStop		(HWND hDlg);
static BOOL AskForceStop	(HWND hDlg);
static BOOL AskForceRemove	(HWND hDlg);
static void AppendMessage	(HWND hDlg, UINT msg, DWORD err);

//
//	window message dispatcher
//

BOOL CALLBACK DriverProc(
	HWND hDlg,
	UINT msg,
	WPARAM wParam,
	LPARAM lParam)
{
	switch (msg) {
	case WM_INITDIALOG:
		OnInitDialog(hDlg);
		break;

	case WM_COMMAND:
		switch (wParam) {
		case MAKELONG(IDC_IMAGE_PATH, EN_CHANGE):
			OnImagePathEdit(hDlg, (HWND)lParam);
			break;

		case IDC_IMAGE_BROWSE:
			OnImageBrowse(hDlg);
			break;

		case IDC_IMAGE_MOUNT:
			OnImageMount(hDlg, (HWND)lParam);
			break;

		case IDC_IMAGE_UNMOUNT:
			OnImageUnmount(hDlg, (HWND)lParam);
			break;

		case MAKELONG(IDC_DRIVER_PATH, EN_CHANGE):
			OnDriverPathEdit(hDlg, (HWND)lParam);
			break;

		case IDC_DRIVER_BROWSE:
			OnDriverBrowse(hDlg);
			break;

		case IDC_DRIVER_INSTALL:
			OnDriverInstall(hDlg, (HWND)lParam);
			break;

		case IDC_DRIVER_REMOVE:
			OnDriverRemove(hDlg, (HWND)lParam);
			break;

		case IDC_DRIVER_START:
			OnDriverStart(hDlg, (HWND)lParam);
			break;

		case IDC_DRIVER_STOP:
			OnDriverStop(hDlg, (HWND)lParam);
			break;
		}
		break;
	}
	return 0;
}

//
//	initialize drive dialog
//
void OnInitDialog(HWND hDlg)
{
	HWND	hCombo;
	DWORD	logical_drives;
	char	drive[] = " :";
	DWORD	len;
	char	driver_path[MAX_PATH];

	//	Init Drive Letter combo box

	hCombo = GetDlgItem(hDlg, IDC_IMAGE_DRIVE);

	if (hCombo == NULL) {
		DEBUG_TRACE1(
			"OnImageInitDialog: GetDlgItem - %s", ErrMsg(GetLastError()));

		return;
	}

	SendMessage(hCombo, CB_RESETCONTENT, 0, 0);

	//	search all unused drive letters

	logical_drives = GetLogicalDrives();

	if (logical_drives == 0) {
		DEBUG_TRACE1(
			"OnImageInitDialog: GetLogicalDrives - %s", ErrMsg(GetLastError()));

		return;
	}

	for (drive[0] = 'A'; drive[0] <= 'Z'; drive[0]++) {
		if (!(logical_drives & 0x01)) {
			SendMessage(hCombo, CB_ADDSTRING, 0, (LPARAM)drive);
		}
		logical_drives >>= 1;
	}

	//	add drive letter used by Virtual FD

	if (VfdGetDriveLetter(&drive[0]) == ERROR_SUCCESS && isalpha(drive[0])) {
		SendMessage(hCombo, CB_ADDSTRING, 0, (LPARAM)drive);
	}
	else {
		drive[0] = ChooseDriveLetter();
	}

	SendMessage(hCombo, CB_SELECTSTRING, 0, (LPARAM)drive);

	//	prepare default driver path

	len = GetModuleFileName(
			hAppInstance, driver_path, sizeof(driver_path));

	if (len == 0) {
		DEBUG_TRACE1(
			"OnDriverInitDialog : GetModuleFileName - %s", ErrMsg(GetLastError()));

		return;
	}

	//	search the last '\' char

	while (len > 0 && driver_path[len - 1] != '\\') {
		len --;
	}

	strcpy(&driver_path[len], "vfd.sys");

	SetDlgItemText(hDlg, IDC_DRIVER_PATH, driver_path);

	//
	//	choose default radio buttons
	//
	CheckRadioButton(hDlg, IDC_IMAGE_720KB, IDC_IMAGE_2P88MB, IDC_IMAGE_1P44MB);
	CheckRadioButton(hDlg, IDC_DRIVER_MANUAL, IDC_DRIVER_AUTO, IDC_DRIVER_MANUAL);

	return;
}

//
//	Image path is changed
//
void OnImagePathEdit(HWND hDlg, HWND hEdit)
{
	EnableWindow(
		GetDlgItem(hDlg, IDC_IMAGE_MOUNT),
		(GetWindowTextLength(hEdit) != 0));
}

//
//	Driver path is changed
//
void OnDriverPathEdit(HWND hDlg, HWND hEdit)
{
	BOOL enable = (GetWindowTextLength(hEdit) != 0);

	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_MOUNT), enable);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_START), enable);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_INSTALL), enable);
}

//
//	image browse button is clicked
//
void OnImageBrowse(HWND hDlg)
{
	OPENFILENAME ofn;
	TCHAR path[MAX_PATH], dir[MAX_PATH];
	TCHAR filter[150];
	TCHAR title[50];
	LPTSTR p;
	DWORD attr;
	int len;

	//
	//	prepare filter for the open file dialog
	//
	if (LoadString(hAppInstance, IDS_IMAGE_OPEN_FILTER, filter, sizeof(filter))) {
		LPTSTR p = filter;

		while (*p) {
			if (*p == '|') {
				*p = '\0';
			}
			p++;
		}
	}
	else {
		DEBUG_TRACE1(
			"OnImageBrowse : LoadString - %s", ErrMsg(GetLastError()));

		memcpy(filter, FALLBACK_IMAGE_FILTER, sizeof(FALLBACK_IMAGE_FILTER));
	}

	//
	//	prepare title text for the open file dialog
	//
	if (!LoadString(hAppInstance, IDS_IMAGE_OPEN_TITLE, title, sizeof(title))) {
		DEBUG_TRACE1(
			"OnImageBrowse : LoadString - %s", ErrMsg(GetLastError()));

		strcpy(title, FALLBACK_IMAGE_TITLE);
	}

	GetDlgItemText(hDlg, IDC_IMAGE_PATH, path, sizeof(path));

	//
	//	Prepare OPENFILENAME structure
	//
	memset(&ofn, 0, sizeof(ofn));

	ofn.lStructSize = sizeof(ofn);
	ofn.hwndOwner	= hDlg;
	ofn.lpstrFilter = filter;
	ofn.lpstrFile	= path;
	ofn.nMaxFile	= sizeof(path);
	ofn.lpstrTitle	= title;
	ofn.Flags		= OFN_CREATEPROMPT;

	if (GetFullPathName(path, sizeof(dir), dir, &p) == 0) {
		dir[0] = '\0';
	}

	attr = GetFileAttributes(dir);

	if (attr == INVALID_FILE_ATTRIBUTES) {
		if (GetLastError() == ERROR_FILE_NOT_FOUND) {
			strcpy(path, dir);
		}
		else {
			path[0] = '\0';
		}
	}
	else if (attr & FILE_ATTRIBUTE_DIRECTORY) {
		path[0] = '\0';
		ofn.lpstrInitialDir = dir;
	}
	else {
		strcpy(path, dir);
	}

	len = strlen(path);

	while (len && path[len - 1] == '\\') {
		path[--len] = '\0';
	}

	if (GetOpenFileName(&ofn)) {
		SetDlgItemText(hDlg, IDC_IMAGE_PATH, path);
		CheckDlgButton(hDlg, IDC_IMAGE_READONLY,
			(ofn.Flags & OFN_READONLY) ? BST_CHECKED : BST_UNCHECKED);
	}
}

//
//	Driver path browse button is clicked
//
void OnDriverBrowse(HWND hDlg)
{
	OPENFILENAME ofn;
	TCHAR filter[50];
	TCHAR title[50];
	TCHAR path[MAX_PATH];

	//	get current text in path edit

	GetDlgItemText(hDlg, IDC_DRIVER_PATH, path, sizeof(path));

	//	prepare filter for the open file dialog

	if (LoadString(hAppInstance, IDS_DRIVER_OPEN_FILTER, filter, sizeof(filter))) {
		LPTSTR p = filter;

		while (*p) {
			if (*p == '|') {
				*p = '\0';
			}
			p++;
		}
	}
	else {
		DEBUG_TRACE1(
			"OnDriverBrowse : LoadString - %s", ErrMsg(GetLastError()));

		memcpy(filter, FALLBACK_DRIVER_FILTER, sizeof(FALLBACK_DRIVER_FILTER));
	}

	//	prepare title text for the open file dialog

	if (!LoadString(hAppInstance, IDS_DRIVER_OPEN_TITLE, title, sizeof(title))) {
		DEBUG_TRACE1(
			"OnDriverBrowse : LoadString - %s", ErrMsg(GetLastError()));

		strcpy(title, FALLBACK_DRIVER_TITLE);
	}

	memset(&ofn, 0, sizeof(ofn));

	ofn.lStructSize = sizeof(ofn);
	ofn.hwndOwner	= hDlg;
	ofn.lpstrFilter = filter;
	ofn.lpstrFile	= path;
	ofn.nMaxFile	= sizeof(path);
	ofn.lpstrTitle	= title;
	ofn.Flags		= OFN_FILEMUSTEXIST
					| OFN_NONETWORKBUTTON
					| OFN_HIDEREADONLY
					| OFN_DONTADDTORECENT
					| OFN_FORCESHOWHIDDEN;

	if (GetOpenFileName(&ofn)) {
		SetDlgItemText(hDlg, IDC_DRIVER_PATH, path);
	}
}

//
//	Mount button is clicked
//
void OnImageMount(HWND hDlg, HWND hButton)
{
	DWORD	driver_state;
	DWORD	ret;

	//
	//	ensure that the driver is running
	//
	if ((ret = VfdGetDriverState(&driver_state)) != ERROR_SUCCESS) {
		AppendMessage(hDlg, IDS_ERR_IMAGE_MOUNT, ret);
		return;
	}

	if (driver_state != SERVICE_RUNNING) {
		if (driver_state == VFD_NOT_INSTALLED) {
			if (!DriverInstall(hDlg)) {
				return;
			}
		}

		if (!DriverStart(hDlg)) {
			return;
		}
	}

	//
	//	mount the image
	//
	if (ImageMount(hDlg)) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_IMAGE_UNMOUNT));
	}
}

//
//	Unmount button is clicked
//
void OnImageUnmount(HWND hDlg, HWND hButton)
{
	DWORD ret;

	ret = EnsureUnmount((ENSURE_CALLBACK)Retry_Callback, NULL, (DWORD)hDlg);

	if (ret == ERROR_SUCCESS || ret == ERROR_NOT_READY) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_IMAGE_MOUNT));
	}
	else {
		AppendMessage(hDlg, IDS_ERR_IMAGE_UNMOUNT, ret);
	}
}

//
//	Install button is clicked
//
void OnDriverInstall(HWND hDlg, HWND hButton)
{
	if (DriverInstall(hDlg)) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_DRIVER_START));
	}
}

//
//	Uninstall button is clicked
//
void OnDriverRemove(HWND hDlg, HWND hButton)
{
	DWORD	driver_state;
	DWORD	ret;

	//
	//	Ensure that the driver is stopped
	//
	if ((ret = VfdGetDriverState(&driver_state)) != ERROR_SUCCESS) {
		AppendMessage(hDlg, IDS_ERR_DRIVER_REMOVE, ret);
		return;
	}

	if (driver_state != SERVICE_STOPPED) {
		ret = VfdGetMediaState();

		if (ret == ERROR_SUCCESS || ret == ERROR_WRITE_PROTECT) {
			ret = EnsureUnmount(
				(ENSURE_CALLBACK)Retry_Callback,
				(ENSURE_CALLBACK)AskForceStop,
				(DWORD)hDlg);

			if (ret != ERROR_SUCCESS && ret != ERROR_NOT_READY) {
				AppendMessage(hDlg, IDS_ERR_IMAGE_UNMOUNT, ret);
				return;
			}
		}
		else if (ret != ERROR_NOT_READY) {
			AppendMessage(hDlg, IDS_ERR_DRIVER_STOP, ret);
			return;
		}

		if (!DriverStop(hDlg) && !AskForceRemove(GetParent(hDlg))) {
			return;
		}
	}

	//
	//	remove the driver
	//
	if (DriverRemove(hDlg)) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_DRIVER_INSTALL));
	}
}

//
//	start button is clicked
//
void OnDriverStart(HWND hDlg, HWND hButton)
{
	DWORD	driver_state;
	DWORD	ret;

	//
	//	ensure that the driver is installed
	//
	if ((ret = VfdGetDriverState(&driver_state)) != ERROR_SUCCESS) {
		AppendMessage(hDlg, IDS_ERR_DRIVER_START, ret);
		return;
	}

	if (driver_state == VFD_NOT_INSTALLED) {
		if (!DriverInstall(hDlg)) {
			return;
		}
	}

	//
	//	start the driver
	//
	if (DriverStart(hDlg)) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_DRIVER_STOP));
	}
}

//
//	stop button is clicked
//
void OnDriverStop(HWND hDlg, HWND hButton)
{
	DWORD	ret;

	//
	//	ensure that the drive is empty
	//
	ret = VfdGetMediaState();

	if (ret == ERROR_SUCCESS || ret == ERROR_WRITE_PROTECT) {
		ret = EnsureUnmount(
			(ENSURE_CALLBACK)Retry_Callback,
			(ENSURE_CALLBACK)AskForceStop,
			(DWORD)hDlg);

		if (ret != ERROR_SUCCESS && ret != ERROR_NOT_READY) {
			AppendMessage(hDlg, IDS_ERR_IMAGE_UNMOUNT, ret);
			return;
		}
	}
	else if (ret != ERROR_NOT_READY) {
		AppendMessage(hDlg, IDS_ERR_DRIVER_STOP, ret);
		return;
	}

	//
	//	stop the driver
	//
	if (DriverStop(hDlg)) {
		SendMessage(hButton, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
		SetFocus(GetDlgItem(hDlg, IDC_DRIVER_START));
	}
}

//
//	Mounts an image file
//
BOOL ImageMount(HWND hDlg)
{
	TCHAR	drive_letter[5];
	TCHAR	image_path[MAX_PATH];
	ULONG	image_size;
	BOOL	read_only;
	DWORD	ret;

	//
	//	get dialog data
	//
	if (!GetDlgItemText(hDlg, IDC_IMAGE_PATH, image_path, sizeof(image_path))) {
		ret = GetLastError();
		goto error_exit;
	}

	read_only =	(IsDlgButtonChecked(hDlg, IDC_IMAGE_READONLY) == BST_CHECKED);

	if (IsDlgButtonChecked(hDlg, IDC_IMAGE_720KB) == BST_CHECKED) {
		image_size = VFD_FILESIZE_720KB;
	}
	else if (IsDlgButtonChecked(hDlg, IDC_IMAGE_2P88MB) == BST_CHECKED) {
		image_size = VFD_FILESIZE_2P88MB;
	}
	else {
		image_size = VFD_FILESIZE_1P44MB;
	}

	if (!GetDlgItemText(hDlg, IDC_IMAGE_DRIVE, drive_letter, sizeof(drive_letter))) {
		ret = GetLastError();
		goto error_exit;
	}

	//
	//	assign a drive letter
	//
	if (!isalpha(drive_letter[0])) {
		ret = ERROR_INVALID_DATA;
		goto error_exit;
	}

	if ((ret = VfdSetDriveLetter(drive_letter[0])) != ERROR_SUCCESS) {
		goto error_exit;
	}

	//
	//	mount the image file
	//

	if ((ret = VfdMount(image_path, read_only, image_size)) != ERROR_SUCCESS) {
		goto error_exit;
	}

	return TRUE;

error_exit:
	AppendMessage(hDlg, IDS_ERR_IMAGE_MOUNT, ret);
	return FALSE;
}

//
//	Install the driver
//
BOOL DriverInstall(HWND hDlg)
{
	TCHAR	driver_path[MAX_PATH];
	BOOL	auto_start;
	DWORD	ret;

	//
	//	get dialog data
	//
	if (!GetDlgItemText(hDlg, IDC_DRIVER_PATH, driver_path, sizeof(driver_path))) {
		ret = GetLastError();
		goto error_exit;
	}

	auto_start = (IsDlgButtonChecked(hDlg, IDC_DRIVER_AUTO) == BST_CHECKED);

	//
	//	install the driver
	//
	if ((ret = VfdInstall(driver_path, auto_start)) != ERROR_SUCCESS) {
		goto error_exit;
	}

	return TRUE;

error_exit:
	AppendMessage(hDlg, IDS_ERR_DRIVER_INSTALL, ret);
	return FALSE;
}

//
//	Remove the driver
//
BOOL DriverRemove(HWND hDlg)
{
	DWORD ret;

	ret = VfdRemove();

	if (ret == ERROR_SUCCESS) {
		return TRUE;
	}
	else {
		DEBUG_TRACE1("DriverRemove: VfdRemove - %s", ErrMsg(ret));
		AppendMessage(hDlg, IDS_ERR_DRIVER_REMOVE, ret);
		return FALSE;
	}
}

//
//	start the driver
//
BOOL DriverStart(HWND hDlg)
{
	DWORD ret;

	if ((ret = VfdStart(NULL)) == ERROR_SUCCESS) {
		return TRUE;
	}
	else {
		DEBUG_TRACE1("DriverStart: VfdStart - %s", ErrMsg(ret));
		AppendMessage(hDlg, IDS_ERR_DRIVER_START, ret);
		return FALSE;
	}
}

//
//	stop the driver
//
BOOL DriverStop(HWND hDlg)
{
	DWORD ret;

	ret = VfdStop(NULL);

	if (ret == ERROR_SUCCESS) {
		return TRUE;
	}
	else {
		DEBUG_TRACE1("DriverStop: VfdStop - %s", ErrMsg(ret));
		AppendMessage(hDlg, IDS_ERR_DRIVER_STOP, ret);
		return FALSE;
	}
}

//
//	Refresh the Drive dialog
//
void OnDriveRefresh	(HWND hDlg, WPARAM reason, DWORD driver_state)
{
	BOOL	install, remove, start, stop, mount, unmount;
	DWORD	media_state = ERROR_NOT_READY;
	DWORD	ret;

	//
	//	Get driver and image information
	//
	if (driver_state == 0) {
		ret = VfdGetDriverState(&driver_state);

		if (ret != ERROR_SUCCESS) {
			DEBUG_TRACE1("OnVfdRefresh : VfdGetDriverState - %s", ErrMsg(ret));
			return;
		}
	}

	if (driver_state != VFD_NOT_INSTALLED) {
		TCHAR	driver_path[MAX_PATH];
		DWORD	start_type;

		//	controls for installing will be disabled and
		//	controls for removing will be enabled
		install	= FALSE;
		remove	= TRUE;

		//	get driver configuration
		ret = VfdGetDriverConfig(driver_path, &start_type);

		if (ret == ERROR_SUCCESS) {
			//	set driver path edit
			SetDlgItemText(hDlg, IDC_DRIVER_PATH, driver_path);

			//	set start type radio button
			switch (start_type) {
			case SERVICE_AUTO_START:
				CheckRadioButton(hDlg, IDC_DRIVER_MANUAL, IDC_DRIVER_AUTO, IDC_DRIVER_AUTO);
				break;

			case SERVICE_DEMAND_START:
				CheckRadioButton(hDlg, IDC_DRIVER_MANUAL, IDC_DRIVER_AUTO, IDC_DRIVER_MANUAL);
				break;
			}
		}
		else {
			DEBUG_TRACE1("OnDriveRefresh: VfdGetDriverConfig - %s", ErrMsg(ret));
		}

		if (driver_state == SERVICE_RUNNING) {
			//	start control will be disabled and stop control will be enabled
			start	= FALSE;
			stop	= TRUE;

			//	get current media state
			media_state = VfdGetMediaState();

			if (media_state == ERROR_SUCCESS || media_state == ERROR_WRITE_PROTECT) {
				TCHAR	drive_letter[]	= " :";
				TCHAR	image_path[MAX_PATH];
				ULONG	image_size;
				BOOL	read_only;

				//	controls for mounting will be disabled and
				//	controls for unmounting will be enabled
				mount	= FALSE;
				unmount = TRUE;

				//	get current drive letter
				ret = VfdGetDriveLetter(&drive_letter[0]);

				if (ret == ERROR_SUCCESS) {
					//	set drive letter combo
					SendDlgItemMessage(hDlg, IDC_IMAGE_DRIVE,
						CB_SELECTSTRING, 0, (LPARAM)drive_letter);
				}
				else {
					DEBUG_TRACE1("OnDriveRefresh: VfdGetDriveLetter - %s", ErrMsg(ret));
				}

				//	get current image info
				ret = VfdGetFileInfo(image_path, &image_size, &read_only);

				if (ret == ERROR_SUCCESS) {
					//	set image path edit
					SetDlgItemText(hDlg, IDC_IMAGE_PATH, image_path);

					//	set read-only check button
					CheckDlgButton(hDlg, IDC_IMAGE_READONLY, read_only ? BST_CHECKED : BST_UNCHECKED);

					//	set image size radio button
					switch(image_size) {
					case VFD_FILESIZE_720KB:
						CheckRadioButton(hDlg, IDC_IMAGE_720KB, IDC_IMAGE_2P88MB, IDC_IMAGE_720KB);
						break;

					case VFD_FILESIZE_1P44MB:
						CheckRadioButton(hDlg, IDC_IMAGE_720KB, IDC_IMAGE_2P88MB, IDC_IMAGE_1P44MB);
						break;

					case VFD_FILESIZE_2P88MB:
						CheckRadioButton(hDlg, IDC_IMAGE_720KB, IDC_IMAGE_2P88MB, IDC_IMAGE_2P88MB);
						break;
					}
				}
				else {
					DEBUG_TRACE1("OnDriveRefresh: VfdGetFileInfo - %s", ErrMsg(ret));
				}

			}
			else {
				mount	= TRUE;
				unmount	= FALSE;
			}
		}
		else if (driver_state == SERVICE_STOPPED ||
			driver_state == SERVICE_PAUSED) {
			//	The driver is not running and can be started
			start	= TRUE;
			stop	= FALSE;
			mount	= TRUE;
			unmount	= FALSE;
		}
		else {
			//	The driver is in pending state - cannot be started nor stopped
			start	= FALSE;
			stop	= FALSE;
			mount	= FALSE;
			unmount	= FALSE;
		}
	}
	else {
		//	The driver is not installed
		install	= TRUE;
		remove	= FALSE;
		start	= TRUE;
		stop	= FALSE;
		mount	= TRUE;
		unmount	= FALSE;
	}

	//
	//	enable / disable controls according to current state
	//
	SendDlgItemMessage(hDlg, IDC_IMAGE_PATH, EM_SETREADONLY, !mount, 0);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_BROWSE),	mount);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_READONLY),	mount);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_720KB),		mount);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_1P44MB),	mount);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_2P88MB),	mount);
	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_DRIVE),		mount);

	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_MOUNT),
		mount && GetWindowTextLength(GetDlgItem(hDlg, IDC_IMAGE_PATH)));

	EnableWindow(GetDlgItem(hDlg, IDC_IMAGE_UNMOUNT),	unmount);

	SendDlgItemMessage(hDlg, IDC_DRIVER_PATH, EM_SETREADONLY, !install, 0);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_BROWSE),	install);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_MANUAL),	install);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_AUTO),		install);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_INSTALL),
		install && GetWindowTextLength(GetDlgItem(hDlg, IDC_DRIVER_PATH)));

	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_START),	start);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_STOP),		stop);
	EnableWindow(GetDlgItem(hDlg, IDC_DRIVER_REMOVE),	remove);

	//
	//	Show reason of refresh
	//
	if (reason) {
		UINT	msg = 0;

		switch (reason) {
		case VFD_OPERATION_INSTALL:
			if (driver_state != VFD_NOT_INSTALLED) {
				msg = IDS_MSG_DRIVER_INSTALLED;
			}
			break;

		case VFD_OPERATION_REMOVE:
			if (driver_state == VFD_NOT_INSTALLED) {
				msg = IDS_MSG_DRIVER_REMOVED;
			}
			break;

		case VFD_OPERATION_START:
			if (driver_state == SERVICE_RUNNING) {
				msg = IDS_MSG_DRIVER_STARTED;
			}
			break;

		case VFD_OPERATION_STOP:
			if (driver_state == SERVICE_STOPPED || driver_state == VFD_NOT_INSTALLED) {
				msg = IDS_MSG_DRIVER_STOPPED;
			}
			break;

		case VFD_OPERATION_MOUNT:
			if (media_state == ERROR_SUCCESS || media_state == ERROR_WRITE_PROTECT) {
				msg = IDS_MSG_IMAGE_MOUNTED;
			}
			break;

		case VFD_OPERATION_UMOUNT:
			if (media_state != ERROR_SUCCESS && media_state != ERROR_WRITE_PROTECT) {
				msg = IDS_MSG_IMAGE_UNMOUNTED;
			}
			break;
		}

		if (msg) {
			AppendMessage(hDlg, msg, 0);
		}
	}

	return;
}

//	Ask if proceed to stop the driver when the current image
//	cannot be unmounted

BOOL AskForceStop(HWND hWnd)
{
	TCHAR msg[200];

	if (!LoadString(hAppInstance, IDS_MSG_FORCE_STOP, msg, sizeof(msg))) {
		DEBUG_TRACE1(
			"AskForceStop: LoadString(IDS_MSG_FORCE_STOP) - %s",
			ErrMsg(GetLastError()));

		strcpy(msg, FALLBACK_FORCE_STOP);
	}

	return (MessageBox(hWnd, msg, VFD_APP_NAME, MB_ICONQUESTION | MB_YESNO) == IDYES);
}

//	Ask if proceed to remove the driver when the driver cannot be stopped

BOOL AskForceRemove(HWND hDlg)
{
	TCHAR msg[200];

	if (!LoadString(hAppInstance, IDS_MSG_FORCE_REMOVE, msg, sizeof(msg))) {
		DEBUG_TRACE1(
			"AskForceRemove: LoadString(IDS_MSG_FORCE_REMOVE) - %s",
			ErrMsg(GetLastError()));

		strcpy(msg, FALLBACK_FORCE_REMOVE);
	}

	return (MessageBox(hDlg, msg, VFD_APP_NAME, MB_ICONQUESTION | MB_YESNO) == IDYES);
}

//
//	append a text string to the message control
//
void AppendMessage(HWND hDlg, UINT msg, DWORD err)
{
	static TCHAR buf[512];
	HWND	hEdit;
	int		len;
	int		line;
	int		top;

	len = strlen(buf);

	if (len >= sizeof(buf) - 1) {
		return;
	}

	if (msg) {
		len += LoadString(hAppInstance, msg, buf + len, sizeof(buf) - len);

		if (len < sizeof(buf) - 3) {
			strcat(buf, "\r\n");
			len += 2;
		}
	}

	if (err) {
		len += VfdErrorMessage(err, buf + len, sizeof(buf) - len);
	}

	if ((hEdit = GetDlgItem(hDlg, IDC_DRIVER_MESSAGE)) == NULL) {
		DEBUG_TRACE1(
			"AppendMessage : GetDlgItem - %s", ErrMsg(GetLastError()));
		return;
	}

	if (!SetWindowText(hEdit, buf)) {
		DEBUG_TRACE1(
			"AppendMessage : SetWindowText - %s", ErrMsg(GetLastError()));
		return;
	}

	SendMessage(hEdit, EM_SETSEL, 0, -1);
	SendMessage(hEdit, EM_SCROLLCARET, 0, 0);

	line = SendMessage(hEdit, EM_GETFIRSTVISIBLELINE, 0, 0);
	top = SendMessage(hEdit, EM_LINEINDEX, line, 0);

	if (top) {
		len -= top;
		memmove(buf, buf + top, len + 1);
	}
}

