/*
	assocdlg.c

	Virtual Floppy Disk drive control panel
    Copyright (C) 2003 Kenji Kato
*/

#include "sysincl.h"
#include "vfdutil.h"
#include "vfdwin.h"
#include "resource.h"
#include "registry.h"

//
//	Window message handlers
//
static void OnAssocInitDialog(HWND hDlg);
static void OnAssocNew(HWND hDlg, HWND hBtn);
static void OnAssocCheckAll(HWND hDlg, HWND hBtn, int check);
static void OnAssocApply(HWND hDlg, HWND hBtn);
static void OnAssocListNotify(HWND hDlg, LPNMHDR pNMHDR);

//
//	local functions
//
static void UpdateList(HWND hDlg);
static int	AddItem(HWND hList, char *ext, char *prog);
static void ToggleItem(HWND hList, int iItem);

//
//	Add extension dialog
//
static BOOL CALLBACK AddExtProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);
void OnAddExtEdit(HWND hDlg, HWND hEdit);

//
//	Filetype used for registering file association
//
#define REG_FILETYPE_PREFIX	"VfdWin"

//
//	dialog state variables
//
static int	nCheckedItems = 0;
static int	nChangedItems = 0;
static BOOL	bRegWritable = FALSE;

//
//	window massage dispatcher
//
BOOL CALLBACK AssocProc(
	HWND hDlg,
	UINT msg,
	WPARAM wParam,
	LPARAM lParam) 
{ 
    switch (msg) { 
	case WM_INITDIALOG:
		OnAssocInitDialog(hDlg);
		break;

	case WM_ERASEBKGND:
		// somehow list control is not drawn correctly when this dialog is
		// brought forward if the control has input focus, so force redrawing it
		RedrawWindow(hDlg, NULL, NULL, RDW_INVALIDATE);
		break;

	case WM_COMMAND:
		switch(wParam) {
		case IDC_ASSOC_NEW:
			OnAssocNew(hDlg, (HWND)lParam);
			break;

		case IDC_ASSOC_ALL:
			OnAssocCheckAll(hDlg, (HWND)lParam, 1);
			break;

		case IDC_ASSOC_CLEAR:
			OnAssocCheckAll(hDlg, (HWND)lParam, 0);
			break;

		case IDC_ASSOC_APPLY:
			OnAssocApply(hDlg, (HWND)lParam);
			break;
		}
		break;

	case WM_NOTIFY:
		switch (wParam) {
		case IDC_ASSOC_LIST:
			OnAssocListNotify(hDlg, (LPNMHDR)lParam);
			break;
		}
		break;
    }

    return 0; 
} 

//
//	Association dialog initialize
//
void OnAssocInitDialog(HWND hDlg)
{
	HWND hList;
	HIMAGELIST hImage;
	LVCOLUMN column;
	RECT rect;
	char buf[50];

	//
	//	initialize associatoin list view
	//
	hList = GetDlgItem(hDlg, IDC_ASSOC_LIST);

	if (hList == NULL) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnAssocInitDialog : GetDlgItem - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	if (!GetClientRect(hList, &rect)) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnAssocInitDialog : GetClientRect - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	//	insert columns 

	memset(&column, 0, sizeof(column));

	column.mask = LVCF_SUBITEM | LVCF_TEXT | LVCF_WIDTH;
	column.pszText = buf;

	//	first column -- extension

	if (!LoadString(hAppInstance, IDS_ASSOC_HDR_EXT, buf, sizeof(buf))) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnAssocInitDialog : LoadString - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	column.cx = (rect.right - rect.left) / 3;
	column.iSubItem = 0;
	
	if (ListView_InsertColumn(hList, 0, &column) == -1) {
		DWORD err = GetLastError();

		DEBUG_TRACE1( "OnAssocInitDialog : ListView_InsertColumn - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}


	//	second column -- associated program

	if (!LoadString(hAppInstance, IDS_ASSOC_HDR_PGM, buf, sizeof(buf))) {
		DWORD err = GetLastError();

		DEBUG_TRACE1( "OnAssocInitDialog : LoadString - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	column.cx = column.cx * 2 - GetSystemMetrics(SM_CXVSCROLL);
	column.iSubItem = 1;
	
	if (ListView_InsertColumn(hList, 1, &column) == -1) {
		DWORD err = GetLastError();

		DEBUG_TRACE1( "OnAssocInitDialog : ListView_InsertColumn - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	//	set image list for the list control

	hImage = ImageList_LoadBitmap(
		hAppInstance, MAKEINTRESOURCE(IDB_CHECKBOX), 16, 1, RGB(255, 0, 255));

	if (hImage == NULL) {
		DWORD err = GetLastError();

		DEBUG_TRACE1( "OnAssocInitDialog : ImageList_LoadBitmap - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}

	SetLastError(ERROR_SUCCESS);

	hImage = ListView_SetImageList(hList, hImage, LVSIL_SMALL);

	if (hImage == NULL && GetLastError() != ERROR_SUCCESS) {
		DWORD err = GetLastError();

		DEBUG_TRACE1("OnAssocInitDialog : ListView_SetImageList - %s", ErrMsg(err));

		ShowErrorMessage(err, IDS_ERR_APP_INIT);

		return;
	}
	
	//	read current association from registry

	UpdateList(hDlg);

	//	set focus on the first item

	ListView_SetItemState(hList, 0,
		LVIS_SELECTED|LVIS_FOCUSED,
		LVIS_SELECTED|LVIS_FOCUSED);

	return;
}

//
//	read current association from registry and put them into list control
//
void UpdateList(HWND hDlg)
{
	HKEY	hKey;
	char	ext[MAX_PATH], prog[MAX_PATH];
	LONG	ret;
	HWND	hList;
	DWORD	index;

	//
	//	Reset list status
	//
	nCheckedItems = 0;
	nChangedItems = 0;

	hList = GetDlgItem(hDlg, IDC_ASSOC_LIST);

	if (hList == NULL) {
		DEBUG_TRACE1("UpdateList : GetDlgItem - %s", ErrMsg(GetLastError()));

		return;
	}

	//
	//	Open registry root key
	//

	ret = RegOpenKeyEx(HKEY_CLASSES_ROOT, NULL, 0, KEY_ALL_ACCESS, &hKey);

	if (ret == ERROR_SUCCESS) {
		bRegWritable = TRUE;
	}
	else {
		//	Failed to open registry in read-write mode

		char msg[MAX_PATH];
		HWND hMsg;

		DEBUG_TRACE1("UpdateList : RegOpenKeyEx(HKCR) - %s", ErrMsg(GetLastError()));

		bRegWritable = FALSE;

		//	Hide controls for manipulating association

		ShowWindow(GetDlgItem(hDlg, IDC_ASSOC_NEW), SW_HIDE);
		ShowWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), SW_HIDE);
		ShowWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), SW_HIDE);
		ShowWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), SW_HIDE);

		//	open registry in read-only mode

		ret = RegOpenKeyEx(HKEY_CLASSES_ROOT, NULL, 0, KEY_READ, &hKey);

		if (ret == ERROR_SUCCESS) {
			// succeeded to open read-only

			if (!LoadString(hAppInstance, IDS_ASSOC_CANT_WRITE, msg, sizeof(msg))) {
				DEBUG_TRACE1("UpdateList : LoadString - %s", ErrMsg(GetLastError()));

				// in case LoadString has failed
				strcpy(msg, "Registry is read-only");
			}
		}
		else {
			// failed even to open read-only

			DEBUG_TRACE1(
				"UpdateList : RegOpenKeyEx - %s", ErrMsg(ret));

			EnableWindow(hList, FALSE);

			if (!LoadString(hAppInstance, IDS_ASSOC_CANT_READ, msg, sizeof(msg))) {
				DEBUG_TRACE1(
					"UpdateList : LoadString - %s", ErrMsg(GetLastError()));

				// in case LoadString has failed
				strcpy(msg, "Can't read registry");
			}
		}

		// show why users can't modify association

		hMsg = GetDlgItem(hDlg, IDC_ASSOC_MESSAGE);

		if (hMsg == NULL) {
			DEBUG_TRACE1(
				"UpdateList : GetDlgItem - %s", ErrMsg(GetLastError()));
			return;
		}

		if (!SetWindowText(hMsg, msg)) {
			DEBUG_TRACE1(
				"UpdateList : SetWindowText - %s", ErrMsg(GetLastError()));
			return;
		}

		ShowWindow(hMsg, SW_SHOW);

		if (ret != ERROR_SUCCESS) {

			// cannot even to read registry

			return;
		}
	}

	//
	//	Add common floppy image extensions to list view
	//

	if ((ret = GetAssociatedProgram(".bin", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".bin", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".bin\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".dat", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".dat", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".dat\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".fdd", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".fdd", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".fdd\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".flp", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".flp", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".flp\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".img", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".img", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".img\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".ima", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".ima", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".ima\") - %s", ErrMsg(ret));
	}

	if ((ret = GetAssociatedProgram(".vfd", prog)) == ERROR_SUCCESS) {
		AddItem(hList, ".vfd", prog);
	}
	else {
		DEBUG_TRACE1(
			"UpdateList : GetAssociatedProgram(\".vfd\") - %s", ErrMsg(ret));
	}

	//
	//	Search registry for other extensions associated with VfdWin
	//

	index = 0;

	for (;;) {
		DWORD size = sizeof(ext);

		ret = RegEnumKeyEx(hKey, index++, ext, &size, NULL, NULL, NULL, NULL);

		if (ret != ERROR_SUCCESS) {
			if (ret != ERROR_NO_MORE_ITEMS) {
				DEBUG_TRACE1(
					"UpdateList : RegEnumKeyEx - %s", ErrMsg(ret));
			}
			break;
		}

		if (ext[0] != '.') {
			// not an extension - try next item
			continue;
		}

		ext[size] = '\0';

		if ((ret = GetAssociatedProgram(ext, prog)) == ERROR_SUCCESS) {
			if (strnicmp(prog, VFD_APP_FILENAME, sizeof(VFD_APP_FILENAME) - 1) == 0) {
				AddItem(hList, ext, prog);
			}
		}
		else {
			DEBUG_TRACE2(
				"UpdateList : GetAssociatedProgram(\"%s\") - %s", ext, ret);
		}
	}

	RegCloseKey(hKey);

	// Enable / Disable the "Check All" and "Clear All" button
	// according to the number of checked items

	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), (nCheckedItems < ListView_GetItemCount(hList)));
}

//
//	Insert an item to the list constrol
//
int AddItem(HWND hList, char *ext, char *prog)
{
	LVITEM item;

	memset(&item, 0, sizeof(item));

	//	make sure that extension is in lower case
	{
		char *p = ext;

		while (*p) {
			*p = (char)tolower(*p);
			p++;
		}
	}

	if (*prog == '\0') {
		if (!LoadString(hAppInstance, IDS_ASSOC_NONE, prog, MAX_PATH)) {
			// in case LoadString failed

			DEBUG_TRACE1(
				"AddItem : LoadString - %s", ErrMsg(GetLastError()));

			strcpy(prog, "(none)");
		}
	}

	//	select check status

	if (strnicmp(prog, VFD_APP_FILENAME, sizeof(VFD_APP_FILENAME) - 1) == 0) {
		//	use checked icon
		item.iImage = 1;

		//	use "Virtual FD" if program is "vfdwin.exe"
		strcpy(prog, VFD_APP_NAME);
	}

	//	if registry is read-only, use grayed check icons

	if (!bRegWritable) {
		item.iImage += 2;
	}

	//	store initial icon state in lParam

	item.lParam = item.iImage;

	//	find if extension is already in list
	{
		LVFINDINFO find;

		memset(&find, 0, sizeof(find));
		find.flags = LVFI_STRING;
		find.psz = ext;

		item.iItem = ListView_FindItem(hList, -1, &find);
	}

	//
	//	insert an item or update check status
	//

	if (item.iItem == -1) {

		// add a new item

		item.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM;

		item.iItem = ListView_GetItemCount(hList);
		item.pszText = ext;

		item.iItem = ListView_InsertItem(hList, &item);

		if (item.iItem == -1) {
			DEBUG_TRACE1(
				"AddItem : ListView_InsertItem - %s", ErrMsg(GetLastError()));
		}
	}
	else {

		// update icon index

		item.mask = LVIF_IMAGE | LVIF_PARAM;

		if (!ListView_SetItem(hList, &item)) {
			DEBUG_TRACE1(
				"AddItem : ListView_SetItem - %s", ErrMsg(GetLastError()));
		}
	}

	if (item.iItem >= 0) {

		// set program name

		ListView_SetItemText(hList, item.iItem, 1, prog);

		// count items with 'checked' icon

		if (item.iImage == 1) {
			nCheckedItems++;
		}
	}

	return item.iItem;
}

//
//	handles list control notification messages
//	NM_CLICK	- an item is clicked
//	LVN_KEYDOWN	- a key is pressed
//
void OnAssocListNotify(HWND hDlg, LPNMHDR pNMHDR)
{
	// if registry is read-only, do nothing

	if (!bRegWritable) {
		return;
	}

	switch (pNMHDR->code) {
	case NM_CLICK:
		{
			LVHITTESTINFO hit;

			memset(&hit, 0, sizeof(hit));

			if (!GetCursorPos(&hit.pt)) {
				DEBUG_TRACE1(
					"OnAssocListNotify : GetCursorPos - %s", ErrMsg(GetLastError()));
				break;
			}

			if (!ScreenToClient(pNMHDR->hwndFrom, &hit.pt)) {
				DEBUG_TRACE1(
					"OnAssocListNotify : ScreenToClient - %s", ErrMsg(GetLastError()));
				break;
			}

			ListView_HitTest(pNMHDR->hwndFrom, &hit);

			if (hit.iItem >= 0 && (hit.flags & LVHT_ONITEMICON)) {

				// icon of an item is clicked

				ToggleItem(pNMHDR->hwndFrom, hit.iItem);

				//	enable/disable 'clear' and 'apply' buttons

				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), (nCheckedItems < ListView_GetItemCount(pNMHDR->hwndFrom)));
				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), nChangedItems);
			}
		}
		break;

	case LVN_KEYDOWN:
		if (((LPNMLVKEYDOWN)pNMHDR)->wVKey == VK_SPACE) {
			int item;

			item = ListView_GetNextItem(pNMHDR->hwndFrom, -1, LVNI_SELECTED);

			if (item >= 0) {

				// space key is pressed while the focus on an item

				ToggleItem(pNMHDR->hwndFrom, item);

				//	enable/disable 'clear' and 'apply' buttons

				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), (nCheckedItems < ListView_GetItemCount(pNMHDR->hwndFrom)));
				EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), nChangedItems);
			}
		}
		break;
	}

	return;
}

//
//	switch an item's checked state
//
void ToggleItem(HWND hList, int iItem)
{
	LVITEM item;

	memset(&item, 0, sizeof(item));

	item.mask = LVIF_IMAGE | LVIF_PARAM;
	item.iItem = iItem;
	
	if (!ListView_GetItem(hList, &item)) {
		DEBUG_TRACE1(
			"ToggleItem : ListView_GetItem - %s", ErrMsg(GetLastError()));

		return;
	}

	if (item.iImage == 0 || item.iImage == 1) {

		//	deal only with 'checked' or 'unchecked'
		//	ignore grayed icons

		//	is current state the same as the original state?

		if (item.iImage == item.lParam) {
			nChangedItems++;
		}
		else {
			nChangedItems--;
		}

		//	change icon index

		if (item.iImage == 0) {
			nCheckedItems++;
			item.iImage = 1;
		}
		else if (item.iImage == 1) {
			nCheckedItems--;
			item.iImage = 0;
		}

		if (!ListView_SetItem(hList, &item)) {
			DEBUG_TRACE1(
				"ToggleItem : ListView_SetItem - %s", ErrMsg(GetLastError()));
		}
	}

	return;
}

//
//	New button is clicked
//
void OnAssocNew(HWND hDlg, HWND hBtn)
{
	HWND hList;
	char ext[MAX_PATH], prog[MAX_PATH];
	LVITEM item;
	DWORD ret;

	if (!bRegWritable) {
		return;
	}

	if (DialogBoxParam(
		hAppInstance,
		MAKEINTRESOURCE(IDD_NEWEXT),
		hDlg,
		AddExtProc,
		(LPARAM)(ext + 1)) <= 0)
	{
		DEBUG_TRACE1(
			"OnAssocNew : DialogBoxParam - %s", ErrMsg(GetLastError()));
		return;
	}

	hList = GetDlgItem(hDlg, IDC_ASSOC_LIST);

	if (hList == NULL) {
		DEBUG_TRACE1(
			"OnAssocNew : GetDlgItem - %s", ErrMsg(GetLastError()));
		return;
	}

	ext[0] = '.';
	
	memset(&item, 0, sizeof(item));

	ret = GetAssociatedProgram(ext[1] == '.' ? &ext[1] : ext, prog);

	if (ret != ERROR_SUCCESS) {
		DEBUG_TRACE2(
			"OnAssocListNotify : GetAssociatedProgram(\"%s\") - %s",
			ext[1] == '.' ? &ext[1] : ext, ret);
	}

	//	Insert an item to the list control

	item.iItem = AddItem(hList, ext[1] == '.' ? &ext[1] : ext, prog);

	if (item.iItem == -1) {
		return;
	}

	//	Get the new image's check state

	item.mask = LVIF_IMAGE;

	if (!ListView_GetItem(hList, &item)) {
		DEBUG_TRACE1(
			"OnAssocListNotify : ListView_GetItem - %s", ErrMsg(GetLastError()));
		return;
	}

	//	if the item is not checked, check it

	if (item.iImage == 0) {
		item.iImage = 1;
		
		if (!ListView_SetItem(hList, &item)) {
			DEBUG_TRACE1(
				"OnAssocListNotify : ListView_SetItem - %s", ErrMsg(GetLastError()));
			return;
		}

		nCheckedItems++;
		nChangedItems++;
	}

	//	enable / disable 'clear' and 'apply' button

	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), nCheckedItems < ListView_GetItemCount(hList));
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), nChangedItems);

	SendMessage(hBtn, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);

	return;
}

//
//	check all / clear all button is clicked
//
void OnAssocCheckAll(HWND hDlg, HWND hBtn, int check)
{
	HWND hList;
	LVITEM item;
	
	if (!bRegWritable) {
		return;
	}

	hList = GetDlgItem(hDlg, IDC_ASSOC_LIST);

	if (hList == NULL) {
		DEBUG_TRACE1(
			"OnAssocClear : GetDlgItem - %s", ErrMsg(GetLastError()));

		return;
	}

	nChangedItems = 0;

	memset(&item, 0, sizeof(item));
	item.iItem = -1;

	while ((item.iItem = ListView_GetNextItem(hList, item.iItem, LVNI_ALL)) != -1) {
		item.mask = LVIF_PARAM;
		
		if (!ListView_GetItem(hList, &item)) {
			DEBUG_TRACE1(
				"OnAssocClear : ListView_GetItem - %s", ErrMsg(GetLastError()));
			break;
		}

		if (item.lParam != check) {
			nChangedItems++;
		}

		item.mask = LVIF_IMAGE;
		item.iImage = check;
		
		if (!ListView_SetItem(hList, &item)) {
			DEBUG_TRACE1(
				"OnAssocClear : ListView_SetItem - %s", ErrMsg(GetLastError()));
			break;
		}
	}

	nCheckedItems = check ? ListView_GetItemCount(hList) : 0;

	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), (nCheckedItems < ListView_GetItemCount(hList)));
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), nChangedItems);

	SetFocus(GetDlgItem(hDlg, IDC_ASSOC_LIST));

	SendMessage(hBtn, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);

	return;
}

//
//	apply button is clicked
//
void OnAssocApply(HWND hDlg, HWND hBtn)
{
	HWND hList;
	LVITEM item;
	char app_path[MAX_PATH], type_desc[MAX_PATH], verb_desc[MAX_PATH];
	char ext[MAX_PATH], prog[MAX_PATH];

	if (!bRegWritable) {
		return;
	}

	hList = GetDlgItem(hDlg, IDC_ASSOC_LIST);

	if (hList == NULL) {
		DEBUG_TRACE1(
			"OnAssocApply : GetDlgItem - %s", ErrMsg(GetLastError()));

		return;
	}

	if (!GetModuleFileName(hAppInstance, app_path, sizeof(app_path))) {
		DEBUG_TRACE1(
			"OnAssocApply : GetModuleFileName - %s", ErrMsg(GetLastError()));

		return;
	}

	if (!LoadString(hAppInstance, IDS_ASSOC_FILETYPE_DESC, type_desc, sizeof(type_desc))) {
		DEBUG_TRACE1(
			"OnAssocApply : LoadString - %s", ErrMsg(GetLastError()));

		type_desc[0] = '\0';

		// this is not fatal
	}

	if (!LoadString(hAppInstance, IDS_ASSOC_FILETYPE_VERB, verb_desc, sizeof(verb_desc))) {
		DEBUG_TRACE1(
			"OnAssocApply : LoadString - %s", ErrMsg(GetLastError()));

		verb_desc[0] = '\0';

		// this also is not fatal
	}

	nCheckedItems = 0;
	nChangedItems = 0;

	memset(&item, 0, sizeof(item));
	item.mask = LVIF_TEXT | LVIF_IMAGE | LVIF_PARAM;
	item.iItem = -1;
	item.pszText = ext;
	item.cchTextMax = sizeof(ext);

	while ((item.iItem = ListView_GetNextItem(hList, item.iItem, LVNI_ALL)) != -1) {
		DWORD ret;

		if (!ListView_GetItem(hList, &item)) {
			DEBUG_TRACE1(
				"OnAssocApply : ListView_GetItem - %s", ErrMsg(GetLastError()));
			break;
		}

		if (item.iImage == 0 && item.lParam == 1) {
			// initially checked, but currently unchecked

			ret = RestoreAssociation(ext, REG_FILETYPE_PREFIX);

			if (ret != ERROR_SUCCESS) {
				DEBUG_TRACE2(
					"OnAssocApply : RemoveAssociation(\"%s\") - %s",
					ext, ret);
				break;
			}
		}
		else if (item.iImage == 1 && item.lParam == 0) {
			// initially unchecked, and currently checked

			ret = AddAssociation(ext,
				REG_FILETYPE_PREFIX,
				type_desc,
				REG_FILETYPE_PREFIX,
				verb_desc,
				app_path,
				VFD_ICON_IDX_IMAGE);

			if (ret != ERROR_SUCCESS) {
				DEBUG_TRACE2(
					"OnAssocApply : AddAssociation(\"%s\") - %s",
					ext, ret);
				break;
			}
		}

		if ((ret = GetAssociatedProgram(ext, prog)) == ERROR_SUCCESS) {
			AddItem(hList, ext, prog);
		}
		else {
			DEBUG_TRACE2(
				"OnAssocApply : GetAssociatedProgram(\"%s\") - %s",
				ext, ret);
		}
	}

	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_CLEAR), nCheckedItems);
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_ALL), (nCheckedItems < ListView_GetItemCount(hList)));
	EnableWindow(GetDlgItem(hDlg, IDC_ASSOC_APPLY), nChangedItems);

	SetFocus(GetDlgItem(hDlg, IDC_ASSOC_LIST));

	SendMessage(hBtn, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);

	return;
}

//
//	Add extension dialog procedure
//
BOOL CALLBACK AddExtProc(
	HWND hDlg,
	UINT msg,
	WPARAM wParam,
	LPARAM lParam)
{
	static char *sExtension = NULL;

	switch (msg) {
	case WM_INITDIALOG:
		sExtension = (char *)lParam;
		break;

	case WM_COMMAND:
		switch (wParam) {
		case IDOK:
			EndDialog(hDlg, 
				GetDlgItemText(hDlg, IDC_EXTENSION, sExtension, MAX_PATH - 1));
			return TRUE;

		case IDCANCEL:
			EndDialog(hDlg, 0);
			return TRUE;

		case MAKELONG(IDC_EXTENSION, EN_CHANGE):
			OnAddExtEdit(hDlg, (HWND)lParam);
			break;
		}
	}
	return 0;
}

//
//	text in the extention edit box has been changed
//
void OnAddExtEdit(HWND hDlg, HWND hEdit)
{
	char buf[MAX_PATH], *p;
	BOOL valid = FALSE;

	if (!GetWindowText(hEdit, buf, sizeof(buf))) {
		if (GetLastError() != ERROR_SUCCESS) {
			DEBUG_TRACE1(
				"OnAssocEditChange : GetWindowText - %s", ErrMsg(GetLastError()));

			return;
		}
	}

	p = buf;

	if (*p == '.') {
		p++;
	}

	while (*p) {
		if (strchr(".\\/:,;*?\"<>| ", *p)) {
			valid = FALSE;
			break;
		}
		else {
			valid = TRUE;
		}
		p++;
	}

	//	enable 'OK' button if text is valid for an extension

	EnableWindow(GetDlgItem(hDlg, IDOK), valid);

	return;
}

// End of File
