/*
	vfdctl.c

	Virtual Floppy Disk driver control routines
	Copyright (C) 2003 Kenji Kato
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#pragma warning(push, 3)
#include <winioctl.h>
#pragma warning(pop)

#define _CRTDBG_MAP_ALLOC
#include <stdlib.h>
#include <crtdbg.h>
#include <stdio.h>

#include "vfdctl.h"
#include "vfdutil.h"

//
//	DOS device name (\\.\VirtualFD)
//
#define VFD_DEVICE_DOSNAME		"\\\\.\\"	VFD_DEVICE_BASENAME

//
//	local functions
//
static void VfdNotify(WPARAM operation, LPARAM param);
static HANDLE VfdOpenDevice();

#ifdef DLL_EXPORTS
//
//	handle to this module instance
//
static HINSTANCE DLL_INSTANCE = NULL;

//
//	DLL entry point
//
BOOL APIENTRY DllMain(
	HANDLE hModule,
	DWORD  reason,
	LPVOID lpReserved)
{
	UNREFERENCED_PARAMETER(lpReserved);

	if (reason == DLL_PROCESS_ATTACH) {
		//	save this DLL's module instance handle

		DLL_INSTANCE = (HINSTANCE)hModule;
	}

	return TRUE;
}
#else	// DLL_EXPORTS
#define DLL_INSTANCE	NULL
#endif	// DLL_EXPORTS

//
//	broadcast state change to all the top level windows
//
void VfdNotify(WPARAM operation, LPARAM param)
{
	static UINT uVfdMsg = 0;

	if (uVfdMsg == 0) {
		//	register VFD message

		uVfdMsg = RegisterWindowMessage(VFD_BROADCAST_MSG_STR);
	}

	//	send/post the broadcast message

	SendNotifyMessage(HWND_BROADCAST, uVfdMsg, operation, param);
}

//
//	open the Virtual FD device without showing the "Insert Floppy"
//	dialog when the drive is empty.
//
HANDLE VfdOpenDevice()
{
	HANDLE hDevice = NULL;

	// change error mode in order to avoid "Insert Floppy" dialog

	UINT err_mode = SetErrorMode(SEM_FAILCRITICALERRORS);

	//	open the Virtual FD device

	hDevice = CreateFile(
		VFD_DEVICE_DOSNAME,
		GENERIC_READ | GENERIC_WRITE,
		FILE_SHARE_READ | FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		FILE_FLAG_NO_BUFFERING,
		NULL);

	//	revert to the previous error mode

	SetErrorMode(err_mode);

	//	return the handle to the Virtual FD device

	return hDevice;
}

//
//	Install Virtual Floppy Driver dynamically
//
DWORD VfdInstall(
	LPCTSTR		driver_path,
	BOOL		auto_start)
{
	SC_HANDLE	hScManager;				// Service Control Manager
	SC_HANDLE	hService;				// Service (= Driver)
	TCHAR		full_path[MAX_PATH];
	DWORD		ret = ERROR_SUCCESS;

	//	Prepare driver binary's full path

	if (driver_path == NULL || *driver_path == '\0') {

		// default driver file is vfd.sys in the same directory as executable

		DWORD len = GetModuleFileName(
			NULL, full_path, sizeof(full_path));

		if (len == 0) {
			ret = GetLastError();

			DEBUG_TRACE1("VfdInstall: GetModuleFileName - %s",
				ErrMsg(ret));

			return ret;
		}

		//	search the last '\' char

		while (len > 0 && full_path[len - 1] != '\\') {
			len --;
		}

		strcpy(&full_path[len], "vfd.sys");
	}
	else {
		//	ensure that tha path is absolute full path

		LPTSTR file_part;

		if (GetFullPathName(
			driver_path,
			sizeof(full_path),
			full_path,
			&file_part) == 0)
		{
			ret = GetLastError();

			DEBUG_TRACE2("VfdInstall: GetFullPathName(%s) - %s\n",
				driver_path, ErrMsg(ret));

			return ret;
		}
	}

	//	Check if the file is a valid Virtual FD driver

	ret = VfdCheckDriverFile(full_path);

	if (ret != ERROR_SUCCESS) {
		DEBUG_TRACE1("VfdInstall: VfdCheckDriverFile(%s)\n", full_path);

		return ret;
	}

	//	If the path is under %SystemRoot% make it relative to %SystemRoot%
	{
		TCHAR windir[MAX_PATH];
		int len;

		len = GetEnvironmentVariable("SystemRoot", windir, sizeof(windir));

		if (len > sizeof(windir)) {
			DEBUG_TRACE0(
				"VfdInstall: %%SystemRoot%% contains too long text\n");

			return ERROR_BAD_ENVIRONMENT;
		}
		else if (len && _strnicmp(windir, full_path, len) == 0) {
			memmove(full_path, full_path + len + 1, strlen(full_path) - len);
		}
	}

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(
		NULL,							// local machine
		NULL,							// local database
		SC_MANAGER_CREATE_SERVICE);		// access required

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdInstall: OpenSCManager() - %s", ErrMsg(ret));

		return ret;
	}

	//	Create a new service object

	hService = CreateService(
		hScManager,						// service control manager
		VFD_DEVICE_BASENAME,			// internal service name
		VFD_DEVICE_BASENAME,			// display name
		SERVICE_ALL_ACCESS,				// access mode
		SERVICE_KERNEL_DRIVER,			// service type
		auto_start
			? SERVICE_AUTO_START
			: SERVICE_DEMAND_START,		// service start type
		SERVICE_ERROR_NORMAL,			// start error sevirity
		full_path,						// service image file path
		NULL,							// service group
		NULL,							// service tag
		NULL,							// service dependency
		NULL,							// use LocalSystem account
		NULL							// password for the account
	);

	if (!hService) {
		// Failed to create a service object
		ret = GetLastError();

		DEBUG_TRACE1("VfdInstall: CreateService() - %s", ErrMsg(ret));

		goto cleanup;
	}

cleanup:
	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_INSTALL, 0);
	}

	return ret;
}

//
//	Remove Virtual Floppy Driver entry from system registry
//
DWORD VfdRemove()
{
	SC_HANDLE	hScManager;				// Service Control Manager
	SC_HANDLE	hService;				// Service (= Driver)
	DWORD		ret = ERROR_SUCCESS;

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(NULL, NULL, 0);

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdRemove: OpenSCManager() - %s", ErrMsg(ret));

		return ret;
	}

	//	Open Existing Service Object

	hService = OpenService(
		hScManager,						// Service control manager
		VFD_DEVICE_BASENAME,			// service name
		DELETE);						// service access mode

	if (hService == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdRemove: OpenService(DELETE) - %s", ErrMsg(ret));

		goto cleanup;
	}

	//	Remove driver entry from registry

	if (!DeleteService(hService)) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdRemove: DeleteService() - %s", ErrMsg(ret));

		goto cleanup;
	}

cleanup:
	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_REMOVE, 0);
	}

	return ret;
}

//
//	Start Virtual Floppy Driver
//
DWORD VfdStart(DWORD *state)
{
	SC_HANDLE hScManager;				// Service Control Manager
	SC_HANDLE hService;					// Service (= Driver)
	SERVICE_STATUS	stat;
	DWORD ret = ERROR_SUCCESS;
	int i;

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(NULL, NULL, 0);

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStart: OpenSCManager() - %s", ErrMsg(ret));

		return ret;
	}

	//	Open Existing Service Object

	hService = OpenService(
		hScManager,						// Service control manager
		VFD_DEVICE_BASENAME,			// service name
		SERVICE_START
		| SERVICE_QUERY_STATUS);		// service access mode

	if (hService == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStart: OpenService(SERVICE_START) - %s", ErrMsg(ret));

		goto cleanup;
	}

	//	Start the driver

	if (!StartService(hService, 0, NULL)) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStart: StartService() - %s", ErrMsg(ret));

		goto cleanup;
	}

	//	Ensure the driver is started
	for (i = 0;;) {
		if (!QueryServiceStatus(hService, &stat)) {
			ret = GetLastError();

			DEBUG_TRACE1("VfdStart: QueryServiceStatus() - %s",
				ErrMsg(ret));

			break;
		}

		if (stat.dwCurrentState == SERVICE_RUNNING || ++i == 5) {
			break;
		}

		Sleep(1000);
	}

	if (state) {
		*state = stat.dwCurrentState;
	}

cleanup:
	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_START, 0);
	}

	return ret;
}

//
//	Stop Virtual Floppy Driver
//
DWORD VfdStop(DWORD *state)
{
	SC_HANDLE		hScManager;			// Service Control Manager
	SC_HANDLE		hService;			// Service (= Driver)
	SERVICE_STATUS	stat;
	DWORD			ret = ERROR_SUCCESS;
	int				i = 0;

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(NULL, NULL, 0);

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStop: OpenSCManager() - %s", ErrMsg(ret));

		return ret;
	}

	//	Open Existing Service Object

	hService = OpenService(
		hScManager,						// Service control manager
		VFD_DEVICE_BASENAME,			// service name
		SERVICE_STOP
		| SERVICE_QUERY_STATUS);		// service access mode

	if (hService == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStop: OpenService(SERVICE_STOP) - %s", ErrMsg(ret));

		goto cleanup;
	}

	//	Stop the driver

	if (!ControlService(hService, SERVICE_CONTROL_STOP, &stat)) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdStop: ControlService(SERVICE_CONTROL_STOP) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	//	Ensure the driver is stopped
	while (stat.dwCurrentState != SERVICE_STOPPED && ++i < 5) {
		Sleep(1000);

		if (!QueryServiceStatus(hService, &stat)) {
			ret = GetLastError();

			DEBUG_TRACE1("VfdStop: QueryServiceStatus() - %s",
				ErrMsg(ret));

			break;
		}
	}

	if (state) {
		*state = stat.dwCurrentState;
	}

cleanup:
	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_STOP, 0);
	}

	return ret;
}

//
//	Get Virtual FD driver configuration
//
DWORD VfdGetDriverConfig(
	LPTSTR	driver_path,
	LPDWORD	start_type)
{
	SC_HANDLE	hScManager;				// Service Control Manager
	SC_HANDLE	hService;				// Service (= Driver)
	LPQUERY_SERVICE_CONFIG config = NULL;
	DWORD		size;
	DWORD		ret = ERROR_SUCCESS;

	if (driver_path) {
		ZeroMemory(driver_path, MAX_PATH);
	}

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(NULL, NULL, 0);

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetDriverInfo: OpenSCManager() - %s", ErrMsg(ret));

		return ret;
	}

	//	Open Existing Service Object

	hService = OpenService(
		hScManager,						// Service control manager
		VFD_DEVICE_BASENAME,			// service name
		SERVICE_QUERY_CONFIG);			// service access mode

	if (hService == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetDriverInfo: OpenService(SERVICE_QUERY_CONFIG) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	//	Get the size of config information

	if (!QueryServiceConfig(hService, NULL, 0, &size)) {
		ret = GetLastError();

		if (ret == ERROR_INSUFFICIENT_BUFFER) {
			ret = ERROR_SUCCESS;
		}
		else {
			DEBUG_TRACE1("VfdGetDriverInfo: QueryServiceConfig() - %s",
				ErrMsg(ret));

			goto cleanup;
		}
	}

	//	allocate a required buffer

	config = malloc(size);

	if (config == NULL) {
		ret = GetLastError();

		DEBUG_TRACE2("VfdGetDriverInfo: malloc(%lu) - %s\n",
			size, ErrMsg(ret));

		goto cleanup;
	}

	//	get the config information

	if (!QueryServiceConfig(hService, config, size, &size)) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetDriverInfo: QueryServiceConfig() - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	//	copy information to output buffer

	if (driver_path) {
		if (strncmp(config->lpBinaryPathName, "\\??\\", 4) == 0) {
			strncpy(
				driver_path,
				config->lpBinaryPathName + 4,
				MAX_PATH);
		}
		else {
			strncpy(
				driver_path,
				config->lpBinaryPathName,
				MAX_PATH);
		}
	}

	if (start_type) {
		*start_type = config->dwStartType;
	}

cleanup:
	//	Free service config buffer

	if (config) {
		free(config);
	}

	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	return ret;
}

//
//	Get Virtual FD driver state
//
DWORD VfdGetDriverState(
	LPDWORD current_state)
{
	SC_HANDLE		hScManager = NULL;	// Service Control Manager
	SC_HANDLE		hService = NULL;	// Service (= Driver)
	SERVICE_STATUS	status;
	DWORD			ret = ERROR_SUCCESS;

	if (current_state) {
		*current_state = 0;
	}

	//	Connect to the Service Control Manager

	hScManager = OpenSCManager(NULL, NULL, 0);

	if (hScManager == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetDriverState: OpenSCManager() - %s",
			ErrMsg(ret));

		return ret;
	}

	//	Open Existing Service Object

	hService = OpenService(
		hScManager,						// Service control manager
		VFD_DEVICE_BASENAME,			// service name
		SERVICE_QUERY_STATUS);			// service access mode

	if (hService == NULL) {
		ret = GetLastError();

		DEBUG_TRACE1(
			"VfdGetDriverState: OpenService(SERVICE_QUERY_STATUS) - %s",
			ErrMsg(ret));

		if (ret == ERROR_SERVICE_DOES_NOT_EXIST) {
			if (current_state) {
				*current_state = VFD_NOT_INSTALLED;
			}
			ret = ERROR_SUCCESS;
		}

		goto cleanup;
	}

	//	Get current driver status

	memset(&status, 0, sizeof(status));

	if (!QueryServiceStatus(hService, &status)) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetDriverState: QueryServiceStatus() - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	if (current_state) {
		*current_state = status.dwCurrentState;
	}

cleanup:
	//	Close the service object handle

	if (hService) {
		CloseServiceHandle(hService);
	}

	//	Close handle to the service control manager.

	if (hScManager) {
		CloseServiceHandle(hScManager);
	}

	return ret;
}

//
//	Mount Virtual Floppy Image File
//
DWORD VfdMount(
	LPCTSTR	file_name,
	BOOL	read_only,
	ULONG	file_size)
{
	PVFD_FILE_INFO	fileinfo;
	LPCTSTR			prefix;
	TCHAR			abspath[MAX_PATH];
	DWORD			length;
	HANDLE			hDevice;
	DWORD			ret = ERROR_SUCCESS;

	//	Check image file

	if (file_name && *file_name) {
		ret = VfdCheckImageFile(file_name, &file_size, &read_only);

		if (ret == ERROR_FILE_NOT_FOUND) {
			//	If file does not exist, filesize must be specified
			//	and read-only option must *not* be specified

			if (file_size != VFD_FILESIZE_720KB &&
				file_size != VFD_FILESIZE_1P44MB &&
				file_size != VFD_FILESIZE_2P88MB)
			{
				DEBUG_TRACE1("VfdMount: Invalid file size - %lu\n",
					file_size);
				return VFD_ERROR_IMAGE_SIZE;
			}

			if (read_only) {
				DEBUG_TRACE0( "VfdMount: Create read-only\n");
				return VFD_ERROR_CREATE_READONLY;
			}

			ret = ERROR_SUCCESS;
		}
		else if (ret != ERROR_SUCCESS) {
			return ret;
		}

		//	Prepare absolute path in WIN32 namespace

		if (*file_name == '\\' && *(file_name + 1) == '\\') {

			// \\server\share\path\floppy.img

			prefix = "\\??\\UNC";
			file_name++;			// omit first '\'
		}
		else {
			LPTSTR file_part;

			if (GetFullPathName(file_name, sizeof(abspath), abspath, &file_part) == 0) {
				ret =  GetLastError();

				DEBUG_TRACE2("VfdMount: GetFullPathName(%s) - %s\n",
					file_name, ErrMsg(ret));

				return ret;
			}

			prefix = "\\??\\";
			file_name = abspath;
		}

		length = strlen(prefix) + strlen(file_name);
	}
	else {
		prefix = NULL;
		length = 0;
	}

	//	prepare VFD_FILE_INFO structure

	fileinfo = malloc(sizeof(VFD_FILE_INFO) + length);

	if (fileinfo == NULL) {
		ret = GetLastError();

		DEBUG_TRACE2("VfdMount: malloc(%lu) - %s\n",
			sizeof(VFD_FILE_INFO) + length, ErrMsg(ret));

		return ret;
	}

	memset(fileinfo, 0, sizeof(VFD_FILE_INFO) + length);

	if (length) {
		sprintf((char *)fileinfo->FileName, "%s%s", prefix, file_name);
	}
	fileinfo->FileNameLength = (USHORT)length;

	fileinfo->ReadOnly = (BOOLEAN)read_only;
	fileinfo->FileSize = file_size;

	DEBUG_TRACE3("Mounting file \"%s\" %lu bytes %s\n",
		fileinfo->FileName, file_size, read_only ? "ro" : "");

	//	Open Virtual FD device
	hDevice = VfdOpenDevice();

	if (hDevice == INVALID_HANDLE_VALUE) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdMount: CreateFile(" VFD_DEVICE_DOSNAME ") - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	//	Mount image file

	if (!DeviceIoControl(
		hDevice,
		IOCTL_VFD_OPEN_FILE,
		fileinfo,
		sizeof(VFD_FILE_INFO) + fileinfo->FileNameLength,
		NULL,
		0,
		&length,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdMount: DeviceIoControl(IOCTL_VFD_MOUNT_FILE) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

cleanup:
	if (hDevice != INVALID_HANDLE_VALUE) {
		CloseHandle(hDevice);
	}

	if (fileinfo) {
		free(fileinfo);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_MOUNT, 0);
	}

	return ret;
}

//
//	Unmount Image File
//
DWORD VfdUmount()
{
	HANDLE	hDevice;
	DWORD	size;
	DWORD	ret = ERROR_SUCCESS;

	//	Open Virtual FD device
	hDevice = VfdOpenDevice();

	if (hDevice == INVALID_HANDLE_VALUE) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdUmount: CreateFile(" VFD_DEVICE_DOSNAME ") - %s",
			ErrMsg(ret));

		return ret;
	}

	if (!DeviceIoControl(
		hDevice,
		FSCTL_LOCK_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&size,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdUmount: DeviceIoControl(FSCTL_LOCK_VOLUME) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	if (!DeviceIoControl(
		hDevice,
		IOCTL_VFD_CLOSE_FILE,
		NULL,
		0,
		NULL,
		0,
		&size,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdUmount: DeviceIoControl(IOCTL_VFD_UNMOUNT_FILE) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	if (!DeviceIoControl(
		hDevice,
		FSCTL_DISMOUNT_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&size,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdUmount: DeviceIoControl(FSCTL_DISMOUNT_VOLUME) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	if (!DeviceIoControl(
		hDevice,
		FSCTL_UNLOCK_VOLUME,
		NULL,
		0,
		NULL,
		0,
		&size,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdUmount: DeviceIoControl(FSCTL_UNLOCK_VOLUME) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

cleanup:
	if (hDevice != INVALID_HANDLE_VALUE) {
		CloseHandle(hDevice);
	}

	//	Broadcast the successful operation
	if (ret == ERROR_SUCCESS) {
		VfdNotify(VFD_OPERATION_UMOUNT, 0);
	}

	return ret;
}

//
//	Get Virtual FD image file info
//
DWORD VfdGetFileInfo(
	LPTSTR	file_name,
	ULONG	*file_size,
	BOOL	*read_only)
{
	HANDLE			hDevice;
	PVFD_FILE_INFO	file_info;
	DWORD			size;
	DWORD			ret = ERROR_SUCCESS;

	if ((file_info = malloc(sizeof(VFD_FILE_INFO) + MAX_PATH)) == NULL) {
		ret = GetLastError();

		DEBUG_TRACE2("VfdGetFileInfo: malloc(%lu) - %s\n",
			sizeof(VFD_FILE_INFO) + MAX_PATH, ErrMsg(ret));

		return ret;
	}

	memset(file_info, 0, sizeof(VFD_FILE_INFO) + MAX_PATH);

	//	Open Virtual FD device
	hDevice = VfdOpenDevice();

	if (hDevice == INVALID_HANDLE_VALUE) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetFileInfo: CreateFile(" VFD_DEVICE_DOSNAME ") - %s",
			ErrMsg(ret));

		goto cleanup;;
	}

	//	Query file information

	if (!DeviceIoControl(
		hDevice,
		IOCTL_VFD_QUERY_FILE,
		NULL,
		0,
		file_info,
		sizeof(VFD_FILE_INFO) + MAX_PATH,
		&size,
		NULL))
	{
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetFileInfo: DeviceIoControl(IOCTL_VFD_QUERY_FILE) - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	//	copy obtained information to output buffer

	if (file_name) {
		//	if filename is too long, clip it
		if (file_info->FileNameLength >= MAX_PATH) {
			file_info->FileNameLength = MAX_PATH - 1;
		}
		// ensure the name is properly terminated
		file_info->FileName[file_info->FileNameLength] = '\0';

		if (strncmp(file_info->FileName, "\\??\\UNC", 7) == 0) {
			*file_name = '\\';
			strcpy(file_name + 1, file_info->FileName + 7);
		}
		else if (strncmp(file_info->FileName, "\\??\\", 4) == 0) {
			strcpy(file_name, file_info->FileName + 4);
		}
		else {
			strcpy(file_name, file_info->FileName);
		}
	}

	if (read_only) {
		*read_only = file_info->ReadOnly;
	}

	if (file_size) {
		*file_size = file_info->FileSize;
	}

cleanup:
	if (hDevice != INVALID_HANDLE_VALUE) {
		CloseHandle(hDevice);
	}

	if (file_info) {
		free(file_info);
	}

	return ret;
}

//
//	Get current media state (mounted / write protected)
//
DWORD VfdGetMediaState()
{
	HANDLE	hDevice;
	DWORD	size;
	DWORD	ret = ERROR_SUCCESS;

	//	Open Virtual FD device
	hDevice = VfdOpenDevice();

	if (hDevice == INVALID_HANDLE_VALUE) {
		ret = GetLastError();

		DEBUG_TRACE1("VfdGetMediaState: CreateFile(" VFD_DEVICE_DOSNAME ") - %s",
			ErrMsg(ret));

		return ret;
	}

	//	Query file information

	if (!DeviceIoControl(
		hDevice,
		IOCTL_DISK_IS_WRITABLE,
		NULL,
		0,
		NULL,
		0,
		&size,
		NULL))
	{
		ret = GetLastError();

		if (ret != ERROR_NOT_READY) {
			DEBUG_TRACE1(
				"VfdGetFileInfo: DeviceIoControl(IOCTL_DISK_IS_WRITABLE) - %s",
				ErrMsg(ret));
		}
	}

	CloseHandle(hDevice);

	return ret;
}

//
//	Assign a DOS drive letter to Virtual Floppy Drive
//
DWORD VfdSetDriveLetter(
	TCHAR drive_letter)
{
	DWORD ret = ERROR_SUCCESS;
	TCHAR dos_device[] = " :";
	TCHAR device_name[MAX_PATH];

	if (!isalpha(drive_letter)) {
		return ERROR_INVALID_DRIVE;
	}

	dos_device[0] = (TCHAR)toupper(drive_letter);

	//	Check if the drive letter is already in use

	if (QueryDosDevice(dos_device, device_name, sizeof(device_name))) {
		if (strcmp(device_name, VFD_DEVICE_FULLNAME) == 0) {
			return ERROR_SUCCESS;
		}
		else {
			DEBUG_TRACE2(
				"VfdLink: Drive letter '%c' is linked to '%s'\n",
				dos_device[0], device_name);

			return ERROR_ALREADY_ASSIGNED;
		}
	}
	else {
		if ((ret = GetLastError()) != ERROR_FILE_NOT_FOUND) {
			DEBUG_TRACE2(
				"VfdLink: QueryDosDevice(%s) - %lu\n",
				dos_device, ErrMsg(ret));

			return ret;
		}
	}

	//	Check if the Virtual FD drive has a DriveLetter already assigned

	if ((ret = VfdGetDriveLetter(&drive_letter)) != ERROR_SUCCESS) {
		DEBUG_TRACE1("VfdLink: VfdGetDriveLetter - %s",
			ErrMsg(ret));

		return ret;
	}

	if (isalpha(drive_letter)) {
		if (drive_letter == dos_device[0]) {
			return ERROR_SUCCESS;
		}
		else {
			DEBUG_TRACE1("Drive Letter '%c' is already assigned\n",
				drive_letter);

			return VFD_ERROR_ALREADY_LINKED;
		}
	}

	//	Assign the new drive letter

	if (DefineDosDevice(DDD_RAW_TARGET_PATH, dos_device, VFD_DEVICE_FULLNAME)) {
		//	operation successfull
		VfdNotify(VFD_OPERATION_LINK, 0);
	}
	else {
		ret = GetLastError();

		DEBUG_TRACE2("VfdLink: DefineDosDevice(%s) - %s\n",
			dos_device, ErrMsg(ret));
	}

	return ret;
}

//
//	Remove Dos Drive Letter
//
DWORD VfdDelDriveLetter(
	TCHAR drive_letter)
{
	TCHAR dos_device[] = " :";
	DWORD ret = ERROR_SUCCESS;

	if (!isalpha(drive_letter)) {
		return ERROR_INVALID_DRIVE;
	}

	dos_device[0] = drive_letter;

	if (DefineDosDevice(
		(DDD_RAW_TARGET_PATH | DDD_REMOVE_DEFINITION | DDD_EXACT_MATCH_ON_REMOVE),
		dos_device,
		VFD_DEVICE_FULLNAME)) {
		//	operation successfull
		VfdNotify(VFD_OPERATION_ULINK, 0);
	}
	else {
		ret = GetLastError();

		DEBUG_TRACE2(
			"VfdUnlink: DefineDosDevice(%s) - %s\n",
			dos_device, ErrMsg(ret));
	}

	return ret;
}

//
//	Get Virtual FD drive letter
//
DWORD VfdGetDriveLetter(
	TCHAR *drive_letter)
{
	DWORD	logical_drives;
	TCHAR	dos_device[] = " :";
	TCHAR	device_name[MAX_PATH];
	DWORD	ret = ERROR_SUCCESS;

	if (!drive_letter) {
		return ERROR_INVALID_PARAMETER;
	}

	*drive_letter = '\0';

	logical_drives = GetLogicalDrives();

	if (logical_drives == 0) {
		ret = GetLastError();

		DEBUG_TRACE1(
			"VfdGetDriveLetter: GetLogicalDrives - %s", ErrMsg(ret));

		return ret;
	}

	dos_device[0] = 'A';

	while (logical_drives) {
		if (logical_drives & 0x01) {
			if (QueryDosDevice(dos_device, device_name, sizeof(device_name))) {
				if (_stricmp(device_name, VFD_DEVICE_FULLNAME) == 0) {
					*drive_letter = dos_device[0];
					return ERROR_SUCCESS;
				}
			}
			else {
				ret = GetLastError();

				DEBUG_TRACE2(
					"VfdGetDriveLetter: QueryDosDevice(%s) - %s\n",
					dos_device, ErrMsg(ret));

				return ret;
			}
		}
		logical_drives >>= 1;
		dos_device[0]++;
	}

	return ret;
}

//
//	Check if specified file is valid VFD driver
//
DWORD VfdCheckDriverFile(
	LPCTSTR	driver_path)
{
	DWORD	size;
	DWORD	dummy;
	LPVOID	info;
	VS_FIXEDFILEINFO	*fixedinfo;
	DWORD	ret = ERROR_SUCCESS;
	LPTSTR	str;

	//	Check parameter

	if (!driver_path || !*driver_path) {
		return ERROR_INVALID_PARAMETER;
	}

	//	Check if the driver file is accessible?
	{
		HANDLE hFile = CreateFile(
			driver_path,
			GENERIC_READ,
			0,
			NULL,
			OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL,
			NULL);

		if (hFile == INVALID_HANDLE_VALUE) {
			ret = GetLastError();

			return ret;
		}

		CloseHandle(hFile);
	}

	//	Ensure that the driver binary is located on a local drive
	//	because device driver cannot be started on network drives.

	if (*driver_path == '\\' && *(driver_path + 1) == '\\') {
		//	full path is a UNC path -- \\server\dir\...

		DEBUG_TRACE0( "VfdCheckDriverFile: Driver is located on a network drive\n");

		return VFD_ERROR_DRIVER_LOCATION;
	}
	else {
		//	ensure that the drive letter is not a network drive

		char root[] = " :\\";

		root[0] = *driver_path;

		if (GetDriveType(root) == DRIVE_REMOTE) {
			// the drive is a network drive

			DEBUG_TRACE0( "VfdCheckDriverFile: Driver is located on a network drive\n");

			return VFD_ERROR_DRIVER_LOCATION;
		}
	}

	//	check file version

	size = GetFileVersionInfoSize((LPTSTR)driver_path, &dummy);

	if (size == 0) {
		ret = VFD_ERROR_DRIVER_FILE;

		DEBUG_TRACE0(
			"VfdCheckDriverFile: GetFileVersionInfoSize == 0\n");

		return ret;
	}

	if ((info = malloc(size)) == NULL) {
		ret = GetLastError();

		DEBUG_TRACE2(
			"VfdCheckDriverFile: malloc(%lu) - %s\n", size, ErrMsg(ret));

		return ret;
	}

	if (!GetFileVersionInfo((LPTSTR)driver_path, 0, size, info)) {
		ret = GetLastError();

		DEBUG_TRACE1(
			"VfdCheckDriverFile: GetFileVersionInfo - %s", ErrMsg(ret));

		goto cleanup;
	}

	size = sizeof(fixedinfo);

	if (!VerQueryValue(info, "\\", &fixedinfo, (PUINT)&size)) {
		ret = GetLastError();

		DEBUG_TRACE1(
			"VfdCheckDriverFile: VerQueryValue(\"\\\") - %s", ErrMsg(ret));

		goto cleanup;
	}

	if (fixedinfo->dwFileOS				!= VOS_NT_WINDOWS32 ||
		fixedinfo->dwFileType			!= VFT_DRV			||
		fixedinfo->dwFileSubtype		!= VFT2_DRV_SYSTEM) {

		DEBUG_TRACE0(
			"VfdCheckDriverFile: Invalid file type flags\n");

		ret = VFD_ERROR_DRIVER_FILE;

		goto cleanup;
	}

	if (!VerQueryValue(info, "\\StringFileInfo\\000004b0\\OriginalFileName", &str, (PUINT)&size)) {
		ret = GetLastError();

		DEBUG_TRACE1(
			"VfdCheckDriverFile: VerQueryValue(\"OriginalFileName\") - %s",
			ErrMsg(ret));

		goto cleanup;
	}

	if (strcmp(str, "vfd.sys")) {
		DEBUG_TRACE0(
			"VfdCheckDriverFile: Invalid original file name\n");

		ret = VFD_ERROR_DRIVER_FILE;

		goto cleanup;
	}

	if (fixedinfo->dwFileVersionMS		< 0x00010000 ||
		fixedinfo->dwProductVersionMS	< 0x00010000) {

		DEBUG_TRACE2(
			"VfdCheckDriverFile: Invalid version values - file:%08x, prod: %08x\n",
			fixedinfo->dwFileVersionMS, fixedinfo->dwProductVersionMS);

		ret = VFD_ERROR_DRIVER_VERSION;

		goto cleanup;
	}

cleanup:
	free(info);

	return ret;
}

//
//	check if specified path is valid for image file
//
DWORD VfdCheckImageFile(
	LPCTSTR	file_name,
	ULONG	*file_size,
	BOOL	*read_only)
{
	WIN32_FILE_ATTRIBUTE_DATA attrib;
	HANDLE	hFile;
	DWORD	ret = ERROR_SUCCESS;

	if (!file_name) {
		return ERROR_INVALID_PARAMETER;
	}

	//	check file attribute

	if (!GetFileAttributesEx(file_name, GetFileExInfoStandard, &attrib)) {
		ret = GetLastError();

		if (ret != ERROR_FILE_NOT_FOUND) {
			DEBUG_TRACE2(
				"VfdCheckImageFile: GetFileAttributesEx(%s) - %s\n",
				file_name, ErrMsg(ret));
		}

		return ret;
	}

	if ((attrib.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
		return VFD_ERROR_DIRECTORY;
	}

	if ((attrib.dwFileAttributes & FILE_ATTRIBUTE_COMPRESSED)	||
		(attrib.dwFileAttributes & FILE_ATTRIBUTE_ENCRYPTED))
	{
		return VFD_ERROR_IMAGE_ATTRIBUTE;
	}

	if (attrib.nFileSizeHigh != 0 ||
		(attrib.nFileSizeLow != VFD_FILESIZE_720KB &&
		attrib.nFileSizeLow != VFD_FILESIZE_1P44MB &&
		attrib.nFileSizeLow != VFD_FILESIZE_2P88MB))
	{
		return VFD_ERROR_IMAGE_SIZE;
	}

	if (file_size) {
		*file_size = attrib.nFileSizeLow;
	}

	//	check if file can be opened

	hFile = CreateFile(file_name,
		(read_only && *read_only) ? GENERIC_READ : GENERIC_READ | GENERIC_WRITE,
		(read_only && *read_only) ? FILE_SHARE_READ : 0,
		NULL, OPEN_EXISTING, 0, NULL);

	if (hFile != INVALID_HANDLE_VALUE) {
		CloseHandle(hFile);
		return ERROR_SUCCESS;
	}

	//	failed to open

	ret = GetLastError();

	if (ret != ERROR_ACCESS_DENIED || (read_only && *read_only)) {
		return ret;
	}

	// try opening it read-only

	hFile = CreateFile(file_name, GENERIC_READ,
		FILE_SHARE_READ, NULL, OPEN_EXISTING, 0, NULL);

	if (hFile == INVALID_HANDLE_VALUE) {

		// cannot open even read-only

		ret = GetLastError();

		DEBUG_TRACE2(
			"VfdCheckImageFile: CreateFile(%s) - %s\n",
			file_name, ErrMsg(ret));

		return ret;
	}

	CloseHandle(hFile);

	// file can be opened read-only

	if (read_only) {
		*read_only = TRUE;
	}

	return ERROR_SUCCESS;
}

//
//	Format an error message from error code
//
int VfdErrorMessage(DWORD err, LPTSTR buf, DWORD size)
{
	int len;

	if (IS_VFD_ERROR(err)) {
		len = FormatMessage(FORMAT_MESSAGE_FROM_HMODULE,
			DLL_INSTANCE, err, 0, buf, size, NULL);

		if (len == 0) {
			len = sprintf(buf, "Unknown VFD error %lu\n", err);
		}
	}
	else {
		len = FormatMessage(
			(FORMAT_MESSAGE_IGNORE_INSERTS | FORMAT_MESSAGE_FROM_SYSTEM),
			NULL, err, 0, buf, size, NULL);

		if (len == 0) {
			len = sprintf(buf, "Unknown system error %lu\n", err);
		}
	}

	return len;
}

// End Of File
