﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Threading;

namespace JoinNotes
{
    /// <summary>
    /// Search.xaml の相互作用ロジック
    /// </summary>
    public partial class Search : Window
    {
        internal enum SearchTypes { Unspecified, And, Or, Regex, Word }
        internal SearchTypes SearchType { get; set; }

        public Search()
        {
            InitializeComponent();
            this.Loaded += new RoutedEventHandler(Search_Loaded);
            this.Closed += new EventHandler(Search_Closed);

            this.resultListView.Items.Clear();   // resultList.ItemsSourceを有効にするため
            this.resultListView.ItemsSource = new ObservableCollection<ListViewItem>();

            {
                // zebra stripes style definition

                //var darkColor = SystemColors.WindowColor;
                //darkColor.ScR *= 0.75f;
                //darkColor.ScG *= 0.75f;
                //darkColor.ScB *= 0.75f;
                //darkColor.Clamp();

                var darkColor = Colors.Black;
                darkColor.ScA = 0.05f;
                darkColor.Clamp();

                //var lightColor = SystemColors.WindowColor;
                //lightColor.ScR *= 1.25f;
                //lightColor.ScG *= 1.25f;
                //lightColor.ScB *= 1.25f;
                //lightColor.Clamp();

                var lightColor = Colors.White;
                lightColor.ScA = 0.05f;
                lightColor.Clamp();

                var lightStyle = new Style();
                lightStyle.Setters.Add(new Setter(BackgroundProperty, new SolidColorBrush(lightColor)));
                //lightStyle.Setters.Add(new Setter(BorderBrushProperty, SystemColors.ControlTextBrush));
                //lightStyle.Setters.Add(new Setter(BorderThicknessProperty, new Thickness(0, 0, 0, 1)));
                lightStyle.Setters.Add(new Setter(PaddingProperty, new Thickness(0, 5, 0, 5)));

                var darkStyle = new Style();
                darkStyle.Setters.Add(new Setter(BackgroundProperty, new SolidColorBrush(darkColor)));
                //darkStyle.Setters.Add(new Setter(BorderBrushProperty, SystemColors.ControlTextBrush));
                //darkStyle.Setters.Add(new Setter(BorderThicknessProperty, new Thickness(0, 0, 0, 1)));
                darkStyle.Setters.Add(new Setter(PaddingProperty, new Thickness(0, 5, 0, 5)));

                this.listViewItemStyles = new[] { lightStyle, lightStyle, darkStyle };
            }

            this.resultListView.SelectionChanged += new SelectionChangedEventHandler(resultListView_SelectionChanged);
        }

        private Style[] listViewItemStyles;

        static IntPtr WndProc(IntPtr hwnd, int msg, IntPtr wParam, IntPtr lParam, ref bool handled)
        {
            if (msg == Application.WM_APP_ACTIVATEAPP)
            {
                Trace.Fail("Search:WndProc");
                handled = true;
            }

            return IntPtr.Zero;
        }

        void Search_Loaded(object sender, RoutedEventArgs e)
        {
            HwndSource source = HwndSource.FromHwnd(new WindowInteropHelper(this).Handle);
            source.AddHook(new HwndSourceHook(WndProc));

            this.andRadioButton.IsChecked = true;

            foreach (var word in Properties.Settings.Default.Search_RecentWords.Split(new[] { ", " }, StringSplitOptions.RemoveEmptyEntries))
                this.searchWordBox.Items.Add(word);
            this.searchWordBox.Focus();
        }

        void Search_Closed(object sender, EventArgs e)
        {
            Application.searchWindow = null;
        }

        void resultListView_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            e.Handled = true;
        }

        void searchWordBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            Util.p("searchWordBox_SelectionChanged");
        }

        void searchWordBox_TextInput(object sender, TextCompositionEventArgs e)
        {
            Util.p("searchWordBox_TextInput");
        }

        void searchWordBox_TargetUpdated(object sender, DataTransferEventArgs e)
        {
            Util.p("searchWordBox_TargetUpdated");
        }

        void searchWordBox_IsKeyboardFocusedChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            Util.p("searchWordBox_IsKeyboardFocusedChanged");
        }

        void searchWordBox_IsKeyboardFocusWithinChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
        }

        void searchWordBox_KeyDown(object sender, KeyEventArgs e)
        {
            var c = (ComboBox)sender;

            if (e.Key == Key.Return && c.Text.Length > 0)
            {
                if (!c.Items.Contains(c.Text))
                {
                    c.Items.Insert(0, c.Text);
                    //FIXME: 個数制限するのでなく、履歴消去コマンドでSearch_RecentWordsをクリア
                    //FIXME: 個数制限はリソースに保存可能な長さを超えないようにするためのもの。
                    var items = new List<string>(c.Items.OfType<string>().Take(10));
                    Properties.Settings.Default.Search_RecentWords = string.Join(", ", items);
                    Properties.Settings.Default.Save();
                }
                BeginSearch();
                e.Handled = true;
            }
        }

        void ListViewItem_KeyDown(object sender, KeyEventArgs e)
        {
            var c = (ListViewItem)sender;
            var item = (SearchResult)c.Content;

            if (e.Key == Key.Return)
            {
                Debug.WriteLine(item.Name, "open(listView, keyDown)");
                OpenNotes(new[] { item });
                e.Handled = true;
            }
        }

        SearchResult selectedListViewItem = default(SearchResult);
        void ListViewItem_MouseUp(object sender, MouseButtonEventArgs e)
        {
            var c = (ListViewItem)sender;

            var item = (SearchResult)c.Content;
            if (ReferenceEquals(item, this.selectedListViewItem))
            {
                Debug.WriteLine(this.selectedListViewItem.Name, "open(listView)");
                OpenNotes(new[] { this.selectedListViewItem });
            }

            e.Handled = true;
        }

        void ListViewItem_PreviewMouseDown(object sender, MouseButtonEventArgs e)
        {
            var c = (ListViewItem)sender;

            c.Focus();
            this.selectedListViewItem = (SearchResult)c.Content;
            Debug.WriteLine(this.selectedListViewItem.Name, "mousedown(listView)");

            e.Handled = true;
        }

        SearchResult selectedThumbnailItem = default(SearchResult);
        void ThumbnailViewItem_MouseUp(object sender, MouseButtonEventArgs e)
        {
            var c = (Frame)sender;

            var item = this.searchResults[(Uri)c.Resources[typeof(Uri)]];
            if (ReferenceEquals(item, this.selectedThumbnailItem))
            {
                Debug.WriteLine(this.selectedThumbnailItem.Name, "open(thumbnailView)");
                OpenNotes(new[] { this.selectedThumbnailItem });
            }

            this.selectedThumbnailItem = default(SearchResult);
            e.Handled = true;
        }

        void ThumbnailViewItem_MouseDown(object sender, MouseButtonEventArgs e)
        {
            var c = (Frame)sender;

            c.Focus();
            this.selectedThumbnailItem = this.searchResults[(Uri)c.Resources[typeof(Uri)]];
            Debug.WriteLine(this.selectedThumbnailItem.Name, "mousedown(thumbnailView)");

            e.Handled = true;
        }

        void TabControl_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            //foreach (var result in this.searchResultsMemo.Values)
            //{
            //    AddItemToView(result);
            //}
        }

        void RadioButton_Checked(object sender, RoutedEventArgs e)
        {
            Debug.Assert(this.andRadioButton.IsChecked.HasValue && this.orRadioButton.IsChecked.HasValue && this.regexRadioButton.IsChecked.HasValue);

            if (this.andRadioButton.IsChecked.Value)
                this.SearchType = SearchTypes.And;
            else if (this.orRadioButton.IsChecked.Value)
                this.SearchType = SearchTypes.Or;
            else if (this.regexRadioButton.IsChecked.Value)
                this.SearchType = SearchTypes.Regex;
            else if (this.wordRadioButton.IsChecked.Value)
                this.SearchType = SearchTypes.Word;
            else
            {
                //FIXME: ここだけでなく他の箇所もnullよりdefault(TYPE)に
                this.SearchType = SearchTypes.Unspecified;
                Debug.Fail("Illegal searchType");
            }
        }

        private System.ComponentModel.ListSortDirection listViewItemOrder = System.ComponentModel.ListSortDirection.Ascending;
        void ListViewColumnHeader_Click(object sender, RoutedEventArgs e)
        {
            var c = (GridViewColumnHeader)sender;

            if (this.resultListView.ItemsSource == null)
                return;

            var dataView = CollectionViewSource.GetDefaultView(this.resultListView.ItemsSource);

            if (!dataView.CanSort)
                return;

            dataView.SortDescriptions.Clear();

            if (this.listViewItemOrder == System.ComponentModel.ListSortDirection.Ascending)
                this.listViewItemOrder = System.ComponentModel.ListSortDirection.Descending;
            else
                this.listViewItemOrder = System.ComponentModel.ListSortDirection.Ascending;

            dataView.SortDescriptions.Add(new System.ComponentModel.SortDescription("Content" + "." + c.Tag.ToString(), this.listViewItemOrder));

            // redraw zebra stripes
            for (dataView.MoveCurrentToFirst(); !dataView.IsCurrentAfterLast; dataView.MoveCurrentToNext())
            {
                var item = (ListViewItem)dataView.CurrentItem;
                item.Style = this.listViewItemStyles[dataView.CurrentPosition % this.listViewItemStyles.Length];
            }
        }

        void Window_PreviewKeyDown(object sender, KeyEventArgs e)
        {
            if ((e.Key == Key.W && e.KeyboardDevice.Modifiers == ModifierKeys.Control)
                || e.Key == Key.Escape && e.KeyboardDevice.Modifiers == ModifierKeys.None)
            {
                this.Close();
            }
        }

        internal string SearchQuery
        {
            get { return this.searchWordBox.Text; }
            set { this.searchWordBox.Text = value; }
        }

        private Dictionary<Uri, SearchResult> searchResultsMemo = new Dictionary<Uri, SearchResult> { };
        private Dictionary<Uri, SearchResult> searchResults = new Dictionary<Uri, SearchResult> { };
        //Dictionary<CancellationToken, Task> searchTask = new Dictionary<CancellationToken, Task> { };

        internal void BeginSearch()
        {
            if (this.SearchQuery.Length == 0)
                return;

            if (this.beginSearchButton.Tag == null)
                this.beginSearchButton.Tag = new Dictionary<string, object>();

            ((Dictionary<string, object>)this.beginSearchButton.Tag)["_Content"] = this.beginSearchButton.Content;
            this.beginSearchButton.Content = "検索やり直し";

            {
                this.progressBar.Value = 0;
                //FIXME: 検索中のEnumeratesFilesと統合
                this.progressBar.Maximum = Directory.EnumerateFiles(Application.DocumentPath.FullName, "*.join.rtf", SearchOption.TopDirectoryOnly).Sum(f => new FileInfo(f).Length);
                this.progressBar.Minimum = 0.0;
            }

            this.searchResultsMemo.Clear();
            this.searchResults.Clear();

            // Clear ListView
            var items = (ObservableCollection<ListViewItem>)this.resultListView.ItemsSource;
            items.Clear();

            // Clear ThumbnailView
            this.thumbnailWrapPanel.Children.Clear();
            this.thumbnailWrapPanel.Margin = new Thickness(10);

            Regex[] patterns;

            switch (this.SearchType)
            {
                case SearchTypes.And:
                    patterns = this.SearchQuery.Split(new[] { ' ', '　' }, StringSplitOptions.RemoveEmptyEntries).Distinct().Select<string, Regex>(_ => new Regex(Regex.Escape(_), RegexOptions.IgnoreCase | RegexOptions.Multiline)).ToArray();
                    break;

                case SearchTypes.Or:
                    patterns = new[] { new Regex(string.Join("|", this.SearchQuery.Split(new[] { ' ', '　' }, StringSplitOptions.RemoveEmptyEntries).Distinct()), RegexOptions.IgnoreCase | RegexOptions.Multiline) };
                    break;

                case SearchTypes.Regex:
                    patterns = new[] { new Regex(this.SearchQuery, RegexOptions.IgnoreCase | RegexOptions.Multiline) };
                    break;

                case SearchTypes.Word:
                    patterns = new[] { new Regex(@"\b" + this.SearchQuery + @"\b", RegexOptions.IgnoreCase | RegexOptions.Multiline) };
                    break;

                default:
                    patterns = new Regex[] { };
                    Debug.Fail("Illegal SearchType");
                    break;
            }

            this.progressBar.IsIndeterminate = true;

            {
                // ファイル名だけを検索
                var directory = Application.DocumentPath;
                var fileInfos = directory.EnumerateFiles("*.join.rtf", SearchOption.TopDirectoryOnly).OrderByDescending(_ => _.LastWriteTimeUtc);

                foreach (var fileinfo in fileInfos)
                {
                    {
                        var matchedPatterns = new HashSet<Regex> { };
                        var filename = fileinfo.Name;
                        SearchResult result = SearchResult.Failed;

                        foreach (var pattern in patterns)
                        {
                            var match = pattern.Match(filename);
                            if (match.Success)
                            {
                                if (!matchedPatterns.Contains(pattern))
                                    matchedPatterns.Add(pattern);

                                //FIXME: ファイル内容から得るプレビューと同じ処理なので、統一。
                                const int previewLength = 30;
                                var startIndex = Math.Max(0, match.Index - (int)(previewLength / 2));
                                var endIndex = Math.Min(filename.Length - 1, match.Index + match.Length + (int)(previewLength / 2));
                                var pre = filename.Substring(startIndex, match.Index - startIndex).Replace("\r\n", " ↲ ").Trim('\r', '\n');

                                //FIXME: 複数回マッチした場合にresultが上書きされる→全て残すか代入を一度きりにするか
                                //FIXME: ファイル名・全文・リンクのみ、3箇所の new SearchResult() を統一
                                result = new SearchResult()
                                {
                                    Success = true,
                                    FileInfo = fileinfo,
                                    Name = fileinfo.Name,
                                    Modified = fileinfo.LastWriteTime.ToString(),
                                    FoundPhraseNumber = 0,
                                    PrePreview = pre,
                                    PrePreviewForSort = string.Join("", pre.Reverse()).ToString(),
                                    FoundPhrase = match.Value.Trim(),   // String.ToLower()しない
                                    PostPreview = match.Value.Trim() + filename.Substring(match.Index + match.Length, endIndex - (match.Index + match.Length) + 1).Replace("\r\n", " ↲ ").Trim('\r', '\n'),
                                };

                                Debug.WriteLine(new { result.PrePreview, result.PrePreviewForSort }, "SearchResult");
                            }
                        }

                        if (result.Success
                                && (this.SearchType == SearchTypes.Or || this.SearchType == SearchTypes.Regex || this.SearchType == SearchTypes.Word
                                    || (this.SearchType == SearchTypes.And && matchedPatterns.Count == patterns.Length)
                                )
                            )
                        {
                            AddItemToView(result);
                        }
                    }

                    this.resultViewTabControl.Dispatcher.BeginInvoke(new Action<Regex[], SearchTypes, FileInfo>((_patterns, _type, _fileinfo) =>
                    {
                        if (this.SearchTarget_Text.IsChecked.Value == true)
                            searchText(_patterns, this.SearchType, _fileinfo);
                        else
                            searchHyperlink(_patterns, this.SearchType, _fileinfo);

                        this.progressBar.Value += _fileinfo.Length;
                        if (this.progressBar.Value >= this.progressBar.Maximum)
                        {
                            this.progressBar.Value = this.progressBar.Minimum;
                            this.beginSearchButton.Content = ((Dictionary<string, object>)this.beginSearchButton.Tag)["_Content"];
                        }
                    }), DispatcherPriority.Background, patterns, this.SearchType, fileinfo);

                    //{
                    //// TreeView
                    //    var item = new TreeViewItem();
                    //    item.Header = fileinfo.Name;
                    //    var doc = new FlowDocument();
                    //    var pointer = new TextRange(doc.ContentStart, doc.ContentEnd);
                    //    pointer.Load(fileinfo.OpenRead(), DataFormats.Rtf);
                    //    var range = patterns.Match(pointer.Text);
                    //    while (range.Success)
                    //    {
                    //        item.Items.Add(range.Value);
                    //        range = patterns.Match(pointer.Text, range.Index + range.Length);
                    //    }
                    //    this.treeView1.Items.Add(item);
                    //}

                    //{
                    //    var doc = new FlowDocument();
                    //    var pointer = new TextRange(doc.ContentStart, doc.ContentEnd);
                    //    pointer.Load(fileinfo.OpenRead(), DataFormats.Rtf);
                    //
                    //    // Thumb View (using RichTextBox)
                    //    if (patterns.IsMatch(pointer.Text))
                    //    {
                    //        var box = new RichTextBox();
                    //        {
                    //            box.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    //            box.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    //            box.IsReadOnly = true;
                    //            box.IsReadOnlyCaretVisible = false;
                    //            box.Document = doc;
                    //            box.FontSize = 12;
                    //            box.FontFamily = new FontFamily("Meiryo UI");
                    //            box.Width = 1000;
                    //            box.Height = 180;
                    //            box.Background = Brushes.WhiteSmoke;
                    //            box.BorderBrush = null;
                    //            box.BorderThickness = new Thickness(0);
                    //            box.Padding = new Thickness(5);
                    //            box.Margin = new Thickness(20);
                    //        }
                    //        var viewbox = new Viewbox();
                    //        viewbox.Child = box;
                    //        viewbox.ClipToBounds = true;
                    //        viewbox.Width = 120;
                    //        viewbox.Height = 160;
                    //        viewbox.Stretch = Stretch.UniformToFill;
                    //        this.wrapPanel1.Children.Add(viewbox);
                    //    }
                    //}

                }
            }

            this.progressBar.IsIndeterminate = false;
        }

        private void searchHyperlink(Regex[] patterns, SearchTypes searchType, FileInfo fileinfo)
        {
            // リンク検索は完全一致。部分一致では適合しない。
            //FIXME: バックリンクを探すためのものなので、オプションを無視していい。または完全一致だけを探す"word"オプションを作る。
            Debug.Assert(searchType != SearchTypes.Unspecified);
            Debug.Assert((searchType == SearchTypes.Regex || searchType == SearchTypes.Word) ? patterns.Length == 1 : patterns.Length >= 1);

            //var matches = new List<Match> { };
            var isSatisfySearchConditions = false;

            var doc = new FlowDocument();
            var range = new TextRange(doc.ContentStart, doc.ContentEnd);

            var matchedRanges = new List<TextRange> { };

            try
            {
                using (var stream = fileinfo.OpenRead())
                    range.Load(stream, DataFormats.Rtf);

                var matchedPatterns = new HashSet<Regex> { };

                for (var pointer = range.Start; pointer != null && pointer.CompareTo(range.End) < 1; pointer = pointer.GetNextContextPosition(LogicalDirection.Forward))
                {
                    if (pointer.GetPointerContext(LogicalDirection.Backward) == TextPointerContext.ElementStart && pointer.Parent is Hyperlink)
                    {
                        //Debug.Assert(pointer.GetPointerContext(LogicalDirection.Forward) == TextPointerContext.Text);
                        //var matchingTarget = pointer.GetTextInRun(LogicalDirection.Forward);
                        var matchingTarget = Util.TextOf(Util.ContentRangeOf((Hyperlink)pointer.Parent));
                        Debug.Assert(matchingTarget.Length > 0);

                        foreach (var pattern in patterns)
                        {
                            var match = pattern.Match(matchingTarget);
                            if (match.Success)
                            {
                                if (!matchedPatterns.Contains(pattern))
                                    matchedPatterns.Add(pattern);
                                //TODO: HyperlinkのElementRangeではなく適合箇所のRangeに。
                                matchedRanges.Add(Util.ElementRangeOf((Hyperlink)pointer.Parent));
                                //matchedPatterns[range.Value.ToLower()] = true;
                                //matches.Add(range);
                            }
                        }
                    }
                }

                if (searchType == SearchTypes.Or || searchType == SearchTypes.Regex || searchType == SearchTypes.Word
                    || (searchType == SearchTypes.And && matchedPatterns.Count == patterns.Length))
                {
                    isSatisfySearchConditions = true;
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            catch (ArgumentException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            //FIXME: ドキュメントファイルロック時の例外に対処

            if (isSatisfySearchConditions)
            {
                var count = new Dictionary<string, int> { };

                foreach (var matchedRange in matchedRanges)
                {
                    var result = new SearchResult();
                    {
                        result.Success = true;
                        result.FileInfo = fileinfo;
                        result.Name = fileinfo.Name;
                        result.Modified = fileinfo.LastWriteTime.ToString();

                        {
                            var key = Util.TextOf(matchedRange);
                            if (!count.ContainsKey(key))
                                count[key] = 0;

                            count[key]++;

                            result.FoundPhraseNumber = count[key];
                        }

                        {
                            var text = Util.TextOf(new TextRange(range.Start, matchedRange.Start));
                            var previewLength = Math.Min(text.Length, 15);
                            var pre = text.Substring(0 + previewLength, text.Length - previewLength);

                            result.PrePreview = pre;
                            result.PrePreviewForSort = string.Join("", pre.Reverse()).ToString();
                        }

                        result.FoundPhrase = Util.TextOf(matchedRange);

                        {
                            var text = Util.TextOf(new TextRange(matchedRange.End, range.End));
                            var previewLength = Math.Min(text.Length, 15);
                            result.PostPreview = Util.TextOf(matchedRange) + text.Substring(0, previewLength);
                        }

                        this.searchResultsMemo[new Uri(fileinfo.FullName)] = result;

                        AddItemToView(result);
                    }
                }
            }
        }

        private void searchText(Regex[] patterns, SearchTypes searchType, FileInfo fileinfo)
        {
            Debug.Assert(searchType != SearchTypes.Unspecified);
            Debug.Assert((searchType == SearchTypes.Regex || searchType == SearchTypes.Word) ? patterns.Length == 1 : patterns.Length >= 0);

            var matches = new List<Match> { };
            var isSatisfySearchConditions = false;

            var doc = new FlowDocument();
            var range = new TextRange(doc.ContentStart, doc.ContentEnd);

            try
            {
                using (var stream = fileinfo.OpenRead())
                    range.Load(stream, DataFormats.Rtf);

                // matchedPatterns: AND条件を満たすかの判定に使うだけ
                var matchedPatterns = new HashSet<Regex> { };
                //var matchedPatterns = new Dictionary<string, bool> { };

                foreach (var pattern in patterns)
                {
                    var match = pattern.Match(Util.TextOf(range));
                    while (match.Success)
                    {
                        if (!matchedPatterns.Contains(pattern))
                            matchedPatterns.Add(pattern);
                        //matchedPatterns[range.Value.ToLower()] = true;
                        matches.Add(match);
                        match = pattern.Match(Util.TextOf(range), match.Index + match.Length);
                    }
                }

                if (searchType == SearchTypes.Or || searchType == SearchTypes.Regex || searchType == SearchTypes.Word
                    || (searchType == SearchTypes.And && matchedPatterns.Count == patterns.Length))
                {
                    isSatisfySearchConditions = true;
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            catch (ArgumentException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            //FIXME: ドキュメントファイルロック時の例外に対処

            if (isSatisfySearchConditions)
            {
                var count = new Dictionary<string, int> { };

                foreach (var match in matches)
                {
                    var result = new SearchResult();
                    {
                        result.Success = true;
                        result.FileInfo = fileinfo;
                        result.Name = fileinfo.Name;
                        result.Modified = fileinfo.LastWriteTime.ToString();

                        if (!count.ContainsKey(match.Value.ToLower()))
                            count[match.Value.ToLower()] = 0;
                        count[match.Value.ToLower()]++;

                        result.FoundPhraseNumber = count[match.Value.ToLower()];

                        {
                            const int previewLength = 30;
                            var startIndex = Math.Max(0, match.Index - (int)(previewLength / 2));
                            var endIndex = Math.Min(Util.TextOf(range).Length - 1, match.Index + match.Length + (int)(previewLength / 2));
                            var pre = Util.TextOf(range).Substring(startIndex, match.Index - startIndex).Replace("\r\n", " ↲ ").Trim('\r', '\n');

                            result.PrePreview = pre;
                            result.PrePreviewForSort = string.Join("", pre.Reverse()).ToString();
                            result.FoundPhrase = match.Value.Trim();   // String.ToLower()しない
                            result.PostPreview = match.Value.Trim() + Util.TextOf(range).Substring(match.Index + match.Length, endIndex - (match.Index + match.Length) + 1).Replace("\r\n", " ↲ ").Trim('\r', '\n');
                        }
                    }

                    this.searchResultsMemo[new Uri(fileinfo.FullName)] = result;

                    AddItemToView(result);
                }
            }
        }

        void AddItemToView(SearchResult result)
        {
            AddItemToListView(result);
            AddItemToThumbnailView(result);

            //// 開いているタブにだけ検索結果を表示する
            //var tabItem = (TabItem)this.tabControl1.SelectedItem;
            //switch (tabItem.Name)
            //{
            //    case "listViewTab":
            //        AddItemToListView(result);
            //        break;
            //
            //    case "thumbnailViewTab":
            //        AddItemToThumbnailView(result);
            //        break;
            //
            //    default:
            //        Debug.Fail("BeginSearch");
            //        break;
            //}
        }

        void AddItemToListView(SearchResult result)
        {
            //this.searchResults[result.GetHashCode()] = result;

            var items = (ObservableCollection<ListViewItem>)this.resultListView.ItemsSource;

            var item = new ListViewItem() { Content = result };
            {
                //HACK: MouseDown/MouseUpでシングルクリック風にしている
                //HACK: なぜかMouseDownイベントが発生しないのでPreviewMouseDownにしている
                item.PreviewMouseDown += new MouseButtonEventHandler(ListViewItem_PreviewMouseDown);
                item.MouseUp += new MouseButtonEventHandler(ListViewItem_MouseUp);
                item.KeyDown += new KeyEventHandler(ListViewItem_KeyDown);
                item.Cursor = Cursors.Hand;
                item.Style = this.listViewItemStyles[items.Count % this.listViewItemStyles.Length];
            }

            items.Add(item);
        }

        void AddItemToThumbnailView(SearchResult result)
        {
            // ThumbViewでは FoundPhraseNumber == 1 のSearchResultだけ表示
            var uri = new Uri(result.FileInfo.FullName);
            if (!this.searchResults.ContainsKey(uri))
            {
                this.searchResults[uri] = result;

                //TODO: スライダーで選択可能な範囲…の最大解像度に合わせて
                var scale = 0.5;

                RichTextBox box;
                {
                    // 新しいEditorが欲しいのでApp.Instance.GetWindowは使わない。
                    var org = Editor.Create();
                    box = org.RichTextBox;

                    using (box.DeclareChangeBlock())
                    {
                        box.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                        box.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
                        box.IsReadOnly = true;
                        box.IsReadOnlyCaretVisible = false;

                        //FIXME: RichTextBoxの大きさにしたい
                        box.Width = org.Width;
                        box.Height = org.Height;
                    }
                }

                Image thumbnail;
                {
                    // RichTextBox.Documentの設定は RichTextBox.IsReadOnly = true; のあとで。自動保存を防ぐため。
                    var doc = box.Document = new FlowDocument();
                    {
                        var range = new TextRange(doc.ContentStart, doc.ContentEnd);
                        range.Load(result.FileInfo.OpenRead(), DataFormats.Rtf);
                        //FIXME: ドキュメントファイルロック時の例外に対処
                    }

                    var bmp = FlowDocumentToBitmapSource(doc, new Size { Width = box.Width, Height = box.Height }, Util.Ppi(this));
                    thumbnail = new Image
                    {
                        Source = bmp,
                        Width = box.Width * scale,
                        Height = box.Height * scale,
                        Margin = new Thickness(5),
                        Stretch = Stretch.UniformToFill,
                        ClipToBounds = true,
                    };
                }

                var item = new Frame();
                {
                    item.Content = thumbnail;
                    //HACK: MouseDown/MouseUpでシングルクリック風にしている
                    item.MouseDown += new MouseButtonEventHandler(ThumbnailViewItem_MouseDown);
                    item.MouseUp += new MouseButtonEventHandler(ThumbnailViewItem_MouseUp);
                    item.Cursor = Cursors.Hand;
                    item.Resources[typeof(Uri)] = new Uri(result.FileInfo.FullName);
                }

                this.thumbnailWrapPanel.Children.Add(item);
            }
        }

        void OpenNotes(IEnumerable<SearchResult> notes)
        {
            foreach (var note in notes)
            {
                //Process.Start(Path.Combine(basepath, note.Name));
                var editor = Editor.Create(new Uri(Path.Combine(Application.DocumentPath.FullName, note.Name)), note);
                editor.SearchString = note.FoundPhrase;
                editor.Show();
                editor.Activate();

                if (editor.RichTextBox.Selection.Start.Paragraph != null)
                    editor.RichTextBox.Selection.Start.Paragraph.BringIntoView();
            }
        }

        BitmapSource FlowDocumentToBitmapSource(FlowDocument document, Size size, Size ppi)
        {
            // FlowDocumentはバックグラウンド処理不可…UIスレッドで扱わなければいけない？何かがUIスレッドに依存している？

            var paginator = ((IDocumentPaginatorSource)document).DocumentPaginator;
            //HACK: 分割できない要素（例えば画像）が丸々次のページに送られてしまうのを避けるためHeightを増やす。
            paginator.PageSize = new Size(size.Width, size.Height * 2);
            paginator.ComputePageCount();

            var bitmap = new RenderTargetBitmap((int)size.Width, (int)size.Height, ppi.Width, ppi.Height, PixelFormats.Pbgra32);
            var page = paginator.GetPage(0);    // 0: First page
            if (page != DocumentPage.Missing)
            {
                var visual = new DrawingVisual();
                visual.Children.Add(page.Visual);
                bitmap.Render(visual);
            }

            return bitmap;
        }

        FlowDocument CloneDocument(FlowDocument document)
        {
            var copy = new FlowDocument();
            var sourceRange = new TextRange(document.ContentStart, document.ContentEnd);
            var targetRange = new TextRange(copy.ContentStart, copy.ContentEnd);

            using (var stream = new MemoryStream())
            {
                sourceRange.Save(stream, DataFormats.XamlPackage);
                targetRange.Load(stream, DataFormats.XamlPackage);
                //FIXME: ドキュメントファイルロック時の例外に対処
            }

            return copy;
        }

        private void GridViewColumnHeader_PreviewMouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            //FIXME: カラムヘッダーをダブルクリックしたときに起きる自動レイアウトで、セル内の書式が無視される
            var c = (GridViewColumnHeader)sender;
            if (ReferenceEquals(c, this.PrePreviewColumnHeader))
                e.Handled = true;
        }

        private void GridViewColumnHeader_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
        }
    }

    public class SearchResult
    {
        public static SearchResult Failed = new SearchResult() { Success = false };
        public bool Success { get; set; }
        public FileInfo FileInfo { get; set; }
        public string Name { get; set; }
        public string Modified { get; set; }
        public string PrePreview { get; set; }
        public string PrePreviewForSort { get; set; }
        public string PostPreview { get; set; }
        public string FoundPhrase { get; set; }
        public int FoundPhraseNumber { get; set; }
    }
}
