﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Text.RegularExpressions;
using System.IO;
using System.Diagnostics;
using System.Threading.Tasks;
using System.Threading;
using System.Windows.Threading;
using System.Collections.ObjectModel;
using Forms = System.Windows.Forms;
using System.Windows.Interop;

namespace JoinNotes
{
    /// <summary>
    /// Search.xaml の相互作用ロジック
    /// </summary>
    public partial class Search : Window
    {
        private Style[] _listViewItemStyles;

        private static IntPtr WndProc(IntPtr hwnd, int msg, IntPtr wParam, IntPtr lParam, ref bool handled)
        {
            if (msg == Menu.WM_APP_ACTIVATEAPP)
            {
                Trace.Fail("Search:WndProc");
                handled = true;
            }

            return IntPtr.Zero;
        }

        private void Window_Loaded(object sender, RoutedEventArgs e)
        {
            HwndSource source = HwndSource.FromHwnd(new WindowInteropHelper(this).Handle);
            source.AddHook(new HwndSourceHook(WndProc));

            this.andRadioButton.IsChecked = true;

            foreach (var word in Properties.Settings.Default.Search_RecentWords.Split(new[] { ", " }, StringSplitOptions.RemoveEmptyEntries))
                this.comboBox1.Items.Add(word);
            this.comboBox1.Focus();
        }

        private void Window_Closed(object sender, EventArgs e)
        {
        }

        public Search()
        {
            InitializeComponent();

            this.listView1.Items.Clear();   // listView1.ItemsSourceを有効にするため
            this.listView1.ItemsSource = new ObservableCollection<ListViewItem>();

            {

                var darkColor = SystemColors.WindowColor;
                darkColor.ScR -= 0.025f;
                darkColor.ScG -= 0.025f;
                darkColor.ScB -= 0.025f;
                darkColor.Clamp();
                var lightColor = SystemColors.WindowColor;
                lightColor.ScR += 0.025f;
                lightColor.ScG += 0.025f;
                lightColor.ScB += 0.025f;
                lightColor.Clamp();
                var lightStyle = new Style();
                lightStyle.Setters.Add(new Setter(BackgroundProperty, new SolidColorBrush(lightColor)));
                lightStyle.Setters.Add(new Setter(BorderBrushProperty, SystemColors.WindowFrameBrush));
                lightStyle.Setters.Add(new Setter(BorderThicknessProperty, new Thickness(0, 0, 0, 1)));
                lightStyle.Setters.Add(new Setter(PaddingProperty, new Thickness(0, 5, 0, 5)));
                var darkStyle = new Style();
                darkStyle.Setters.Add(new Setter(BackgroundProperty, new SolidColorBrush(darkColor)));
                darkStyle.Setters.Add(new Setter(BorderBrushProperty, SystemColors.WindowFrameBrush));
                darkStyle.Setters.Add(new Setter(BorderThicknessProperty, new Thickness(0, 0, 0, 1)));
                darkStyle.Setters.Add(new Setter(PaddingProperty, new Thickness(0, 5, 0, 5)));
                this._listViewItemStyles = new[] { darkStyle, darkStyle, darkStyle, lightStyle, lightStyle, lightStyle };
            }

            //((System.Windows.Controls.GridViewColumn)(this.listView1.View as GridView).Columns[0]).HeaderContainerStyle = Style;

            //var obj = (this.listView1.View as GridView).Columns[0].CellTemplate.FindName("tb", ((GridViewColumn)(this.listView1.View as GridView).Columns[0]));
            //((TextBlock)obj).Background = Brushes.NavajoWhite;

            //(this.listView1.View as GridView).Columns[0].DisplayMemberBinding = new Binding("Name");
            //(this.listView1.View as GridView).Columns[1].DisplayMemberBinding = new Binding("Modified");
            //(this.listView1.View as GridView).Columns[2].DisplayMemberBinding = new Binding("FoundPhraseNumber");
            //(this.listView1.View as GridView).Columns[3].DisplayMemberBinding = new Binding("PrePreview");
            //(this.listView1.View as GridView).Columns[4].DisplayMemberBinding = new Binding("FoundPhrase");
            //(this.listView1.View as GridView).Columns[5].DisplayMemberBinding = new Binding("PostPreview");

            this.listView1.SelectionChanged += new SelectionChangedEventHandler(listView1_SelectionChanged);
        }

        void listView1_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            e.Handled = true;
        }

        private void comboBox1_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            Util.p("comboBox1_SelectionChanged");
        }

        private void comboBox1_TextInput(object sender, TextCompositionEventArgs e)
        {
            Util.p("comboBox1_TextInput");
        }

        private void comboBox1_TargetUpdated(object sender, DataTransferEventArgs e)
        {
            Util.p("comboBox1_TargetUpdated");
        }

        private void comboBox1_IsKeyboardFocusedChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            Util.p("comboBox1_IsKeyboardFocusedChanged");
        }

        private void comboBox1_IsKeyboardFocusWithinChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
        }

        private Dictionary<Uri, SearchResult> _searchResultsMemo = new Dictionary<Uri, SearchResult> { };

        private void comboBox1_KeyDown(object sender, KeyEventArgs e)
        {
            var c = (ComboBox)sender;

            if (e.Key == Key.Return && c.Text.Length > 0)
            {
                if (!c.Items.Contains(c.Text))
                {
                    c.Items.Insert(0, c.Text);
                    //FIXME: 個数制限するのでなく、履歴消去コマンドでSearch_RecentWordsをクリア
                    //FIXME: 個数制限はリソースに保存可能な長さを超えないようにするためのもの。
                    var items = new List<string>(c.Items.OfType<string>().Take(10));
                    Properties.Settings.Default.Search_RecentWords = string.Join(", ", items);
                    Properties.Settings.Default.Save();
                }
                BeginSearch();
                e.Handled = true;
            }
        }

        private Dictionary<Uri, SearchResult> _searchResults = new Dictionary<Uri, SearchResult> { };

        public void BeginSearch()
        {
            if (this.SearchQuery.Length == 0)
                return;

            //FIXME: ファイルを開く処理をバックグラウンドでできないので、無意味。

            this._searchResultsMemo.Clear();
            this._searchResults.Clear();

            // Clear ListView
            var items = (ObservableCollection<ListViewItem>)this.listView1.ItemsSource;
            items.Clear();

            // Clear ThumbnailView
            this.wrapPanel1.Children.Clear();
            this.wrapPanel1.Margin = new Thickness(10);

            Regex[] patterns;
            {
                if (this.SearchType == "and")
                    patterns = this.SearchQuery.Split(new[] { ' ', '　' }, StringSplitOptions.RemoveEmptyEntries).Distinct().Select<string, Regex>(_ => new Regex(Regex.Escape(_), RegexOptions.IgnoreCase | RegexOptions.Multiline)).ToArray();
                else if (this.SearchType == "or")
                    patterns = new[] { new Regex(string.Join("|", this.SearchQuery.Split(new[] { ' ', '　' }, StringSplitOptions.RemoveEmptyEntries).Distinct()), RegexOptions.IgnoreCase | RegexOptions.Multiline) };
                else if (this.SearchType == "regex")
                    patterns = new[] { new Regex(this.SearchQuery, RegexOptions.IgnoreCase | RegexOptions.Multiline) };
                else
                {
                    patterns = new Regex[] { };
                    Debug.Fail("SearchType");
                }
            }

            {
                // ファイル名だけを検索
                var directory = App.DocumentPath;
                var fileInfos = directory.EnumerateFiles("*.join.rtf", SearchOption.TopDirectoryOnly).OrderByDescending(_ => _.LastWriteTimeUtc);

                foreach (var fileinfo in fileInfos)
                {
                    {
                        var matchedPatterns = new HashSet<Regex> { };
                        var filename = fileinfo.Name;
                        SearchResult result = SearchResult.Failed;

                        foreach (var pattern in patterns)
                        {
                            var match = pattern.Match(filename);
                            if (match.Success)
                            {
                                if (!matchedPatterns.Contains(pattern))
                                    matchedPatterns.Add(pattern);

                                //FIXME: ファイル内容から得るプレビューと同じ処理なので、統一。
                                const int previewLength = 100;
                                var startIndex = Math.Max(0, match.Index - (int)(previewLength / 2));
                                var endIndex = Math.Min(filename.Length - 1, match.Index + match.Length + (int)(previewLength / 2));

                                //FIXME: 複数回マッチした場合にresultが上書きされる→全て残すか代入を一度きりにするか
                                result = new SearchResult()
                                {
                                    Success = true,
                                    FileInfo = fileinfo,
                                    Name = fileinfo.Name,
                                    Modified = fileinfo.LastWriteTime.ToString(),
                                    FoundPhraseNumber = 0,
                                    PrePreview = filename.Substring(startIndex, match.Index - startIndex).Replace("\r\n", "↲ ").Trim('\r', '\n') + " ",
                                    FoundPhrase = match.Value.Trim(),   // String.ToLower()しない
                                    PostPreview = match.Value.Trim()
                                        + " " + filename.Substring(match.Index + match.Length, endIndex - (match.Index + match.Length) + 1).Replace("\r\n", "↲ ").Trim('\r', '\n'),
                                };
                            }
                        }

                        if (result.Success
                            && (
                                this.SearchType.ToLower() == "regex"
                                || this.SearchType.ToLower() == "or"
                                || (this.SearchType.ToLower() == "and" && matchedPatterns.Count == patterns.Length)
                            ))
                        {
                            AddItemToView(result);
                        }
                    }

                    if (this.SearchTarget_Text.IsChecked.Value)
                        this.tabControl1.Dispatcher.BeginInvoke(new Action<Regex[], string, FileInfo>((_patterns, _type, _fileinfo) => _searchText(_patterns, this.SearchType, _fileinfo)), DispatcherPriority.Background, patterns, this.SearchType, fileinfo);
                    else
                        this.tabControl1.Dispatcher.BeginInvoke(new Action<Regex[], string, FileInfo>((_patterns, _type, _fileinfo) => _searchHyperlink(_patterns, this.SearchType, _fileinfo)), DispatcherPriority.Background, patterns, this.SearchType, fileinfo);

                    {
                        // TreeView
                        //var item = new TreeViewItem();
                        //item.Header = fileinfo.Name;
                        //var doc = new FlowDocument();
                        //var pointer = new TextRange(doc.ContentStart, doc.ContentEnd);
                        //pointer.Load(fileinfo.OpenRead(), DataFormats.Rtf);
                        //var range = patterns.Match(pointer.Text);
                        //while (range.Success)
                        //{
                        //    item.Items.Add(range.Value);
                        //    range = patterns.Match(pointer.Text, range.Index + range.Length);
                        //}
                        //this.treeView1.Items.Add(item);
                    }

                    //{
                    //    var doc = new FlowDocument();
                    //    var pointer = new TextRange(doc.ContentStart, doc.ContentEnd);
                    //    pointer.Load(fileinfo.OpenRead(), DataFormats.Rtf);

                    //    // Thumb View (using RichTextBox)
                    //    if (patterns.IsMatch(pointer.Text))
                    //    {
                    //        var box = new RichTextBox();
                    //        {
                    //            box.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    //            box.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    //            box.IsReadOnly = true;
                    //            box.IsReadOnlyCaretVisible = false;
                    //            box.Document = doc;
                    //            box.FontSize = 12;
                    //            box.FontFamily = new FontFamily("Meiryo UI");
                    //            box.Width = 1000;
                    //            box.Height = 180;
                    //            box.Background = Brushes.WhiteSmoke;
                    //            box.BorderBrush = null;
                    //            box.BorderThickness = new Thickness(0);
                    //            box.Padding = new Thickness(5);
                    //            box.Margin = new Thickness(20);
                    //        }
                    //        var viewbox = new Viewbox();
                    //        viewbox.Child = box;
                    //        viewbox.ClipToBounds = true;
                    //        viewbox.Width = 120;
                    //        viewbox.Height = 160;
                    //        viewbox.Stretch = Stretch.UniformToFill;
                    //        this.wrapPanel1.Children.Add(viewbox);
                    //    }
                    //}

                }
            }
        }

        Dictionary<CancellationToken, Task> searchTask = new Dictionary<CancellationToken, Task> { };

        private void _searchHyperlink(Regex[] patterns, string searchType, FileInfo fileinfo)
        {
            Debug.Assert(searchType.ToLower() == "regex" || searchType.ToLower() == "and" || searchType.ToLower() == "or");
            Debug.Assert(searchType.ToLower() == "regex" ? patterns.Length == 1 : patterns.Length >= 1);

            //var matches = new List<Match> { };
            var isSatisfySearchConditions = false;

            var doc = new FlowDocument();
            var range = new TextRange(doc.ContentStart, doc.ContentEnd);

            var matchedRanges = new List<TextRange> { };

            try
            {
                using (var stream = fileinfo.OpenRead())
                    range.Load(stream, DataFormats.Rtf);

                var matchedPatterns = new HashSet<Regex> { };

                for (var pointer = range.Start; pointer != null && pointer.CompareTo(range.End) < 1; pointer = pointer.GetNextContextPosition(LogicalDirection.Forward))
                {
                    if (pointer.GetPointerContext(LogicalDirection.Backward) == TextPointerContext.ElementStart && pointer.Parent is Hyperlink)
                    {
                        //Debug.Assert(pointer.GetPointerContext(LogicalDirection.Forward) == TextPointerContext.Text);
                        //var matchingTarget = pointer.GetTextInRun(LogicalDirection.Forward);
                        var matchingTarget = Util.TextOf(Util.ContentRangeOf((Hyperlink)pointer.Parent));
                        Debug.Assert(matchingTarget.Length > 0);

                        foreach (var pattern in patterns)
                        {
                            var match = pattern.Match(matchingTarget);
                            if (match.Success)
                            {
                                if (!matchedPatterns.Contains(pattern))
                                    matchedPatterns.Add(pattern);
                                //TODO: HyperlinkのElementRangeではなく適合箇所のRangeに。
                                matchedRanges.Add(Util.ElementRangeOf((Hyperlink)pointer.Parent));
                                //matchedPatterns[range.Value.ToLower()] = true;
                                //matches.Add(range);
                            }
                        }
                    }
                }

                if (searchType.ToLower() == "regex"
                    || searchType.ToLower() == "or"
                    || (searchType.ToLower() == "and" && matchedPatterns.Count == patterns.Length))
                {
                    isSatisfySearchConditions = true;
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            catch (ArgumentException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }

            if (isSatisfySearchConditions)
            {
                var count = new Dictionary<string, int> { };

                foreach (var matchedRange in matchedRanges)
                {
                    var result = new SearchResult();
                    {
                        result.Success = true;
                        result.FileInfo = fileinfo;
                        result.Name = fileinfo.Name;
                        result.Modified = fileinfo.LastWriteTime.ToString();

                        {
                            var key = Util.TextOf(matchedRange);
                            if (!count.ContainsKey(key))
                                count[key] = 0;
                            count[key]++;

                            result.FoundPhraseNumber = count[key];
                        }

                        {
                            var text = Util.TextOf(new TextRange(range.Start, matchedRange.Start));
                            var previewLength = Math.Min(text.Length, 50);
                            result.PrePreview = text.Substring(0 + previewLength, text.Length - previewLength) + " ";
                        }

                        result.FoundPhrase = Util.TextOf(matchedRange);

                        {
                            var text = Util.TextOf(new TextRange(matchedRange.End, range.End));
                            var previewLength = Math.Min(text.Length, 50);
                            result.PostPreview = Util.TextOf(matchedRange)
                                + " " + text.Substring(0, previewLength);
                        }

                        this._searchResultsMemo[new Uri(fileinfo.FullName)] = result;

                        AddItemToView(result);
                    }
                }
            }
        }

        //FIXME: searchTypeを列挙型に
        private void _searchText(Regex[] patterns, string searchType, FileInfo fileinfo)
        {
            Debug.Assert(searchType.ToLower() == "regex" || searchType.ToLower() == "and" || searchType.ToLower() == "or");
            Debug.Assert(searchType.ToLower() == "regex" ? patterns.Length == 1 : patterns.Length >= 1);

            var matches = new List<Match> { };
            var isSatisfySearchConditions = false;

            var doc = new FlowDocument();
            var range = new TextRange(doc.ContentStart, doc.ContentEnd);

            try
            {
                using (var stream = fileinfo.OpenRead())
                    range.Load(stream, DataFormats.Rtf);

                // matchedPatterns: AND条件を満たすかの判定に使うだけ
                var matchedPatterns = new HashSet<Regex> { };
                //var matchedPatterns = new Dictionary<string, bool> { };

                foreach (var pattern in patterns)
                {
                    var match = pattern.Match(Util.TextOf(range));
                    while (match.Success)
                    {
                        if (!matchedPatterns.Contains(pattern))
                            matchedPatterns.Add(pattern);
                        //matchedPatterns[range.Value.ToLower()] = true;
                        matches.Add(match);
                        match = pattern.Match(Util.TextOf(range), match.Index + match.Length);
                    }
                }

                if (searchType.ToLower() == "regex"
                    || searchType.ToLower() == "or"
                    || (searchType.ToLower() == "and" && matchedPatterns.Count == patterns.Length))
                {
                    isSatisfySearchConditions = true;
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }
            catch (ArgumentException ex)
            {
                Debug.WriteLine(ex.Message, ex.Source);
            }

            if (isSatisfySearchConditions)
            {
                var count = new Dictionary<string, int> { };

                foreach (var match in matches)
                {
                    var result = new SearchResult();
                    {
                        result.Success = true;
                        result.FileInfo = fileinfo;
                        result.Name = fileinfo.Name;
                        result.Modified = fileinfo.LastWriteTime.ToString();

                        if (!count.ContainsKey(match.Value.ToLower()))
                            count[match.Value.ToLower()] = 0;
                        count[match.Value.ToLower()]++;

                        result.FoundPhraseNumber = count[match.Value.ToLower()];

                        {
                            const int previewLength = 100;
                            var startIndex = Math.Max(0, match.Index - (int)(previewLength / 2));
                            var endIndex = Math.Min(Util.TextOf(range).Length - 1, match.Index + match.Length + (int)(previewLength / 2));
                            result.PrePreview = Util.TextOf(range).Substring(startIndex, match.Index - startIndex).Replace("\r\n", "↲ ").Trim('\r', '\n') + " ";
                            result.FoundPhrase = match.Value.Trim();   // String.ToLower()しない
                            result.PostPreview = match.Value.Trim()
                                + " " + Util.TextOf(range).Substring(match.Index + match.Length, endIndex - (match.Index + match.Length) + 1).Replace("\r\n", "↲ ").Trim('\r', '\n');
                        }
                    }

                    this._searchResultsMemo[new Uri(fileinfo.FullName)] = result;

                    AddItemToView(result);
                }
            }
        }

        private void AddItemToView(SearchResult result)
        {
            AddItemToListView(result);
            //AddItemToThumbnailView(result);
            this.wrapPanel1.Dispatcher.BeginInvoke(new Action(() => AddItemToThumbnailView(result)), DispatcherPriority.Background, null);

            //var tabItem = (TabItem)this.tabControl1.SelectedItem;

            //switch (tabItem.Name)
            //{
            //    case "listViewTab1":
            //        AddItemToListView(result);
            //        break;

            //    case "thumbnailViewTab1":
            //        AddItemToThumbnailView(result);
            //        break;

            //    default:
            //        Debug.Fail("BeginSearch");
            //        break;
            //}
        }

        private void AddItemToListView(SearchResult result)
        {
            //this._searchResults[result.GetHashCode()] = result;

            var items = (ObservableCollection<ListViewItem>)this.listView1.ItemsSource;

            var item = new ListViewItem() { Content = result };
            {
                //FIXME: シングルクリックに変更
                item.MouseDoubleClick += new MouseButtonEventHandler(ListViewItem_MouseDoubleClick);
                item.KeyDown += new KeyEventHandler(ListViewItem_KeyDown);
                item.Style = this._listViewItemStyles[items.Count % this._listViewItemStyles.Length];
            }

            items.Add(item);
        }

        private void AddItemToThumbnailView(SearchResult result)
        {
            // ThumbViewでは FoundPhraseNumber == 1 のSearchResultだけ表示
            var uri = new Uri(result.FileInfo.FullName);
            if (!this._searchResults.ContainsKey(uri))
            {
                this._searchResults[uri] = result;

                //TODO: スライダーで選択可能な範囲…の最大解像度に合わせて
                var scale = 0.5;

                var org = new Editor();
                var box = org.RichTextBox;

                using (box.DeclareChangeBlock())
                {
                    box.HorizontalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    box.VerticalScrollBarVisibility = ScrollBarVisibility.Hidden;
                    box.IsReadOnly = true;
                    box.IsReadOnlyCaretVisible = false;

                    //FIXME: RichTextBoxの大きさにしたい
                    box.Width = org.Width;
                    box.Height = org.Height;
                }

                // RichTextBox.Documentの設定は RichTextBox.IsReadOnly = true; のあとで。自動保存を防ぐため。
                var doc = box.Document = new FlowDocument();
                var range = new TextRange(doc.ContentStart, doc.ContentEnd);
                range.Load(result.FileInfo.OpenRead(), DataFormats.Rtf);

                //var bmpsource = FlowDocumentToBitmap(doc, new Size { Width = 1000, Height = 360 });
                //var image = new Image { Source = bmpsource, Width = 1000, Height = 360, Margin = new Thickness(5), Stretch = Stretch.UniformToFill, ClipToBounds = true };
                var bmpsource = FlowDocumentToBitmap(doc, new Size { Width = box.Width, Height = box.Height }, App.Ppi(this));
                var image = new Image { Source = bmpsource, Width = box.Width * scale, Height = box.Height * scale, Margin = new Thickness(5), Stretch = Stretch.UniformToFill, ClipToBounds = true };
                var item = new Frame();
                item.Content = image;
                //FIXME: シングルクリックに変更
                item.MouseDoubleClick += new MouseButtonEventHandler(Frame_MouseDoubleClick);
                item.Resources[typeof(Uri)] = new Uri(result.FileInfo.FullName);

                //var viewbox = new Viewbox();
                //viewbox.Child = image;
                //viewbox.ClipToBounds = true;
                //viewbox.Width = 120;
                //viewbox.Height = 180;
                //viewbox.Stretch = Stretch.UniformToFill;
                //this.wrapPanel1.Children.Add(viewbox);

                this.wrapPanel1.Children.Add(item);
            }
        }

        void ListViewItem_KeyDown(object sender, KeyEventArgs e)
        {
            var c = (ListViewItem)sender;
            var result = (SearchResult)c.Content;

            if (e.Key == Key.Return)
            {
                Debug.WriteLine(Path.Combine(App.DocumentPath.FullName, result.Name), "open");
                OpenNotes(new[] { result });
                e.Handled = true;
            }
        }

        void ListViewItem_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            var c = (ListViewItem)sender;
            var result = (SearchResult)c.Content;

            Debug.WriteLine(Path.Combine(App.DocumentPath.FullName, result.Name), "open");
            OpenNotes(new[] { result });
            e.Handled = true;
        }

        void Frame_MouseDoubleClick(object sender, MouseButtonEventArgs e)
        {
            var c = (Frame)sender;
            var result = this._searchResults[(Uri)c.Resources[typeof(Uri)]];

            Debug.WriteLine(Path.Combine(App.DocumentPath.FullName, result.Name), "open");
            OpenNotes(new[] { result });
            e.Handled = true;
        }

        void OpenNotes(IEnumerable<SearchResult> notes)
        {
            foreach (var note in notes)
            {
                //Process.Start(Path.Combine(basepath, note.Name));
                var editor = Editor.Create(new Uri(Path.Combine(App.DocumentPath.FullName, note.Name)), note);
                editor.SearchString = note.FoundPhrase;
                editor.Show();
                editor.Activate();

                if (editor.RichTextBox.Selection.Start.Paragraph != null)
                    editor.RichTextBox.Selection.Start.Paragraph.BringIntoView();
            }
        }

        public string SearchQuery
        {
            get { return this.comboBox1.Text; }
            set { this.comboBox1.Text = value; }
        }

        public BitmapSource FlowDocumentToBitmap(FlowDocument document, Size size, Size ppi)
        {
            document = CloneDocument(document);

            var paginator = ((IDocumentPaginatorSource)document).DocumentPaginator;
            paginator.PageSize = size;

            var visual = new DrawingVisual();
            //using (var drawingContext = visual.RenderOpen())
            //{
            //    // draw white background
            //    drawingContext.DrawRectangle(Brushes.White, null, new Rect(size));
            //}
            visual.Children.Add(paginator.GetPage(0).Visual);

            //FIXME: 適切な解像度に
            var bitmap = new RenderTargetBitmap((int)size.Width, (int)size.Height, ppi.Width, ppi.Height, PixelFormats.Pbgra32);
            bitmap.Render(visual);
            return bitmap;
        }

        public FlowDocument CloneDocument(FlowDocument document)
        {
            var copy = new FlowDocument();
            var sourceRange = new TextRange(document.ContentStart, document.ContentEnd);
            var targetRange = new TextRange(copy.ContentStart, copy.ContentEnd);

            using (var stream = new MemoryStream())
            {
                sourceRange.Save(stream, DataFormats.XamlPackage);
                targetRange.Load(stream, DataFormats.XamlPackage);
            }

            return copy;
        }

        private void TabControl_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            //foreach (var result in this._searchResultsMemo.Values)
            //{
            //    AddItemToView(result);
            //}
        }

        private void UpdateThumbnailView()
        {
            //this.wrapPanel1.Children.Clear();
            //this.wrapPanel1.Margin = new Thickness(10);

            //var callback = new Action<Regex, FileInfo>((patterns, fileinfo) =>
            //{
            //    //var patterns = (Regex)parameters[0];
            //    //var fileinfo = (FileInfo)parameters[1];

            //    Debug.WriteLine(new { fileinfo }, "Thumb *.join.rtf");

            //    try
            //    {
            //        var doc = new FlowDocument();
            //        var range = new TextRange(doc.ContentStart, doc.ContentEnd);
            //        using (var stream = fileinfo.OpenRead())
            //            range.Load(stream, DataFormats.Rtf);

            //        var range = patterns.Match(range.Text);

            //        if (range.Success)
            //        {
            //            var result = new SearchResult();
            //            {
            //                result.FileInfo = fileinfo;
            //                result.Name = fileinfo.Name;
            //                result.Modified = fileinfo.LastWriteTime.ToString();
            //                result.FoundPhrase = range.Value;
            //                result.FoundPhraseNumber = 1;
            //            }

            //            //this.wrapPanel1.Dispatcher.BeginInvoke(new Action(() =>
            //            //{
            //            AddItemToThumbnailView(result);
            //            //}), null);
            //        }
            //    }
            //    catch (IOException ex)
            //    {
            //        Debug.WriteLine(ex.Message, ex.Source);
            //    }
            //    catch (ArgumentException ex)
            //    {
            //        Debug.WriteLine(ex.Message, ex.Source);
            //    }
            //});

            //var canceltoken = CancellationToken.None;

            //var query = Regex.Escape(this.comboBox1.Text);
            //var patternParam = new Regex(query, RegexOptions.IgnoreCase | RegexOptions.Multiline);

            //var task = new Task(new Action(() =>
            //{
            //    _searchText(patternParam, callback);
            //}), canceltoken, TaskCreationOptions.None);

            //searchTask[canceltoken] = task;
            //task.Start();
        }

        private void UpdateListView()
        {
            //this.listView1.Items.Clear();

            //var callback = new Action<Regex, FileInfo>((patterns, fileinfo) =>
            //{
            //    //var patterns = (Regex)parameters[0];
            //    //var fileinfo = (FileInfo)parameters[1];

            //    Debug.WriteLine(new { fileinfo }, "List *.join.rtf");

            //    try
            //    {
            //        // ListView
            //        var doc = new FlowDocument();
            //        var range = new TextRange(doc.ContentStart, doc.ContentEnd);
            //        using (var stream = fileinfo.OpenRead())
            //        {
            //            //Debug.WriteLine(new StreamReader(stream).ReadToEnd(), "***** " + fileinfo.Name);
            //            range.Load(stream, DataFormats.Rtf);
            //        }

            //        var count = new Dictionary<string, int> { };

            //        var range = patterns.Match(range.Text);
            //        while (range.Success)
            //        {
            //            {
            //                var result = new SearchResult();
            //                {
            //                    result.FileInfo = fileinfo;
            //                    result.Name = fileinfo.Name;
            //                    result.Modified = fileinfo.LastWriteTime.ToString();
            //                    result.FoundPhrase = range.Value;

            //                    if (!count.ContainsKey(range.Value))
            //                        count[range.Value] = 0;
            //                    count[range.Value]++;
            //                    result.FoundPhraseNumber = count[range.Value];

            //                    var startIndex = Math.Max(0, range.Index - 10);
            //                    var endIndex = Math.Min(range.Text.Length - 1, range.Index + range.Length + 10);
            //                    result.Preview = range.Text.Substring(startIndex, endIndex - startIndex + 1).Replace("\r\n", "↲ ");
            //                }

            //                //this.listView1.Dispatcher.BeginInvoke(new Action(() =>
            //                //{
            //                AddItemToListView(result);
            //                //}), null);
            //            }
            //            range = patterns.Match(range.Text, range.Index + range.Length);
            //        }
            //    }
            //    catch (IOException ex)
            //    {
            //        Debug.WriteLine(new { ex.Message, fileinfo }, ex.Source);
            //    }
            //    catch (ArgumentException ex)
            //    {
            //        Debug.WriteLine(new { ex.Message, fileinfo }, ex.Source);
            //    }
            //});

            //var canceltoken = CancellationToken.None;

            //var query = Regex.Escape(this.comboBox1.Text);
            //var patternParam = new Regex(query, RegexOptions.IgnoreCase | RegexOptions.Multiline);

            //var task = new Task(new Action(() =>
            //{
            //    _searchText(patternParam, callback);
            //}), canceltoken, TaskCreationOptions.None);

            //searchTask[canceltoken] = task;
            //task.Start();
        }

        internal string SearchType { get; private set; }

        private void RadioButton_Checked(object sender, RoutedEventArgs e)
        {
            Debug.Assert(this.andRadioButton.IsChecked.HasValue && this.orRadioButton.IsChecked.HasValue && this.regexRadioButton.IsChecked.HasValue);
            if (this.andRadioButton.IsChecked.Value)
            {
                this.SearchType = "and";
            }
            else if (this.orRadioButton.IsChecked.Value)
            {
                this.SearchType = "or";
            }
            else if (this.regexRadioButton.IsChecked.Value)
            {
                this.SearchType = "regex";
            }
            else
            {
                Debug.Fail("searchType");
            }
        }

        private System.ComponentModel.ListSortDirection _listViewItemOrder = System.ComponentModel.ListSortDirection.Ascending;
        private void GridViewColumnHeader_Click(object sender, RoutedEventArgs e)
        {
            var c = (GridViewColumnHeader)sender;

            if (this.listView1.ItemsSource == null)
                return;

            var dataView = CollectionViewSource.GetDefaultView(this.listView1.ItemsSource);

            dataView.SortDescriptions.Clear();

            if (this._listViewItemOrder == System.ComponentModel.ListSortDirection.Ascending)
                this._listViewItemOrder = System.ComponentModel.ListSortDirection.Descending;
            else
                this._listViewItemOrder = System.ComponentModel.ListSortDirection.Ascending;
            dataView.SortDescriptions.Add(new System.ComponentModel.SortDescription("Content" + "." + c.Tag.ToString(), this._listViewItemOrder));
        }

        private void Window_PreviewKeyDown(object sender, KeyEventArgs e)
        {
            if ((e.Key == Key.W && e.KeyboardDevice.Modifiers == ModifierKeys.Control)
                || e.Key == Key.Escape && e.KeyboardDevice.Modifiers == ModifierKeys.None)
                this.Close();
        }

    }

    public class SearchResult
    {
        static public SearchResult Failed = new SearchResult() { Success = false };
        public bool Success { get; set; }
        public FileInfo FileInfo { get; set; }
        public string Name { get; set; }
        public string Modified { get; set; }
        public string PrePreview { get; set; }
        public string PostPreview { get; set; }
        public string FoundPhrase { get; set; }
        public int FoundPhraseNumber { get; set; }
    }
}
