#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# bottle と WebSocket を使用した､ﾏﾙﾁｽﾚｯﾄﾞAPｻｰﾊﾞｰです｡
#
# https://qiita.com/_x8/items/d52204b8c31b26c5c42e
# OpenCVとWebSocketで, リアルタイムで様子が見れるもの
#
#	$ pip3 install gevent
#	$ pip3 install gevent-websocket
#
# --module new_obj() ﾒｿｯﾄﾞを持つｵﾌﾞｼﾞｪｸﾄ
#    内部で、filter/module/subapp/callback をキーとする辞書にｵﾌﾞｼﾞｪｸﾄを
#    対応付けることで、filterﾁｪｰﾝを実現しています。
#

from gevent import sleep
from gevent.pywsgi import WSGIServer
from gevent.lock import Semaphore
from geventwebsocket import WebSocketError
from geventwebsocket.handler import WebSocketHandler

import base64
import json
import time
import signal

from datetime import datetime				# filter ｻﾝﾌﾟﾙで日時表示用

import argparse								# 引数の取り込み
from importlib import import_module			# 動的 import(ﾓｼﾞｭｰﾙを動的読み込み)

import bottle								# ｱﾌﾟﾘｹｰｼｮﾝｻｰﾊﾞｰ

from camera2 import Camera					# openCVｶﾒﾗ読取ﾏﾙﾁｽﾚｯﾄﾞｸﾗｽ

################################################
class WsSender() :
	def __init__(self):
		self.handler = None

		# https://stackoverflow.com/questions/31188874/why-is-gevent-websocket-synchronous
		self.lock = Semaphore()	# sleep(0.1) だと、handler.server.clients.values() が途中で書き換わる

	def setHandler(self,handler):
		self.handler = handler

	def sendMessage(self,msgJson):
		'''
			WebSocket に渡すﾒｯｾｰｼﾞ(json形式)を受け取ります。

			:param  msgJson : (json形式)のﾒｯｾｰｼﾞ
		'''
		if self.handler :
			# 同時書き込みを防止するため(ｲﾃﾚｰｼｮﾝ中の変更はｴﾗｰになる)
			with self.lock:
				msg = json.dumps(msgJson)
				# 画面ﾘﾛｰﾄﾞすると 通信が遮断されるが、ｵﾌﾞｼﾞｪｸﾄが残るので削除する。
				lst = list(self.handler.server.clients.items())
				for ip, cl in lst :				# cl は、client
					try :
						if cl.ws.environ:
							cl.ws.send( msg )
						else:
							print('切断')
					except WebSocketError as ex:
						print('更新')			# 画面更新(ﾘﾛｰﾄﾞ)すると接続が切れるので、Socket is dead ｴﾗｰが発生する。
	#					del self.handler.server.clients[ip]				# Ctrl_C の時、ｷｰが存在しないとｴﾗｰになる。
						self.handler.server.clients.pop(ip,None)		# 削除して、値の取り出し。None 指定でｷｰがなくてもｴﾗｰにならない。

########## ｻｰﾊﾞｰのIPｱﾄﾞﾚｽを求めます。 ##########
import socket
def get_ip_address():
	with socket.socket(socket.AF_INET, socket.SOCK_DGRAM) as s:
		s.connect(("8.8.8.8", 80))
		return s.getsockname()[0]

################################################
#def gen(camera,skip=0):
#	"""Video streaming generator function."""
#	cnt = 0
#	btimg = None
#
#	while True:
#		if cnt < skip and btimg is not None :			# 120ﾌﾚｰﾑ返すまで画面表示されない対策
#			cnt += 1									# 無限に数字が加算されるのを避けるため
#			yield btimg									# 意味不明だが、openCVのFPSが遅いので
#			continue									# 表示されるまでに数秒～数十秒かかるため
#
#		frame = camera.get_frame()
#		btimg = b'--frame\r\nContent-Type: image/jpeg\r\n\r\n' + frame.tobytes() + b'\r\n\r\n'
#
#		yield btimg
#		sleep(1/60)					# gevent.sleep なら、受信をﾛｯｸしない

################################################
def enc( key ) :
	return bottle.request.params.getunicode(key, encoding='utf-8') or ""

#############################################################################################
def sysExit(signo, frame):
	import sys
	print( "Ctl+C 終了" )
	sys.exit(0)

################################################
def main(argp):
	'''
		main関数

		:param argp		: ArgumentParserでﾊﾟｰｽ処理された引数
	'''
	signal.signal(signal.SIGINT, sysExit)			# Ctrl-C で終了 (sys.exit(0)で、無限ループのfinallyが呼ばれる)

	TITLE = 'openCV ｲﾒｰｼﾞｽﾄﾘｰﾐﾝｸﾞ'

	modules = {}									# ﾓｼﾞｭｰﾙは、辞書にして、action時に引数のｷｰでｵﾌﾞｼﾞｪｸﾄを渡す。
	closeable = set()								# 最後に close() 処理を行うｵﾌﾞｼﾞｪｸﾄを管理する set

	###### 変数の定義(ｴﾗｰ発生時に finallyで変数未定義を防ぐため) #####
	cam = None
	debug = argp.debug
	try :
		app = bottle.Bottle()						# bottle ｱﾌﾟﾘｹｰｼｮﾝｻｰﾊﾞｰ
		sender = WsSender()
		ip = get_ip_address()						# IPｱﾄﾞﾚｽ を取得(WebSocketのｻｰﾊﾞｰ指定用)

	#	テンプレートフォルダを viewsから変えたい場合(例えば、Flaskに合わせる…とか)
	#	bottle.TEMPLATE_PATH += ['./templates']

		# None 値でも指定すると、None値が渡されて、Camera の初期値は使われない。
		cam = Camera(devno=argp.devno,width=argp.width,height=argp.height,fps=argp.fps,fourcc=argp.fourcc)
		cam.info(debug=debug)

		###### module 処理(plug in) #####
		print( '  module={}'.format(argp.module) )
		for fin in argp.module :
			module = import_module(fin)				# filter名でﾓｼﾞｭｰﾙを動的読み込み
			modDic = module.new_obj(debug)			# ﾓｼﾞｭｰﾙ生成を行う new_obj(debug) は必須とする｡

			subapp = modDic.get( 'subapp' )
			if subapp : app.merge(subapp)			# bottle ｵﾌﾞｼﾞｪｸﾄをﾏｰｼﾞします

			filter = modDic.get( 'filter' )			# filterｵﾌﾞｼﾞｪｸﾄを取得(無ければ､None)
			if filter : cam.add_filter( filter )	# filter の close() は、Cameraｵﾌﾞｼﾞｪｸﾄ内で行われる。

			modobj = modDic.get( 'module' )
			if modobj :
				modkey = 'dummy'					# 仮の名前(template内でmoduleにｱｸｾｽできるｷｰ)
				if hasattr(modobj,'MOD_NAME') :
					modkey = modobj.MOD_NAME		# MOD_NAME が規程されていれば、その値を使う
				else :
					lst = fin.split('_')			# xxxx_module などの、xxx をｷｰにする。
					modkey = lst[0]
				modules[modkey] = modobj			# modkeyの値が、template 内で使用できる。
				closeable.add( modobj )

			callobj = modDic.get( 'callback' )
			if callobj :
				callobj.setCallback( sender.sendMessage )
				closeable.add( callobj )

		###### action ﾙｰﾃｨﾝｸﾞ処理 #####
		@app.route('/')
		@app.route('/<name>')		# nameは / を含まない。階層まで考慮するなら、<name:path> とします。
		def index(name='index'):	# nameの初期値設定が無いと、'/' でｴﾗｰになる。
			'''
				URLﾄｯﾌﾟと、action?gamen=XXXX による画面振り分けﾙｰﾃｨﾝｸﾞを行います。
				gamenのﾊﾟﾗﾒｰﾀが存在しないときは､初期値に､"index" を設定します｡

				:return : views/以下のtemplateの結果
			'''
			# /?action=XXXX で､振り分け画面を指定します｡
			url = bottle.request.params.get( "action",name )	# action 優先

			if url is None or len(url)==0 :
				url = 'index.html'								# index の拡張子は､'tpl', 'html', 'thtml', 'stpl' の何れか
#			elif url == 'stream' :								# /?action=stream でも､/stream でもｱｸｾｽ可
#				return stream()
			elif url == 'snapshot' :							# /?action=snapshot でも､/snapshot でもｱｸｾｽ可
				return snapshot()
			elif url == 'favicon.ico' :							# Edge の先読み機能？
				return ''										# ｱﾄﾞﾚｽ入力だけでﾘｸｴｽﾄされるが、存在しないｴﾗｰになる

			prmDic = {}
			for k in bottle.request.params.keys() :
				prmDic[k] = enc(k)

			params = dict( { "title":TITLE,"enc":enc,"IP":ip },**modules,**prmDic )	# title,enc,ﾓｼﾞｭｰﾙ,ﾊﾟﾗﾒｰﾀ を設定

			return bottle.template(url,params)					# ﾃﾝﾌﾟﾚｰﾄの初期ﾌｫﾙﾀﾞは､views

		###### static ﾙｰﾃｨﾝｸﾞ処理 #####
		# bottle の場合､static ﾙｰﾃｨﾝｸﾞ は､記述が必要｡Flask では不要
		@app.route('/static/<file:path>')
		def static( file ):
			'''
				static 以下は､静的ﾌｧｲﾙをそのまま返します。

				:return : static/以下の静的ﾌｧｲﾙ
			'''
			return bottle.static_file( file, root="./static" )

		###### WebSocket 処理 #####
		# https://doitu.info/blog/5aabc92d31e68500964d9255
		@app.route('/websocket')
		def handle_websocket() :
			print('handle_websocket')
			websocket = bottle.request.environ.get('wsgi.websocket')

			if not websocket:
				abort(400, 'Expected WebSocket request.')

			handler = websocket.handler
			sender.setHandler( handler )

			while True :
				img = cam.get_frame()		# ｼﾞｪﾈﾚｰﾀ(yieldの戻り値)を処理する
				ascimg = base64.b64encode(img).decode('ascii')
				msgJson = { 'video':ascimg }
				sender.sendMessage( msgJson )
				sleep(1/60)					# gevent.sleep なら、受信をﾛｯｸしない

			# handle_websocket ﾒｿｯﾄﾞを抜けると、すべての通信が終了する…ようだ。
			# なので、ｸﾗｲｱﾝﾄからのｱｸｾｽ分だけ、ここで止まっている…みたいだ。

#		###### stream ﾙｰﾃｨﾝｸﾞ処理 #####
#		@app.route('/stream')
#		def stream():
#			'''
#				Cameraｵﾌﾞｼﾞｪｸﾄの movie関数を呼び出します。
#
#				:return : mjpeg 動画生成のｼﾞｪﾈﾚｰﾀ
#			'''
#
#	#		resp = bottle.HTTPResponse(status=200)
#	#		resp.content_type = 'multipart/x-mixed-replace;boundary=frame'
#	#		resp.body = gen(cam)
#	#		return resp
#
#			bottle.response.content_type = 'multipart/x-mixed-replace;boundary=frame'
#			return gen(cam,argp.skip)

		###### snapshot ﾙｰﾃｨﾝｸﾞ処理 #####
		@app.route('/snapshot')
		def snapshot():
			'''
				Cameraｵﾌﾞｼﾞｪｸﾄの snap関数を呼び出します。

				:return : jpeg 静止画
			'''
			bottle.response.content_type = 'image/jpeg'
			return cam.get_frame().tobytes()

		###### ｻｰﾊﾞｰ起動(bottle を引数に､waitress を起動) #####
		# ※ ﾏﾙﾁｽﾚｯﾄﾞでないと、ajaxが処理されない？のか、動かなくなる。
		# app.run(host='localhost', port=8080, reloader=True, debug=True)
		# app.run(host='0.0.0.0'  , port=8088, reloader=True, debug=True)
		# waitress では、debug は、logging を使うみたい。
		#          , host=制限なし , port=8088     ,threads=20
		# serve( app , host='0.0.0.0', port=argp.port,threads=argp.threads )	# threadsの初期値は 4
		server = WSGIServer(("0.0.0.0", 8088), app, handler_class=WebSocketHandler)
		server.serve_forever()

	###### Ctl+C,Exception,finallyでclose処理 #####
	except KeyboardInterrupt  : 				# Ctl+Cが押されたらﾙｰﾌﾟを終了
		print( "\nCtl+C Stop" )
	except Exception as ex:
		print( datetime.today().strftime( '%Y/%m/%d %H:%M:%S' ) )
		print( ex )								# 例外処理の内容をｺﾝｿｰﾙに表示
		import traceback
		traceback.print_exc()					# Exception のﾄﾚｰｽ
	finally :
		if cam : cam.close()					# filter の close() は、cam で実行

		for obj in closeable :
			# close 属性を持ち、実行可能な場合のみ、close 処理を行います。
			if hasattr(obj,'close') and callable(obj.close) :
				obj.close()

		print( "main 終了" )

################################################
# main関数を呼び出します
################################################
if __name__=='__main__':
	# add_argument_group とか、parents=[parser1,parser2,parser3] とか、
	# 色々がんばりましたが、動的組み込みﾓｼﾞｭｰﾙのﾊﾟﾗﾒｰﾀ設定が
	# うまくできませんでした。

	parser = argparse.ArgumentParser(description='webSocket AP Server')
	group = parser.add_mutually_exclusive_group()			# 排他ｸﾞﾙｰﾌﾟ
	group.add_argument('-d','--devno'	, type=int, default=0   ,help='ｶﾒﾗのﾃﾞﾊﾞｲｽﾎﾟｰﾄ番号')
#	group.add_argument('--skip'			, type=int, default=120 ,help='ｶﾒﾗ表示遅延対策(120)')			# gen ｼﾞｪﾈﾚｰﾀ使用時用

	parser.add_argument('--port'		, type=int, default=8088 , help='Web Server http port(8088)')
#	parser.add_argument('--threads'		, type=int, default=20   , help='waitress Server threads(20)')	# 設定できない

	parser.add_argument('-w','--width'	, type=int, help='openCV FRAME WIDTH')
	parser.add_argument('-v','--height'	, type=int, help='openCV FRAME HEIGHT')			# -h は --help が自動的に設定されている。
	parser.add_argument('-f','--fps'	, type=int, help='openCV FRAME FPS')
	parser.add_argument('--fourcc'		, type=str, help='openCV FOURCC (MJPG,YUYV,RGB3…)')
	parser.add_argument('-m','--module',nargs='+', default=[],help='filter,module,subapp')
	# --debug を記述した場合にだけ、True になる。
	parser.add_argument('--debug' , action='store_true', help='Debug mode')
	argp = parser.parse_args()

	if argp.debug :
		print( '  {}'.format( parser.format_usage() ) )		# コマンドラインの短い説明
		print( '  {}'.format( argp ) )						# 設定された値(Namespace) vars(argp) で辞書に出来る
	#	print( '  {}'.format( vals(argp) ) )				# vars(argp) で辞書に出来る

	main(argp)
