#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ﾏﾙﾁｽﾚｯﾄﾞｻｰﾊﾞｰ用 openCV ｽﾄﾘｰﾐﾝｸﾞ本体

# ≪参考≫
# https://qiita.com/RIckyBan/items/a7dea207d266ef835c48
# FlaskとOpenCVでカメラ画像をストリーミングして複数ブラウザでアクセスする
#   今回のｿｰｽの参照元であり、ﾏﾙﾁｽﾚｯﾄﾞでの最終形の解説がされています。
#   本体ｿｰｽでは継承とかstaticmethod、classmethod を使用していましたが、
#   ｲﾝｽﾀﾝｽﾒｿｯﾄﾞ化とﾌｨﾙﾀｰ処理を組み込めるように改造しています。

import time
import cv2

################################################
class Camera(object):
	################################################
	def __init__(self,devno=0,width=640,height=480,fps=None,fourcc='MJPG'):
		'''
			ｺﾝｽﾄﾗｸﾀ

			:param devno=0		: VideoCaptureするﾃﾞﾊﾞｲｽﾎﾟｰﾄ番号
			:param width=640	: ﾌﾚｰﾑに設定する幅
			:param height=480	: ﾌﾚｰﾑに設定する高さ
			:param fps=None		: ﾌﾚｰﾑﾚｰﾄ (数値)
			:param fourcc='MJPG': ﾌｫｰﾏｯﾄ(ｺｰﾃﾞｨｯｸ)指定
		'''
		try :
			cap = cv2.VideoCapture(devno,cv2.CAP_V4L2)				# V4L2から直接キャプチャ
		except TypeError:
			cap = cv2.VideoCapture(devno)

		if not cap.isOpened():
			raise RuntimeError('Could not start camera.')

		cap.set(cv2.CAP_PROP_BUFFERSIZE, 1)							# バッファサイズ(ﾃﾞﾌｫﾙﾄ：4)
#		cap.set(cv2.CAP_PROP_CONVERT_RGB, 0.0)						# BGR形式に変換せずに直接受け取る
		# ↑これを指定すると、jpg変換時にエラーになる。

		if width :
			cap.set(cv2.CAP_PROP_FRAME_WIDTH ,width )				# 画像の幅

		if height :
			cap.set(cv2.CAP_PROP_FRAME_HEIGHT ,height )				# 画像の高さ

		if fps :
			cap.set(cv2.CAP_PROP_FPS ,fps )							# ﾌﾚｰﾑﾚｰﾄ

		if fourcc :
			cap.set(cv2.CAP_PROP_FOURCC, cv2.VideoWriter_fourcc(*fourcc) )	# フォーマット
	#		cap.set(cv2.CAP_PROP_FOURCC, cv2.VideoWriter_fourcc( *'MJPG'))

		self.devno   = devno
		self.camera  = cap
		self.encimg  = None	# current frame is stored here by background thread
		self.filters = []	# 画像にﾌｨﾙﾀﾘﾝｸﾞを行います。

	################################################
	def _filter(self,image):
		""" ﾌｨﾙﾀﾘﾝｸﾞ処理…配列の順番にﾌｨﾙﾀ処理を掛けていきます """
		for fil in self.filters :
			image = fil.filter(image)

		return image

	################################################
	# ﾌｨﾙﾀｵﾌﾞｼﾞｪｸﾄ は、close()ﾒｿｯﾄﾞを持つ事を期待します。
	def add_filter(self,filter):
		""" ﾌｨﾙﾀｵﾌﾞｼﾞｪｸﾄを追加します """
		if filter is not None :					# filter に 数字の 0 は来ないので、if filter : でよいかも
			self.filters.append( filter )

	################################################
	def get_frame(self):
		ret,frame = self.camera.read()
		if ret :
			frame = self._filter(frame)
			_, self.encimg = cv2.imencode('.jpg', frame)	#jpgに変換
			return self.encimg
		else :
			return self.encimg

	################################################
	def info(self,debug=False):
		'''
			画像情報を表示(print)します

			:param debug=False  : ﾃﾞﾊﾞｯｸﾞ情報を表示する場合は、True
		'''

		width	= int( self.camera.get(cv2.CAP_PROP_FRAME_WIDTH) )		# 画像の幅
		height	= int( self.camera.get(cv2.CAP_PROP_FRAME_HEIGHT))		# 画像の高さ
		fps		= int( self.camera.get(cv2.CAP_PROP_FPS) )				# ﾌﾚｰﾑﾚｰﾄ

		ccint	= int( self.camera.get(cv2.CAP_PROP_FOURCC) )			# ﾊﾞｲﾄの数字列
		fourcc	= ccint.to_bytes(4, 'little').decode('utf-8')

		print( '  WIDTH={} HEIGHT={} FPS={} FOURCC={}'.format( width,height,fps,fourcc ) )

		if debug :
			import subprocess
			cmd1 = "v4l2-ctl -d {} --list-formats-ext | grep -e 'Size' -e ']:' -e 'fps'".format(self.devno)
			cmd2 = r" | sed -e 's/ Discrete//g' -e 's/\t/ /g' -e 's/Interval.*(/(/g'"
			cmd3 = r" | sed -z 's/\n//g' | sed 's/)/)\n/g'"
			cmd = cmd1 + cmd2 + cmd3
			ps = subprocess.Popen(cmd,shell=True,stdout=subprocess.PIPE,stderr=subprocess.PIPE)
			outs, errs = ps.communicate()

			if len(errs) == 0 :								# エラーが存在しない場合
				print( '{}'.format(outs.decode()))			# bytes → str

	################################################
	def __del__(self):
		self.close()
		print('Camera release')

	################################################
	def close(self):
		if self.camera is not None : self.camera.release()
		for fil in self.filters :
			# close 属性を持ち、実行可能な場合のみ、close 処理を行います。
			if hasattr(fil,'close') and callable(fil.close) :
				fil.close()
		self.filters = []

################################################
# 動作確認用のテストプログラムです。
def main(argDic):
	import numpy as np							# jpegのﾊﾞｲﾄ画像変換時に使用

	fname = argDic.get('fname')					# ﾌﾟﾛｸﾞﾗﾑ名=Windows画面名
	devno = int(argDic.get('devno'))			# ﾃﾞﾊﾞｲｽNoの設定

	###### 変数の定義(ｴﾗｰ発生時に finallyで変数未定義を防ぐため) #####
	cam = None

	try :
		cam = Camera(devno=devno)
		cam.info()

		while True:
			jpg = cam.get_frame()				# cam から取り出すのは、jpeg画像のｼﾞｪﾈﾚｰﾀ
			img = cv2.imdecode(jpg, cv2.IMREAD_UNCHANGED)		# 画像をデコードする

			cv2.imshow(fname,img)

			key = cv2.waitKey(10)				# ミリ秒単位で表される遅延時間
			if key == 27:						# ESCを押した時
				break

			time.sleep( 0.1 )					# CPU処理の軽減化

	###### Ctl+C,Exception,finallyでclose処理 #####
	except KeyboardInterrupt  : 				# Ctl+Cが押されたらﾙｰﾌﾟを終了
		print( "\nCtl+C Stop" )
	except Exception as ex:
		print( ex )								# 例外処理の内容をｺﾝｿｰﾙに表示
		import traceback
		traceback.print_exc()					# Exception のﾄﾚｰｽ
	finally :
		if cam : cam.close()					# filter の close() は、cam で実行

		print( "main 終了" )

################################################
# main関数を呼び出します
################################################
if __name__=='__main__':
	print( 'python3 camera2.py [devno(=0)]' )

	argDic = {'fname':None , 'devno':'0'}

	import sys								# 引数の読み込み用
	args = sys.argv
	argDic['fname'] = args[0]
	if len(args) > 1 : argDic['devno'] = args[1]

	main(argDic)
