#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import time
import cv2
import os									# ﾀﾞﾝﾌﾟﾌｧｲﾙの存在ﾁｪｯｸ
import pickle								# ﾀﾞﾝﾌﾟのｼﾘｱﾗｲｾﾞｰｼｮﾝ

from datetime import datetime

import numpy as np

WHITE	= (255, 255, 255)					# BGR形式
BLACK	= (  0,   0,   0)					# BGR形式(黒)
RED		= (  0,   0, 255)					# BGR形式(赤)
YELLOW	= (  0, 255, 255)					# BGR形式(黄)

################################################
class Mark2Filter :
	#############################################################################################
	def __init__(self):
		'''
			ｺﾝｽﾄﾗｸﾀ
		'''

		self.rect	= []					# 選択された矩形(actDic)の配列
		self.dump = './mark.dump'			# ﾀﾞﾝﾌﾟﾌｧｲﾙ名
		self.thresh	= 128					# 通常の二値化のしきい値
		self.isOTSU	= False					# False:通常の二値化 / True:OTSUの二値化

		self.read()							# 起動時にｴﾘｱを読み込む

	#############################################################################################
	# action から呼び出されるﾒｿｯﾄﾞ
	def wheel_action(self, actDic):
		# 'wheel' ｷｰが有っても、値がNoneの場合は、get でもNoneが返る。
		wheel = actDic.get('wheel')

		if wheel is not None :
			step = int(wheel)
			whX  = int(actDic.get('whX','0'))			# ﾎｲｰﾙを回した位置。今は未使用
			whY  = int(actDic.get('whY','0'))			# その場を中心に拡大縮小など出来る。

			if step > 0 and self.thresh < 255:
				self.thresh += 1
			elif step < 0 and self.thresh > 0  :
				self.thresh -= 1

			print( 'thresh=' + str(self.thresh) )

	#############################################################################################
	# bottle からﾙｰﾃｨﾝｸﾞされるﾒｿｯﾄﾞ
	def action(self, actDic):
		'''
			bottleｻｰﾊﾞｰから /control ﾘｸｴｽﾄ時の処理

			:param actDic : {'width':'画像幅','height':'画像高さ',
								'stX':'開始X','stY':'開始Y','edX':'終了X','edY':'終了Y',} 形式の辞書
		'''

		if 'wheel' in actDic :					# wheel ｷｰがあれば、wheel処理を行います。
			self.wheel_action(actDic)
			return

		# 開始-終了位置が逆転している場合は、初期化する
		stX = int(actDic.get('stX','0'))
		stY = int(actDic.get('stY','0'))
		edX = int(actDic.get('edX','0'))
		edY = int(actDic.get('edY','0'))

		if stX < edX and stY < edY :
			self.rect.append( actDic )
		else :
			delList = []
			for dic in self.rect :
				if dic['stX'] < stX < dic['edX'] and dic['stY'] < stY < dic['edY'] :
					delList.append(dic)

			cnt = len(delList)
			if cnt == 0 :
				self.rect.clear()
			else :
				for i in range(cnt) :
					self.rect.remove(delList[i])

	#############################################################################################
	# filterｸﾗｽとしては必須ﾒｿｯﾄﾞ
	def filter(self, image):
		'''
			openCvの image を入力して、加工処理を行い、image を返します。

			:param  image : 入力ｲﾒｰｼﾞ
			:return	:	加工処理が行われたｲﾒｰｼﾞ
		'''
		try :
			self._putTime(image)						# 時刻表示

			if len(self.rect) > 0 :
				for areaDic in self.rect :
					h,w,c = image.shape[:3]				# height, width, channels
					fx = areaDic['width']  / w			# 原画とhtml表示ｲﾒｰｼﾞとの倍率
					fy = areaDic['height'] / h			#

					h1 = int(areaDic['stY'] / fy )		# 縦方向（上）
					h2 = int(areaDic['edY'] / fy )		# 縦方向（下）
					w1 = int(areaDic['stX'] / fx )		# 横方向（左）
					w2 = int(areaDic['edX'] / fx )		# 横方向（右）

					cv2.rectangle( image,(w1,h1),(w2,h2),RED,2 )

					img = image[h1:h2,w1:w2]
					img = self._gray(img)				# 二値化のフィルター処理
					image[h1:h2,w1:w2] = img

			return image

		except Exception as ex:
			print( 'filter 実行中に例外が発生しました｡' )
			raise ex											# 例外を投げる｡

	#############################################################################################
	# 日時文字列書き込み処理
	def _putTime(self,img,color=BLACK):
		now = datetime.today()
		ymd = now.strftime( '%Y/%m/%d %H:%M:%S' )			# 年/月/日 時:分:秒

		# 本来は文字の表示領域のﾋﾟｸｾﾙを求めて計算すべきだが…
		h,w,c = img.shape[:3]								# height, width, channels
		fsize = w/3/180										# 180pxが､ｻｲｽﾞ 1 基準で､1/3領域に表示する(決め打ち)
		hpos  = int(20*fsize)								# 表示位置は､ｻｲｽﾞ 1 基準で､20px で計算する(決め打ち)

		# 時間表示	画像,文字,座標(x,y),ﾌｫﾝﾄ                  ,ｻｲｽﾞ ,色    ,太さ,線の種類
		cv2.putText(img ,ymd ,(0,hpos) ,cv2.FONT_HERSHEY_PLAIN,fsize,WHITE ,3   ,cv2.LINE_AA)	# 少し太い文字で背景を書く
		cv2.putText(img ,ymd ,(0,hpos) ,cv2.FONT_HERSHEY_PLAIN,fsize,color ,2   ,cv2.LINE_AA)	# その中に､本来の文字を書けば見やすくなる｡

	#############################################################################################
	# 2値化処理
	def _gray(self, img):
		img_gry = cv2.cvtColor(img, cv2.COLOR_BGR2GRAY)		# GRAY 形式に変更

		if self.isOTSU :
			ret, img_th = cv2.threshold(img_gry,0,255,cv2.THRESH_OTSU)
		else :
			ret, img_th = cv2.threshold(img_gry, self.thresh, 255, cv2.THRESH_BINARY)

		img = cv2.cvtColor(img_th, cv2.COLOR_GRAY2BGR)			# ｶﾗｰに戻す

		cv2.putText(img ,str(ret) ,(0,22) ,cv2.FONT_HERSHEY_PLAIN,1,YELLOW,2,cv2.LINE_AA)

		return img

	#############################################################################################
	def read(self):
		'''
			ﾀﾞﾝﾌﾟﾌｧｲﾙの読み込み（ﾛｰｶﾙ）
			ﾀﾞﾝﾌﾟﾌｧｲﾙを読み込んで､内部のｴﾘｱ情報に値をｾｯﾄします｡
		'''

		# ﾘｽﾄﾌｧｲﾙが存在すれば､読み出す
		if os.path.exists( self.dump ) :
			with open( self.dump,'rb' ) as f :
				self.rect = pickle.load( f )

			print( 'ｴﾘｱ情報をﾌｧｲﾙから読み込みました｡file={:s}'.format( self.dump ) )

	#############################################################################################
	def write(self):
		'''
			ﾀﾞﾝﾌﾟﾌｧｲﾙの書き込み
			内部のｴﾘｱ情報をﾀﾞﾝﾌﾟﾌｧｲﾙに書き込みます｡

		'''
		with open( self.dump, 'wb' ) as f :
			pickle.dump( self.rect, f )

		print( 'ｴﾘｱ情報をﾌｧｲﾙに書き込みました｡file={:s}'.format( self.dump ) )

	#############################################################################################
	def change(self):
		'''
			OTSUの二値化への設定On/Off(トグル処理)

		'''
		self.isOTSU = not self.isOTSU

		print( 'OTSU の二値化:{}'.format( self.isOTSU ) )

	#############################################################################################
	# 終了時に呼ばれるﾒｿｯﾄﾞ
	def close(self):
		'''
			内部変数を初期化します。
		'''
		self.rect.clear()					# 初期化

################################################
# ogServer.py で、bottle/flask に組み込むためのﾌｯｸ
def new_obj():
	'''
		filter/module/subapp処理を行うｸﾗｽのｲﾝｽﾀﾝｽを生成します｡

		:return	:	(filterｲﾝｽﾀﾝｽ,moduleｲﾝｽﾀﾝｽ,subappｲﾝｽﾀﾝｽ)の辞書
	'''
	import json									# ajaxﾃﾞｰﾀ
	import bottle								# ｱﾌﾟﾘｹｰｼｮﾝｻｰﾊﾞｰ

	mark = Mark2Filter()
	subapp = bottle.Bottle()

	###### ajax 通信 #####
	@subapp.route('/mark' , method='POST')		# bottleは､『method』
	def markAction():
		# /mark ﾘｸｴｽﾄ時に送られてきた JSON形式を辞書に変換
		actDic = json.load(bottle.request.body)
		mark.action(actDic)
		return ''								# 非同期通信なので､処理完了時点で値は拾えない

	@subapp.route('/write' , method='GET')		# bottleは､『method』
	def writeAction():
		mark.write()
		return ''								# 非同期通信なので､処理完了時点で値は拾えない

	@subapp.route('/change' , method='GET')		# bottleは､『method』
	def changeAction():
		mark.change()
		return ''								# 非同期通信なので､処理完了時点で値は拾えない

	return {'filter':mark,'subapp':subapp }

################################################
# mjpg-streamer の input_opencv.so filterの 互換処理
#def init_filter():
#	'''
#		filter処理を行うﾒｿｯﾄﾞを返します。
#
#		:return	:	filterﾒｿｯﾄﾞ
#	'''
#	mark = Mark2Filter()
#	return mark.filter

################################################
class MouseClick :
	def __init__(self,fname,width,height,action):
		self.stX = 0						# ｸﾘｯｸ時のX値
		self.stY = 0						# ｸﾘｯｸ時のY値
		self.action = action				# actionﾒｿｯﾄﾞ
		self.width  = width					# 画像の幅
		self.height = height				# 画像の高さ

		cv2.namedWindow(fname)						# Window名を指定しないと、ｲﾍﾞﾝﾄが発生しない
		cv2.setMouseCallback(fname,self.mouseClick)	# ﾏｳｽｲﾍﾞﾝﾄのCallback関数登録

	def mouseClick(self,event,x,y,flags,param):
		if event == cv2.EVENT_LBUTTONDOWN:			# 左ﾎﾞﾀﾝを押す
			print('DOWN x={},y={}'.format(x,y))
			self.stX = x							# X座標(開始点)
			self.stY = y							# Y座標(開始点)
		elif event == cv2.EVENT_LBUTTONUP:			# 左ﾎﾞﾀﾝを離す
			print('UP   x={},y={}'.format(x,y))
			actDic = {	'width' :self.width	,
						'height':self.height,
						'stX':self.stX ,
						'stY':self.stY ,
						'edX':x ,					# X座標(終了点)
						'edY':y						# Y座標(終了点)
					}
			self.action(actDic)
	#	elif event == cv2.EVENT_MOUSEMOVE:			# 座標取得
	#		print('MOVE   x={},y={}'.format(x,y))
		elif event == cv2.EVENT_MOUSEWHEEL:			# ﾏｳｽﾎｲｰﾙ CLAHEのlimitの上げ下げを行う
			flag = 1 if flags > 0 else -1
			print('WHEL x={},y={},flag={}'.format(x,y,flag))
			actDic = {	'wheel' :flag	,
						'width' :self.width	,
						'height':self.height,
						'whX':x ,					# X座標(ﾏｳｽの位置)
						'whY':y						# Y座標(ﾏｳｽの位置)
					}
			self.action(actDic)

################################################
# 動作確認用のテストプログラムです。
def main(argDic):
	mark = Mark2Filter()							# ｵﾌﾞｼﾞｪｸﾄ生成

	fname = argDic.get('fname')					# ﾌﾟﾛｸﾞﾗﾑ名=Windows画面名
	devno = int(argDic.get('devno'))			# ﾃﾞﾊﾞｲｽNoの設定

	cap = cv2.VideoCapture(devno)
	width	= int( cap.get(cv2.CAP_PROP_FRAME_WIDTH)  )	# 画像の幅
	height	= int( cap.get(cv2.CAP_PROP_FRAME_HEIGHT) )	# 画像の高さ

	MouseClick(fname,width=width,height=height,action=mark.action)

	try:
		while True:
			_,img = cap.read()
			img = mark.filter(img)
			cv2.imshow(fname,img)

			key = cv2.waitKey(10)				# ミリ秒単位で表される遅延時間
			if key == 27:						# ESCを押した時
				break

			time.sleep( 0.1 )					# CPU処理の軽減化

	except KeyboardInterrupt  : 				# Ctl+Cが押されたらループを終了
		print( "\nCtl+C Stop" )
	except Exception as ex:
		print( ex )								# 例外処理の内容をコンソールに表示
		import traceback
		traceback.print_exc()					# Exception のトレース
	finally :
		if cap is not None : cap.release()
		cv2.destroyAllWindows()
		print( "終了" )

################################################
# main関数を呼び出します
################################################
if __name__=='__main__':
	print( 'python mark_filter.py [devno(=0)]' )

	argDic = {'fname':None , 'devno':'0'}

	import sys								# 引数の読み込み用
	args = sys.argv
	argDic['fname'] = args[0]
	if len(args) > 1 : argDic['devno'] = args[1]

	main(argDic)
