#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# ﾏﾙﾁｽﾚｯﾄﾞｻｰﾊﾞｰ用 openCV ｽﾄﾘｰﾐﾝｸﾞ本体

# ≪参考≫
# https://qiita.com/RIckyBan/items/a7dea207d266ef835c48
# FlaskとOpenCVでカメラ画像をストリーミングして複数ブラウザでアクセスする
#   今回のｿｰｽの参照元であり、ﾏﾙﾁｽﾚｯﾄﾞでの最終形の解説がされています。
#   本体ｿｰｽでは継承とかstaticmethod、classmethod を使用していましたが、
#   ｲﾝｽﾀﾝｽﾒｿｯﾄﾞ化とﾌｨﾙﾀｰ処理を組み込めるように改造しています。

import time
import threading
import cv2

################################################
class CameraEvent(object):
	"""An Event-like class that signals all active clients when a new frame is
	available.
	"""
	def __init__(self):
		self.events = {}

	def wait(self):
		"""Invoked from each client's thread to wait for the next frame."""
		ident = threading.get_ident()
		if ident not in self.events:
			# this is a new client
			# add an entry for it in the self.events dict
			# each entry has two elements, a threading.Event() and a timestamp
			self.events[ident] = [threading.Event(), time.time()]
		return self.events[ident][0].wait()

	def set(self):
		"""Invoked by the camera thread when a new frame is available."""
		now = time.time()
#		remove = None
		remove = []
		for ident, event in self.events.items():
			if not event[0].isSet():
				# if this client's event is not set, then set it
				# also update the last set timestamp to now
				event[0].set()
				event[1] = now
			else:
				# if the client's event is already set, it means the client
				# did not process a previous frame
				# if the event stays set for more than 5 seconds, then assume
				# the client is gone and remove it
				if now - event[1] > 5:
#					remove = ident
					remove.append(ident)
#		if remove:
#			del self.events[remove]
		for ident in remove:
			del self.events[ident]

	def clear(self):
		"""Invoked from each client's thread after a frame was processed."""
#		self.events[threading.get_ident()][0].clear()
		ident = threading.get_ident()
		if ident in self.events:
			self.events[ident][0].clear()

################################################
class Camera(object):
	def __init__(self,devno=0,width=640,height=480):
		'''
			ｺﾝｽﾄﾗｸﾀ

			:param devno=0		: VideoCaptureするﾃﾞﾊﾞｲｽﾎﾟｰﾄ番号
			:param width=640	: ﾌﾚｰﾑに設定する幅
			:param height=480	: ﾌﾚｰﾑに設定する高さ
		'''
		self.camera = cv2.VideoCapture(devno)
		if not self.camera.isOpened():
			raise RuntimeError('Could not start camera.')

		if width :
			self.camera.set(cv2.CAP_PROP_FRAME_WIDTH ,width )				# 画像の幅

		if height :
			self.camera.set(cv2.CAP_PROP_FRAME_HEIGHT ,height )				# 画像の高さ

		self.thread = None  # background thread that reads frames from camera
		self.frame  = None  # current frame is stored here by background thread
		self.last_access = 0  # time of last client access to the camera
		self.event = CameraEvent()
		self.filters = []	# 画像にﾌｨﾙﾀﾘﾝｸﾞを行います。

	def _filter(self,image):
		""" ﾌｨﾙﾀﾘﾝｸﾞ処理…配列の順番にﾌｨﾙﾀ処理を掛けていきます """
		for fil in self.filters :
			image = fil.filter(image)

		return image

	# ﾌｨﾙﾀｵﾌﾞｼﾞｪｸﾄ は、close()ﾒｿｯﾄﾞを持つ事を期待します。
	def add_filter(self,filter):
		""" ﾌｨﾙﾀｵﾌﾞｼﾞｪｸﾄを追加します """
		if filter is not None :
			self.filters.append( filter )

	# ｶﾒﾗｽﾚｯﾄﾞが停止した後、復活させる処理(独立させてﾌﾚｰﾑ取得時にﾁｪｯｸします)
	def _new_thread(self) :
		"""Start the background camera thread if it isn't running yet."""
		# start background frame thread
		self.thread = threading.Thread(target=self._thread)
		self.thread.setDaemon(True)		# 追加：Ctrl-C でスレッドが終わらなかったので。
		self.thread.start()

	def get_frame(self):
		"""Return the current camera frame."""
		self.last_access = time.time()

		if self.thread is None:
			self._new_thread()

		# wait for a signal from the camera thread
		self.event.wait()
		self.event.clear()

		return self.frame

	# 元 @staticmethod
	def frames(self):
		""""Generator that returns frames from the camera."""
		while True:
			# read current frame
			ret,frame = self.camera.read()
			if ret :
				frame = self._filter(frame)

				# encode as a jpeg image and return it
				yield cv2.imencode('.jpg', frame)[1].tobytes()
			time.sleep(1/60)

	# 元 @classmethod
	def _thread(self):
		"""Camera background thread."""
		print('Starting camera thread.')
		frames_iterator = self.frames()
		for frame in frames_iterator:
			self.frame = frame
			self.event.set()	# send signal to clients
			time.sleep(0)

			# if there hasn't been any clients asking for frames in
			# the last 10 seconds then stop the thread
			if time.time() - self.last_access > 10:
				frames_iterator.close()
				print('Stopping camera thread due to inactivity.')
				break
		self.thread = None

	def info(self):
		''' 画像情報を表示(print)します '''

		width	= int( self.camera.get(cv2.CAP_PROP_FRAME_WIDTH) )		# 画像の幅
		height	= int( self.camera.get(cv2.CAP_PROP_FRAME_HEIGHT))		# 画像の高さ
		fps		= int( self.camera.get(cv2.CAP_PROP_FPS) )				# ﾌﾚｰﾑﾚｰﾄ
		print( '  WIDTH={} HEIGHT={} FPS={}'.format( width,height,fps ) )

	def __del__(self):
		self.close()
		print('Camera release')

	def close(self):
		if self.camera is not None : self.camera.release()
		for fil in self.filters :
			# close 属性を持ち、実行可能な場合のみ、close 処理を行います。
			if hasattr(fil,'close') and callable(fil.close) :
				fil.close()
		self.filters = []

################################################
# 動作確認用のテストプログラムです。
def main(argDic):
	import numpy as np							# jpegのﾊﾞｲﾄ画像変換時に使用

	fname = argDic.get('fname')					# ﾌﾟﾛｸﾞﾗﾑ名=Windows画面名
	devno = int(argDic.get('devno'))			# ﾃﾞﾊﾞｲｽNoの設定

	###### 変数の定義(ｴﾗｰ発生時に finallyで変数未定義を防ぐため) #####
	cam = None

	try :
		cam = Camera(devno=devno)
		cam.info()

		while True:
			jpg = cam.get_frame()				# cam から取り出すのは、jpegのﾊﾞｲﾄ画像
			img_buf= np.frombuffer(jpg, dtype=np.uint8)			# bufferをndarrayに高速変換
			img = cv2.imdecode(img_buf, cv2.IMREAD_UNCHANGED)	# 画像をデコードする

			cv2.imshow(fname,img)

			key = cv2.waitKey(10)				# ミリ秒単位で表される遅延時間
			if key == 27:						# ESCを押した時
				break

			time.sleep( 0.1 )					# CPU処理の軽減化

	###### Ctl+C,Exception,finallyでclose処理 #####
	except KeyboardInterrupt  : 				# Ctl+Cが押されたらﾙｰﾌﾟを終了
		print( "\nCtl+C Stop" )
	except Exception as ex:
		print( ex )								# 例外処理の内容をｺﾝｿｰﾙに表示
		import traceback
		traceback.print_exc()					# Exception のﾄﾚｰｽ
	finally :
		if cam : cam.close()					# filter の close() は、cam で実行

		print( "main 終了" )

################################################
# main関数を呼び出します
################################################
if __name__=='__main__':
	print( 'python camera.py [devno(=0)]' )

	argDic = {'fname':None , 'devno':'0'}

	import sys								# 引数の読み込み用
	args = sys.argv
	argDic['fname'] = args[0]
	if len(args) > 1 : argDic['devno'] = args[1]

	main(argDic)
