/*
 * cppds
 * kdtree.h
 * KDTree class
 * Copylight (C) 2008 mocchi
 * mocchi_2003@yahoo.co.jp
 * License: Boost ver.1
 */

#include <cstdlib>
#include <stack>
#include <list>
#include <vector>
#include <cmath>
#include <limits>
#include <map>
#include <algorithm>

#ifndef NOMINMAX
#define NOMINMAX
#endif

namespace cppds{
template <typename T, int N, typename P = double> class kdtree{
	static const int DIRSTART = 0;
	struct KDNode{
		P pos[N];
		const T value;
		KDNode *left, *right;
		KDNode(const P pos_[N], const T &value_) : value(value_){
			left = right = NULL;
			for (int i = 0; i < N; ++i){
				pos[i] = pos_[i];
			}
		}
	};
	static inline int rotate(int prev){
		prev++;
		if (prev == N) prev = 0;
		return prev;
	}
	static inline int invrotate(int prev){
		prev--;
		if (prev < 0) prev = N - 1;
		return prev;
	}

	static inline bool same_pos(const P *p1, const P *p2){
		for (int i = 0; i < N; ++i){
			if (p1[i] != p2[i]) return false;
		}
		return true;
	}

	typedef std::stack<KDNode *> stack_t;
public:
	static const int DIM = N;
	size_t count;

public:
	template <typename Algo> class iterator_abstract{
	protected:
		stack_t buffer;
		KDNode *root;
		Algo algo;
		bool going_up;
		KDNode *current, *prev;
		int dir;
		iterator_abstract(KDNode *root_, const Algo &algo_) : root(root_), algo(algo_){
			dir = 0;
			going_up = false;
			current = root;
			prev = current;
			if (!algo.is_point_inrange(root)) next();
		}

		KDNode * get_current_node(){
			return current;
		}

	public:
		typedef const T& const_reference;
		typedef T& reference;
		typedef size_t size_type;

		iterator_abstract(const Algo &algo_) : algo(algo_){
			buffer = stack_t();
			dir = 0;
			going_up = false;
			current = NULL;
			prev = NULL;
		}

		iterator_abstract(const iterator_abstract &instance) : algo(instance.algo){
			buffer = instance.buffer;
			root = instance.root;
			dir = instance.dir;
			going_up = instance.going_up;
			current = instance.current;
			prev = instance.prev;
		}

		const_reference operator *() const{
			return current->value;
		}

		const P *get_key(){
			return current->pos;
		}

		inline void next(){
			while(current){
				algo.prepare(current, dir);
				if ((!going_up) && algo.is_nearleaf_inrange()){
					buffer.push(current);
					prev = current;
					current = algo.node_near;
					dir = rotate(dir);
					going_up = false;
				}else if ((!going_up || (prev != NULL && prev == algo.node_near)) && algo.is_farleaf_inrange()){
					buffer.push(current);
					prev = current;
					current = algo.node_far;
					dir = rotate(dir);
					going_up = false;
				}else{
					if (buffer.size() == 0){
						current = prev = NULL;
						break;
					}
					prev = current;
					current = buffer.top();
					buffer.pop();
					going_up = true;
					dir = invrotate(dir);
				}
				if ((!going_up) && algo.is_point_inrange(current)){
					return;
				}
			}
		}

		iterator_abstract & operator ++(){
			next();
			return *this;
		}

		const iterator_abstract operator ++(int){
			iterator_abstract cpy(*this);
			++(*this);
			return cpy;
		}

		bool is_end(){
			return current == NULL;
		}

		bool operator ==( const iterator_abstract &iter ) const {
			if (current == NULL && iter.current == NULL) return true; // current == NULL  algoɊ֌WȂ end Ƃ݂Ȃ
			if (current == iter.current && root == iter.root &&
				buffer.size() == iter.buffer.size() && algo == iter.algo && going_up == iter.going_up){
				if (buffer.size() == 0 || buffer.top() == iter.buffer.top()) return true;
			}
			return false;
		}

		bool operator !=( const iterator_abstract &iter ) const {
			return (*this == iter) ? false : true;
		}
		friend class kdtree;
	};

	struct algo_all{
		KDNode *node_near, *node_far;
		void prepare(KDNode *node, int dir){
			node_near = node->left;
			node_far = node->right;
		}

		inline const bool is_nearleaf_inrange() const {
			return (node_near != NULL);
		}
		inline const bool is_farleaf_inrange() const {
			return (node_far != NULL);
		}
		const bool is_point_inrange(KDNode *node) const {
			return true;
		}
		const bool operator == (const algo_all &instance) const{
			return true;
		}
	};

	struct algo_eukleide_range{
		double distance2;
		double distance2_at_is_point_inrange;
		P pos[N];
		P dx2;
		KDNode *node_near, *node_far;
		inline algo_eukleide_range(){
			dx2 = 0;
			node_near = node_far = NULL;
			distance2 = distance2_at_is_point_inrange = 0;
			for (int i = 0; i < N; ++i){
				this->pos[i] = 0;
			}
		}
		inline algo_eukleide_range(const P pos[N], double range){
			dx2 = 0;
			node_near = node_far = NULL;
			distance2 = range * range;
			for (int i = 0; i < N; ++i){
				this->pos[i] = pos[i];
			}
		}
		inline void prepare(KDNode *node, int dir){
			double dx = pos[dir] - node->pos[dir];
			dx2 = dx * dx;

			if (dx <= 0.0){
				node_near = node->left;
				node_far = node->right;
			}else{
				node_near = node->right;
				node_far = node->left;
			}
		}

		inline const bool is_nearleaf_inrange() const {
			if (!node_near) return false;
			return true;
		}
		inline const bool is_farleaf_inrange() const {
			if (!node_far) return false;
			return dx2 < distance2;
		}

		inline bool is_point_inrange(KDNode *node) {
			double distance2_ = 0;
			for (int i = 0; i < N; ++i){
				P d = node->pos[i] - pos[i];
				distance2_ += d * d;
			}
			distance2_at_is_point_inrange = distance2_;
			return (distance2_ <= distance2);
		}
		inline const bool operator == (const algo_eukleide_range &instance) const {
			if (distance2 != instance.distance2) return false;
			for (int i = 0; i < N; ++i){
				if (pos[i] != instance.pos[i]) return false;
			}
			return true;
		}
	};

	class iterator : public iterator_abstract<algo_all>{
	public:
		iterator() : iterator_abstract<algo_all>(algo_all()){};
		iterator(KDNode *root_) : iterator_abstract<algo_all>(root_, algo_all()){
		}
	};

	class iterator_eukleide_range : public iterator_abstract<algo_eukleide_range>{
		const double &distance2_at_is_point_inrange(){
			return this->algo.distance2_at_is_point_inrange;
		}
		double &distance2(){
			return this->algo.distance2;
		}
	public:
		double get_distance(){
			return std::sqrt(this->algo.distance2_at_is_point_inrange);
		}
		iterator_eukleide_range() : iterator_abstract<algo_eukleide_range>(algo_eukleide_range()){};
		iterator_eukleide_range(KDNode *root_, const P pos[N], double range) : iterator_abstract<algo_eukleide_range>(root_, algo_eukleide_range(pos, range)){
		}
		friend class kdtree;
	};

	KDNode *root;
	kdtree(){
		root = NULL;
		count = 0;
	}
	void clear(){
		if (root){
			//go[XđSdelete
			stack_t buffer;
			buffer.push(root);
			do{ // [DT - s
				KDNode *to_delete = buffer.top();
				buffer.pop();
				if (to_delete->right) buffer.push(to_delete->right);
				if (to_delete->left) buffer.push(to_delete->left);
				delete to_delete;
			}while(buffer.size());
		}
		root = NULL;
		count = 0;
	}
	~kdtree(){
		clear();
	}
	size_t size(){
		return count;
	}
private:
	void insert(KDNode *node){
		if (!root){
			root = node;
			++count;
			return;
		}
		int dir = DIRSTART;
		KDNode *cur = root;
		for(;;){
			if (cur->pos[dir] > node->pos[dir]){
				if (!cur->left){
					cur->left = node;
					++count;
					break;
				}else cur = cur->left;
			}else{
				if (!cur->right){
					if (same_pos(cur->pos, node->pos)) return;
					cur->right = node;
					++count;
					break;
				}else cur = cur->right;
			}
			dir = rotate(dir);
		}
	}

public:
	void insert(const P pos[N], const T &value){
		insert(new KDNode(pos, value));
	}

	bool exist(const P pos[N]){
		iterator_eukleide_range iter = begin_range(pos, 0);
		return iter != end_range();
	}

	iterator begin(){
		return iterator(root);
	}
	iterator end(){
		iterator iter;
		return iter;
	}

	iterator_eukleide_range begin_range(const P pos[N], double range){
		return iterator_eukleide_range(root, pos, range);
	}

	iterator_eukleide_range end_range(){
		iterator_eukleide_range iter;
		return iter;
	}


	struct iterator_knn{
	private:
		struct kditem{
			double pos[N];
			double distance;
			const T &value;
			kditem(const P pos_[N], double distance_, const T &value_) : distance(distance_), value(value_){
				for (int i = 0; i < N; ++i){
					pos[i] = pos_[i];
				}
			}
		};
		kditem **items;
		size_t num;
		size_t index;
		template <typename Iter> iterator_knn(Iter begin, Iter end, int num){
			this->num = num;
			items = new kditem *[num];
			int i = 0;
			for (Iter iter = begin; iter != end; ++iter, ++i){
				items[i] = new kditem(iter->second->pos, std::sqrt(iter->first), iter->second->value);
			}
			index = 0;
		}
		iterator_knn(){
			items = NULL;
			index = num = 0;
		}
	public:
		typedef const T& const_reference;
		typedef T& reference;

		iterator_knn(const iterator_knn &iter){
			num = iter.num;
			index = iter.index;
			items = new kditem *[num];
			for (size_t i = 0; i < num; ++i){
				kditem *&current = iter.items[i];
				items[i] = new kditem(current->pos, current->distance, current->value);
			}
		}
		const bool is_end() const{
			return index >= num;
		}
		bool operator == (const iterator_knn &iter) {
			if (is_end() && iter.is_end()) return true;
			return false;
		}
		bool operator != (const iterator_knn &iter) {
			return (*this == iter) ? false : true;
		}
		const_reference operator *() {
			if (is_end()) throw new std::exception();
			return items[index]->value;
		}

		const P *get_key(){
			if (is_end()) throw new std::exception();
			return items[index]->pos;
		}

		const double get_distance(){
			if (is_end()) throw new std::exception();
			return items[index]->distance;
		}

		iterator_knn & operator ++(){
			++index;
			return *this;
		}

		const iterator_knn operator ++(int){
			iterator_knn cpy(*this);
			++(*this);
			return cpy;
		}


		~iterator_knn(){
			for (size_t i = 0; i < num; ++i){
				delete items[i];
			}
			delete[] items;
		}
		friend class kdtree;
	};

	iterator_knn begin_eukleide_knn(const P pos[N], int k){
		if (!root) return iterator_knn();
		typedef std::pair<double, const KDNode *> pair_t;
		pair_t *buffer = new pair_t[k];
		int csize = 0, backsize = 0;
		for (iterator_eukleide_range iter = begin_range(pos, std::numeric_limits<double>::max()); !iter.is_end(); iter.next()){
			double distance2 = iter.distance2_at_is_point_inrange();
			int i;
			for (i = 0; i < k;++i){
				if (i == csize || buffer[i].first >= distance2){
					for (int h = backsize; h > i; --h){
						buffer[h] = buffer[h-1];
					}
					buffer[i].first = distance2;
					buffer[i].second = iter.get_current_node();
					if (csize < k){
						++csize;
						if (backsize < k - 1){
							++backsize;
						}
					}else{
						iter.distance2() = buffer[k-1].first;
					}
					break;
				}
			}
		}
		iterator_knn iter(buffer, buffer + csize, csize);
		delete[] buffer;
		return iter;
	}

	const iterator_knn end_eukleide_knn(){
		return iterator_knn();
	}

private:
	class inner {
		const int dir;
	public:
		inner(const int dir_) : dir(dir_){}
		bool operator ()(const KDNode * left, const KDNode * right) const{
			return left->pos[dir] < right->pos[dir];
		}
	};

public:
	void balance(){

		size_t count_temp = size();

		KDNode **nodes = new KDNode *[count_temp];
		size_t i = 0;
		for (iterator iter = begin(); iter != end(); ++iter, ++i){
			nodes[i] = iter.get_current_node();
		}

		count = 0;
		for (i = 0; i < count_temp; ++i){
			nodes[i]->left = nodes[i]->right = NULL;
		}
		root = NULL;

		std::deque<std::pair<int, std::pair<KDNode **, KDNode **> > > q;
		q.push_front(std::make_pair(DIRSTART, std::make_pair(nodes, nodes+count_temp)));
		do{
			int dir = q.front().first;
			const inner cmp(dir);
			KDNode **first = q.front().second.first, **second = q.front().second.second;
			std::sort(first, second, cmp);
			q.pop_front();
			KDNode **mid = first + ((second - first) / 2);
			insert(*mid);
			*mid = NULL;
			dir = rotate(dir);
			if (first < mid) q.push_front(std::make_pair(dir,std::make_pair(first, mid)));
			if (mid+1 < second) q.push_front(std::make_pair(dir,std::make_pair(mid+1,second)));
		}while(q.size());


		delete[] nodes;
	}
};
}
