﻿/// Simple File Dialog version 0.1
/// Copylight mocchi 2021
/// mocchi_2003@yahoo.co.jp
/// Distributed under the Boost Software License, Version 1.0.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.IO;

// references
// https://www.codeproject.com/Articles/13097/An-quot-Explorer-Style-quot-TreeView-Control
// https://www.ipentec.com/document/csharp-shell-namespace-create-explorer-tree-view-control-and-linked-list-view
// http://acha-ya.cocolog-nifty.com/blog/2010/09/post-241a.html
// https://nasu38yen.wordpress.com/2010/05/28/%e6%8b%a1%e5%bc%b5%e5%ad%90%e3%81%8b%e3%82%89%e5%b0%8f%e3%81%95%e3%81%aa%e3%82%a2%e3%82%a4%e3%82%b3%e3%83%b3%e3%82%92get%e3%81%99%e3%82%8b%e3%81%ab%e3%81%af%e3%80%81shgetfileinfo%e3%82%92usefileattrib/
// https://dobon.net/vb/bbs/log3-51/30394.html
// https://www.curict.com/item/0a/0a33f42.html

// Todo: select folder
namespace SimpleFileDialog {
	public partial class SimpleFileDialog : Form {

		// properties
		public string InitialDirectory {
			get;
			set;
		}
		public string FileName {
			get;
			set;
		}
		public string Filter {
			get;
			set;
		}
		public string Title {
			get;
			set;
		}

		private FileDialogMode _fileDialogMode;
		public enum FileDialogMode {
			OpenFile, SaveFile, SelectFolder
		}

		private string _currentDirectory = "";
		public string CurrentDirectory {
			get{
				return _currentDirectory;
			}
			set{
				if (!Directory.Exists(value)) {
					throw new DirectoryNotFoundException();
				}
				if (_currentDirectory == value) return;

				try {
					// Todo: 冗長。より効率の良い方法に直したい
					var dirs = Directory.EnumerateDirectories(value);
				} catch {
					MessageBox.Show("access denied", "error", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return;
				}
				_currentDirectory = value;
				RedrawListView();

			}
		}

		public SimpleFileDialog(FileDialogMode fileDialogMode = FileDialogMode.OpenFile) {
			Filter = "";
			_fileDialogMode = fileDialogMode;
			InitializeComponent();

			switch (fileDialogMode) {
				case FileDialogMode.OpenFile:
					buttonOK.Text = "Open";
					break;
				case FileDialogMode.SaveFile:
					buttonOK.Text = "Save";
					break;
				case FileDialogMode.SelectFolder:
					buttonOK.Text = "Select";
					textBoxFileName.ReadOnly = true;
					textBoxFileName.Text = "Select Folder.";
					break;
			}
		}

		private void SimpleFileDialog_Load(object sender, EventArgs e) {

			if (!string.IsNullOrEmpty(Title)) {
				Text = Title;
			} else if (_fileDialogMode == FileDialogMode.OpenFile){
				Text = "Open";
			} else if (_fileDialogMode == FileDialogMode.SaveFile) {
				Text = "Save as";
			} else if (_fileDialogMode == FileDialogMode.SelectFolder) {
				Text = "Select folder";
			}

			// folder icon
			var imageList = new ImageList();
			{
				var folderIconB64 =
@"Qk1GAgAAAAAAADYAAAAoAAAADgAAAAwAAAABABgAAAAAABACAAAAAAAAAAAAAAAA
AAAAAAAA////////////////////////////////////////////////////////
AAD///////9XerlXerlXerlXerlXerlXerlXerlXerlXerlXerlXern///8AAP//
/2jf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Vd6uf///wAA////aN/9
aN/9aN/9aN/9aN/9aN/9aN/9aN/9aN/9aN/9aN/9V3q5////AAD///9o3/1o3/1o
3/1o3/1o3/1o3/1o3/1o3/1o3/1o3/1o3/1Xern///8AAP///2jf/Wjf/Wjf/Wjf
/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Vd6uf///wAA////aN/9aN/9aN/9aN/9aN/9
aN/9aN/9aN/9aN/9aN/9aN/9V3q5////AAD///9o3/1o3/1o3/1o3/1o3/1o3/1o
3/1o3/1o3/1o3/1o3/1Xern///8AAP///2jf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf/Wjf
/Wjf/Wjf/Wjf/Vd6uf///wAA////AqnSAqnSAqnSAqnSAqnSAqnSaN/9aN/9aN/9
aN/9aN/9////////AAD///8CqdICqdICqdICqdICqdICqdL/////////////////
//////////8AAP//////////////////////////////////////////////////
/////wAA";

				var ms = new MemoryStream(System.Convert.FromBase64String(folderIconB64), false);
				ms.Position = 0;
				var img = new Bitmap(ms);
				imageList.Images.Add(img);
			}

			listViewFileList.Clear();
			listViewFileList.SmallImageList = imageList;
			listViewFileList.Columns.Add("Name", 400);
			listViewFileList.Columns.Add("Date modified", 150);
			listViewFileList.Columns.Add("File size", 100);
			listViewFileList.Columns[2].TextAlign = HorizontalAlignment.Right;
			listViewFileList.View = View.Details;

			comboBoxFilter.DataSource = null;
			if (Filter != null) {
				var filters = Enumerable.Range(0, 0).Select(v => new { Display = "", Value = "" }).ToList();
				var filtersStr = Filter.Split('|');
				for (var i = 0; i < filtersStr.Length - 1; i += 2) {
					filters.Add(new { Display = filtersStr[i], Value = filtersStr[i + 1] });
				}
				if (filters.Count > 0) {
					comboBoxFilter.DataSource = filters.ToArray();
					comboBoxFilter.DisplayMember = "Display";
					comboBoxFilter.ValueMember = "Value";
				}
			}

			CurrentDirectory = InitialDirectory;
		}

		private void RedrawListView() {
			listViewFileList.Items.Clear();
			if (!Visible || !Directory.Exists(_currentDirectory)) {
				return;
			}

			var filters = comboBoxFilter.Items.Count > 0 ? ((string)comboBoxFilter.SelectedValue).Split(';') : null;

			var items = new List<ListViewItem>();
			// display folders and files.

			var dirs = Directory.EnumerateDirectories(_currentDirectory);
			var files = Directory.EnumerateFiles(_currentDirectory);
			switch (listViewFileList.View) {
				case View.List: {
						// directories
						foreach (var dir in dirs) {
							items.Add(new ListViewItem(Path.GetFileName(dir)));
						}

						// files
						foreach (var file in files) {
							// Todo: wildcard matching
							items.Add(new ListViewItem(Path.GetFileName(file)));
						}
					}

					break;
				case View.Details: {
						// directories
						foreach (var dir in dirs) {
							var lvi = new ListViewItem(new string[] { Path.GetFileName(dir), "", "" });
							lvi.ImageIndex = 0;
							items.Add(lvi);
						}

						// files
						var unitname = new string[]{ " KB", " MB", " GB", " TB"};
						foreach (var file in files) {
							// Todo: wildcard matching
							var fi = new FileInfo(file);

							var fileSize = fi.Length;
							string fileSizeStr = "> 10 TB";

							long denom = 1024L;
							for (var i = 0; i < 4; ++i){
								if (fileSize < denom * 10240L){
									long reminder;
									long quot = Math.DivRem(fileSize, denom, out reminder);
									fileSizeStr = (quot + (reminder > 0L ? 1L : 0L)).ToString() + unitname[i];
									break;
								}
								denom *= 1024L;
							}

							var lvi = new ListViewItem(new string[] { Path.GetFileName(file), fi.LastWriteTime.ToString(), fileSizeStr });
							lvi.ImageIndex = -1;
							items.Add(lvi);
						}
					}
					break;
			}
			textBoxTargetFolder.Text = CurrentDirectory;

			listViewFileList.SuspendLayout();
			foreach (var itm in items){
				listViewFileList.Items.Add(itm);
			}
			listViewFileList.ResumeLayout();
		}

		private bool CheckValid(string path) {
			var invalidPathChars = Path.GetInvalidPathChars().Concat(new[] { '*', '?' }).ToArray();
			if (path.IndexOfAny(invalidPathChars) >= 0
				|| Regex.IsMatch(path, @"(^|\\|/)(CON|PRN|AUX|NUL|CLOCK\$|COM[0-9]|LPT[0-9])(\.|\\|/|$)", RegexOptions.IgnoreCase)) {
				return false;
			}
			return true;
		}

		private bool Confirm() {
			var target = textBoxFileName.Text;
			if (_fileDialogMode != FileDialogMode.SelectFolder && string.IsNullOrEmpty(target)) return false;

			if (!CheckValid(target)) {
				MessageBox.Show(this, "invalid name", "error", MessageBoxButtons.OK);
				return false;
			}

			var fullPath = Path.Combine(_currentDirectory, target);

			var invalidPathChars = Path.GetInvalidPathChars().Concat(new []{'*', '?'}).ToArray();
			
			bool exists = File.Exists(fullPath);

			if (_fileDialogMode == FileDialogMode.OpenFile && !exists) {
				MessageBox.Show(this, target + " not found.", "file not found", MessageBoxButtons.OK);
				return false;
			} else if (_fileDialogMode == FileDialogMode.SaveFile && exists) {
				var rc = MessageBox.Show(this, target + " exists. override?", "confirm to override", MessageBoxButtons.YesNo);
				if (rc == System.Windows.Forms.DialogResult.No) return false;
			}
			return true;
		}

		// callbacks
		private void radioButtonList_CheckedChanged(object sender, EventArgs e) {
			listViewFileList.View = View.List;
		}

		private void radioButtonDetails_CheckedChanged(object sender, EventArgs e) {
			listViewFileList.View = View.Details;
		}

		private void buttonRedraw_Click(object sender, EventArgs e) {
			RedrawListView();
		}

		private void buttonOK_Click(object sender, EventArgs e)
		{
			DialogResult = System.Windows.Forms.DialogResult.OK;
			Close();
		}

		private void buttonCancel_Click(object sender, EventArgs e)
		{
			DialogResult = System.Windows.Forms.DialogResult.Cancel;
			Close();
		}

		private void SimpleFileDialog_FormClosing(object sender, FormClosingEventArgs e)
		{
			if (DialogResult == System.Windows.Forms.DialogResult.OK){
				if (!Confirm()) {
					e.Cancel = true;
					FileName = "";
				} else {
					FileName = Path.Combine(CurrentDirectory, textBoxFileName.Text);
				}
			}
		}

		private void buttonUp_Click(object sender, EventArgs e) {
			var parentDir = Path.GetDirectoryName(CurrentDirectory);
			if (!Directory.Exists(parentDir)) return;

			CurrentDirectory = parentDir;
		}

		private void textBoxTargetFolder_KeyPress(object sender, KeyPressEventArgs e) {
			if (e.KeyChar == (char)Keys.Enter) {
				var newDir = textBoxTargetFolder.Text;
				if (Directory.Exists(newDir)) {
					CurrentDirectory = newDir;
				} else {
					textBoxTargetFolder.Text = CurrentDirectory;
				}
				e.Handled = true;
			}
		}

		private void textBoxFileName_KeyPress(object sender, KeyPressEventArgs e) {
			if (e.KeyChar == (char)Keys.Enter) {
				var fileName = textBoxFileName.Text;
				var newDir = "";
				if (Directory.Exists(fileName)) {
					CurrentDirectory = fileName;
					textBoxFileName.Text = "";
				} else if (CheckValid(fileName) && Directory.Exists(newDir = Path.Combine(CurrentDirectory, fileName))) {
					CurrentDirectory = newDir;
					textBoxFileName.Text = "";
				} else {
					DialogResult = System.Windows.Forms.DialogResult.OK;
					Close();
				}
				e.Handled = true;
			}
		}

		private void listViewFileList_Click(object sender, EventArgs e) {
			if (listViewFileList.SelectedIndices.Count == 0) return;
			var itm = listViewFileList.SelectedItems[0];
			var fullPath = Path.Combine(CurrentDirectory, itm.Text);

			if (Directory.Exists(fullPath)) {
				return;
			} else {
				textBoxFileName.Text = itm.Text;
			}
		}

		private void listViewFileList_DoubleClick(object sender, EventArgs e) {
			if (listViewFileList.SelectedIndices.Count == 0) return;
			var itm = listViewFileList.SelectedItems[0];
			var fullPath = Path.Combine(CurrentDirectory, itm.Text);

			if (Directory.Exists(fullPath)) {
				textBoxFileName.Text = "";
				CurrentDirectory = fullPath;
			} else {
				DialogResult = System.Windows.Forms.DialogResult.OK;
				FileName = itm.Text;
				Close();
			}
		}

	}
}
