package jp.nanah.bastub.util;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Row;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jp.nanah.bastub.data.KvData;

public class BastubUtils {

	private static final Logger logger = LoggerFactory.getLogger(BastubUtils.class);

	//===============================
	// パス処理
	//===============================

	/**
	 * URLで受け取ったパスのうち、有効なものだけ(パスとして受け取ったものだけ)を持つListにして返す。
	 * 実際は前詰めだから途中で無効な値が出てきたら以降は全部無効値だが、
	 * 汎用性を考えて全引数を判定することにする。
	 *
	 * @param <T>
	 * @param vals
	 * @return
	 */
	@SafeVarargs
	public static <T> List<T> toValidList(Optional<T> ... vals) {
		List<T> list = new ArrayList<T>();
		for (Optional<T> v : vals) {
			if (v.isPresent()) {
				list.add(v.get());
			}
		}
		return list;
	}

	/**
	 * データファイル（エクセル及びJSON）のファイルパスを取得する。
	 *
	 * @param dir
	 * @param filename
	 * @param method
	 * @param ext
	 * @return
	 */
	public static File getPagedataPath(File dir, String filename, String method, String exttext){
		String[] extarray = exttext.split(",", -1);
		File nofile = null;
		for (String ext : extarray){
			//HTTPメソッドありのパスのファイルがあればそれを採用
			File fileByMethod = new File(dir, filename + "_" + method + ext);
			if (fileByMethod.exists()) {
				return fileByMethod;
			}

			//HTTPメソッド指定のファイルがなければ共通ファイルを採用
			File fileAll = new File(dir, filename + ext);
			if (fileAll.exists()) {
				return fileAll;
			}
			if (nofile == null){
				nofile = fileAll;
			}
		}

		return nofile;  //どっちも無かったら共通ファイルを返す
	}

	/**
	 * URLパスの場所指定であればそのパス値を返す。
	 * @param val パス指定または実値
	 * @param pathList URLパス
	 * @param defval URLパス指定でないときに返す値
	 * @return
	 */
	public static void replacePathValue(List<String> valueList, List<String> pathList) {
		for (int i=0; i<valueList.size(); i++) {
			String val = valueList.get(i);
			if (val.startsWith("$")) {
				int index = Integer.parseInt(val.substring(1)) - 1;
				if (index < pathList.size()) {
					valueList.set(i, pathList.get(index));
				}
			}
		}
	}

	public static KvData toKvData(List<String> valueList) {
		int n = valueList.size();
		List<String> k = valueList.subList(0, n - 1);
		String v = valueList.get(n - 1);
		KvData kv = new KvData(k, v);
		return kv;
	}

	public static boolean equalsKey(List<String> v1, List<String> v2) {
		if (v1 == null && v2 == null) {
			return true;
		}
		if (v1 == null || v2 == null) {
			return false;
		}
		if (v1.size() != v2.size()) {
			return false;
		}
		for (int i=0; i<v1.size(); i++) {
			if (v1.get(i).equals(v2.get(i)) == false) {
				return false;
			}
		}
		return true;
	}

	public static String getExtText(String s){
		int n = s.lastIndexOf(".");
		if (n >= 0){
			return s.substring(n).toLowerCase();
		} else {
			return "";
		}
	}

	/**
	 * URLのドメインまでを返す。
	 * 例えば、http://www.foo.co.jp/aa/bb/ccならば、"http://www.foo.co.jp"を返す。
	 * @param url URLパス
	 * @return ドメインまでの文字列。最後の/は返さない。
	 */
	public static String getUrlDomain(String url){
		int n = url.lastIndexOf("//");
		if (n >= 0){
			int m = url.indexOf("/", n + 2);
			if (m >= 0){
				return url.substring(0, m);
			}
		}
		return url;
	}

	//===============================
	// POI関連
	//===============================

	/**
	 * Excelの行データから値のあるセルだけをKeyリストと値を取得する。
	 * セルに何もなければnullになる。空文字にしたい場合は'のみを設定する。(おそらく)
	 *
	 * @param row
	 * @param blankOk 空文字も有効にするか。ログ出力するためならtrue、フィルター用の設定をとるときはfalse
	 * @return
	 */
	public static List<String> getRowValueList(Row row, boolean blankOk){
		List<String> keyList = new ArrayList<String>();
		int coltop = row.getFirstCellNum();
		int colend = row.getLastCellNum();
		for (int i=coltop; i<colend; i++) {
			Cell cell = row.getCell(i);
			String s = getCellText(cell);

			//先頭列が#始まりだったらコメントなのでスキップする
			if (i == coltop) {
				if (StringUtils.startsWith(s, "#")) {
					break;
				}
			}

			if (StringUtils.isBlank(s)) {
				if (blankOk == false){
					continue;
				}
			}

			keyList.add(s);
		}
		return keyList;
	}

	/**
	 * セルの値を文字列で取得する。
	 * 【TODO】★数値型や日付型のときは文字列に変換する。
	 * @param cell
	 * @return
	 */
	public static String getCellText(Cell cell) {
		if (cell == null) {
			return null;
		}
		try {
			if (cell.getCellType() == CellType.STRING) {
				return cell.getStringCellValue();
			} else if(cell.getCellType() == CellType.NUMERIC){
				return String.valueOf((int)cell.getNumericCellValue());  //TODO: 数値は整数のみサポート？
			} else if (cell.getCellType() == CellType.FORMULA){
				CellType formulaType = cell.getCachedFormulaResultType();
				if (formulaType == CellType.NUMERIC){
					return String.valueOf(cell.getNumericCellValue());
				} else {
					return String.valueOf(cell.getStringCellValue());
				}
			} else if (cell.getCellType() == CellType._NONE){
				return cell.getStringCellValue();
			}
		} catch (Exception e){
			logger.warn("変換異常", e);
		}
		return null;
	}

	public static boolean isBlankRow(Row row){
		int coltop = row.getFirstCellNum();
		int colend = row.getLastCellNum();
		for (int i=coltop; i<=colend; i++) {
			Cell cell = row.getCell(i);
			String s = getCellText(cell);
			if (StringUtils.isNotBlank(s)) {
				return false;
			}
		}
		return true;
	}

	//===============================
	// JSON関連
	//===============================

	public static void printJsonTree(String path, JSONObject json) throws JSONException {
		Iterator<String> it = json.keys();
		while (it.hasNext()){
			String key = it.next();
			JSONObject val = json.optJSONObject(key);
			String here = StringUtils.isEmpty(path) ? key : path + "." + key;
			if (json.optJSONArray(key) != null) {
				JSONArray jsarray = json.getJSONArray(key);
				for (int i=0; i<jsarray.length(); i++) {
					if (jsarray.optJSONObject(i) != null) {
						printJsonTree(here + "["+i+"]", jsarray.getJSONObject(i));
					}
				}
			} else if (json.optJSONObject(key) != null) {
				printJsonTree(here , json.getJSONObject(key));
			}
		}
	}

	/**
	 * 置き換え文字を探してその列位置を返す。
	 * @param v
	 * @param rowList
	 * @param defaultColumnName vの置き換え文字が列名未指定("${}")のときにキー名を列名にするときのキー値
	 * @return 0以上: 置換データの列、-1:置換文字だが該当値なし、-2: 非置換文字
	 */
	public static int getColumnNoForReplace(String v, List<Row> rowList, String defaultColumnName) {
		//数値指定の部分は除去する
		boolean isToNumber = v.endsWith("#N");
		if (isToNumber) {
			v = v.substring(0, v.length()-2);
		}

		if (v.startsWith("${") && v.endsWith("}")) {

			if (rowList.size() <= 1) {
				return -1;
			}

			String matchName = v.substring(2,  v.length()-1);
			if (matchName.equals("") && StringUtils.isNoneEmpty(defaultColumnName)){
				matchName = defaultColumnName;
			}

			//列名
			Row row = rowList.get(0);
			int coltop = row.getFirstCellNum();
			int colend = row.getLastCellNum();

			for (int i=coltop; i<=colend; i++) {
				String cellv = BastubUtils.getCellText(row.getCell(i));
				if (cellv != null && cellv.equals(matchName)) {
					return i;
				}
			}
			return -1;
		} else {
			return -2;
		}
	}

	/**
	 * 指定する位置から始まる文字列が数値の置き換え文字のとき、
	 * 置き換え文字の最終位置を返す。
	 *
	 * @param s
	 * @param pos
	 * @return
	 */
	public static int getNumberLast(String s, int n) {
		//置き換え文字の最後を探す
		int last = s.indexOf("}", n);
		if (last <0) {
			return -1;
		}

		for (int i=n-1; i>=0; i--) {
			char c = s.charAt(i);

			//空白文字ならスキップ
			if (Character.isWhitespace(c)) {
				continue;
			}

			//空白以外で最初に出てきた文字が":"のときだけ数値と判断する
			if (c == ':') {
				return last + 1;
			}

			break;
		}

		return -1;
	}

	public static boolean isArrayExists(JSONObject json) throws JSONException{
		Iterator<String> it = json.keys();
		while (it.hasNext()){
			String key = it.next();
			if (json.optJSONArray(key) != null) {
				return true;
			} else if (json.optJSONObject(key) != null) {
				boolean b = isArrayExists(json.getJSONObject(key));
				if (b){
					return true;
				}
			}
		}
		return false;
	}

	public static boolean isJsonEntry(Map.Entry<String, String[]> ent){
		String key = ent.getKey().trim();
		String[] val = ent.getValue();
		if (val == null || val.length == 0 || val[0].trim().length() == 0){
			if (key.startsWith("[" ) && key.endsWith("]")){
				return true;
			}
			if (key.startsWith("{" ) && key.endsWith("}")){
				return true;
			}
		}
		return false;
	}

	//===============================
	// 汎用
	//===============================

	public static boolean equals(List<String> list1, List<String>list2){
		if (list1.size() != list2.size()){
			return false;
		}
		for (int i=0; i<list1.size(); i++){
			if (StringUtils.equals(list1.get(i), list2.get(i)) == false){
				return false;
			}
		}

		return true;
	}

	/**
	 * リストの後ろ部分が一致するかどうか判定する。
	 * JSONの階層を指定しない場合は最後の階層だけで一致と判断させるため。
	 * @param list1
	 * @param list2
	 * @return
	 */
	public static boolean equalsTail(List<String> list1, List<String>list2){
		int size1 = list1.size();
		int size2 = list2.size();

		if (size1 < size2){
			list2 = list2.subList(size2-size1, size2);
		} else if (size1 > size2){
			list1 = list1.subList(size1-size2, size1);
		}
		return equals(list1, list2);
	}

	/**
	 * イテレータの値を昇順にしたリストを取得する。
	 *
	 * @param it
	 * @return
	 */
	public static List<String> sortIterator(Iterator<String> it){
		List<String> keyList = new ArrayList<String>();
		while (it.hasNext()){
			keyList.add(it.next());
		}
		Collections.sort(keyList);
		return keyList;
	}

}
