package jp.nanah.bastub.service;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Row;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jp.nanah.bastub.util.BastubUtils;

/**
 * 適用データのうち、使用した列値を記憶して
 * 下位の階層にデータを受け渡すデータの絞り込みを行う。
 *
 */
public class UsingInfo {

	protected static final Logger logger = LoggerFactory.getLogger(UsingInfo.class);

	/**
	 * データの参照
	 */
	private List<Row> dataList;

	/**
	 * システム側で固定値で置き換える文字列のMap
	 */
	private Map<String, String> replaceConst = new HashMap<String, String>();

	private static SimpleDateFormat DF_DATE = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	/**
	 * JSONの値を固定値に変換するためのMap。
	 * @param pathList
	 * @param req
	 * @return
	 */
	public static UsingInfo getInitInstance(List<String> pathList, HttpServletRequest req){
		UsingInfo ui = new UsingInfo();

		Map<String, String> map = ui.replaceConst;//  new HashMap<String, String>();

		String ymdhms = DF_DATE.format(new Date());
		HttpSession session = req.getSession(true);
		map.put("${#SYSDATE#}", ymdhms);
		map.put("${#SYS_YMDHMS#}", ymdhms);
		map.put("${#SYS_YMD#}", ymdhms.substring(0, 10));
		map.put("${#SYS_HM#}", ymdhms.substring(11, 19));
		map.put("${#SYS_MSEC#}", String.valueOf(System.currentTimeMillis()));
		map.put("${#SYS_SEC#}", String.valueOf(System.currentTimeMillis()/1000));
		map.put("${#THREAD_ID#}", String.valueOf(Thread.currentThread().getId()));
		map.put("${#THREAD_NAME#}", Thread.currentThread().getName());

		//"${#PATHx#}→パス値に変換
		for (int i=0; i<pathList.size(); i++){
			map.put("${#PATH" + i + "#}", pathList.get(i));
		}
		map.put("${#URL#}", req.getRequestURL().toString());
		map.put("${#SESSION_ID#}", (session != null) ? session.getId() : "");

		return ui;
	}

	/**
	 * 使用済みのデータインデックス。0なら未使用、1～はdataListのインデックス値と一致。
	 */
	private int[] usedRow;

	private UsingInfo(){
	}

	public UsingInfo(List<Row> dataList, UsingInfo ui){
		this.dataList = dataList;

		//列数は先頭行。1行目に列名は必須
		int colMax = (dataList.size() > 0) ? dataList.get(0).getLastCellNum() : 0;
		usedRow = new int[colMax];

		//固定情報があれば引き継ぐ
		if (ui != null){
			replaceConst = ui.getReplaceConst();
		}
	}

	public Map<String, String> getReplaceConst(){
		return replaceConst;
	}


	/**
	 * 置換文字ならデータに置き換えて返す。
	 * 置換文字でなければもともとのJSON値を返す。
	 * 置換文字なのに該当データがなければ空文字を返す。(これでよいはず)
	 *
	 * @param keyList ログ用にしか使わない。
	 * @param key JSONのキー値
	 * @param orgval JSONの現在値
	 * @return 置換後の文字列。非null。
	 */
	public String getDirectValue(List<String> keyList, String key, String orgval){
		if (orgval == null){
			return "";
		}

		String dstval = orgval;

		//直値で置き換え文字なら置き換え名が列データにあるか探す
		int colIndex = BastubUtils.getColumnNoForReplace(orgval, dataList, key);

		if (colIndex >=0){
			if (dataList.size() > 1) {
				int dataRow = usedRow[colIndex] + 1;
				if (dataRow < dataList.size()){
					dstval = BastubUtils.getCellText(dataList.get(dataRow).getCell(colIndex));
					usedRow[colIndex]++;
				} else {
					dstval = "";
					logger.warn("[{}]: JSON内でキー {\"{}\" が重複登録されている可能性あり", keyList, key);
				}
			}

		} else {
			String rep = replaceConst.get(orgval);
			if (rep != null){
				dstval = rep;
			}

			//置換文字なのに置換列がないときはそのままにせず空文字にする
			if (colIndex == -1){
				//dstval = "";
				//↑設定が間違っている可能性が高いので空白にせず元の文字列をだすことにする
			}
		}

		return dstval;
	}

	/**
	 * 下層に適用するデータの一覧を取得する。
	 * 下層には、列値として参照したもの
	 *
	 * @return
	 */
	public List<Row> getLoweredData(){
		//１行目は列名なので最低２行以上ないとダメ。１行だけならデータなし。
		if (dataList.size() < 2){
			//return null;
			return dataList;
			//↑nullを返すと配列やJSONオブジェクトの中の置き換え文字が消えないのでそのまま返すようにした。
		}

		List<Row> loweredList = new ArrayList<Row>();

		//１行目の使用データを、列番号→列値のマップ化。
		Map<Integer, List<String>> usedDataMap = getUsedDataMap();

		//データ未使用なら全データが対象
		if (usedDataMap.size() == 0){
			return dataList;
		}

		//各行から先頭データと同じデータを持っていれば、次の層にも適用する（1行目も含む)
		for (int r=0; r<dataList.size(); r++){
			Row rowVal = dataList.get(r);

			if (r == 0){
				//列名の行は無条件でコピー
			} else {
				//データ行は、現階層で使用した値でなければ下層には送らない
				if (isUsedData(usedDataMap, rowVal) == false){
					continue;
				}
			}

			loweredList.add(rowVal);
		}

		return loweredList;
	}

	/**
	 * 使用済みデータのMapを返す。
	 * Key=列番号(0～)、value=列値のリスト
	 * @return
	 */
	Map<Integer, List<String>> getUsedDataMap(){
		Map<Integer, List<String>> usedDataMap = new LinkedHashMap<Integer, List<String>>();
		for (int c=0; c<usedRow.length; c++){
			if (usedRow[c] > 0){
				List<String> usedList = new ArrayList<String>();
				for (int r=1; r<=usedRow[c]; r++){
					Cell cell = dataList.get(r).getCell(c);
					usedList.add( BastubUtils.getCellText(cell) );
				}
				usedDataMap.put(new Integer(c), usedList);
			}
		}
		return usedDataMap;
	}

	public boolean isUsedData(Map<Integer, List<String>> usedMap, Row rowVal){
		Iterator<Integer> it = usedMap.keySet().iterator();
		while (it.hasNext()){
			Integer key = it.next();
			String v = BastubUtils.getCellText(rowVal.getCell(key.intValue()));
			if (usedMap.get(key).contains(v) == false){
				return false;
			}
		}
		return true;
	}

}
