# pkgspecs.sh
# -----------------------------------------------------------------------------
#
# Implementation of a generalised package specification interpreter;
# source to define interpreter functions, prior to loading a spec file.
#
# -----------------------------------------------------------------------------
#
# $Id: pkgspec.sh,v 55ffe60ab6fe 2018/10/31 09:04:02 keith $
#
# Written by Keith Marshall <keith@users.osdn.me>
# Copyright (C) 2011-2014, 2017, 2018, MinGW.org Project
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
# -----------------------------------------------------------------------------
#
: ${PACKAGE_SRCDIR="${SRCDIR-"."}"}
: ${pkgspec="pkgspec"}

# If host architecture isn't otherwise specified, then assume that
# we are building for a "mingw32" host.
#
  ARCH=${ARCH-"mingw32"}

# Provide an absolute canonical reference form for PACKAGE_SRCDIR.
#
  PACKAGE_ABS_SRCDIR=`cd $PACKAGE_SRCDIR; pwd`

# Ensure we have a PACKAGE name specification; if not, attempt to deduce
# a suitable default from the name of the top-level source code directory.
#
  pkgspec_from_srcdir(){
    pkgspec_request_class=pkgspec_$1
    #
    # An optional second argument overrides the default derivation.
    #
    IFS=-; set -- ${2-`basename "$PACKAGE_ABS_SRCDIR"`}; IFS=
    value= fs=; $pkgspec_request_class "$@"
  }
  pkgspec_get_name(){
    for part; do case $part in [0-9]*) break;;
	   *) value="$value$fs$part" fs=-;;
	 esac
      done
    echo "$value"
  }
  PACKAGE=${PACKAGE-`pkgspec_from_srcdir get_name`}
  test "x$PACKAGE" = x && PACKAGE='<package>'

# Likewise, deduce a suitable default for the VERSION specification.
#
  pkgspec_get_version(){
    for part; do case $part in [0-9]*) value="$value$fs$part" fs=-;;
	   *) test "x$value" = x || break;;
	 esac
      done
    echo "$value"
  }
  VERSION=${VERSION-`pkgspec_from_srcdir get_version`}
  test "x$VERSION" = x && VERSION='<version>'

# Define a set of simple macro-style functions, which we will use to
# interpret the content of the package specification file.
#
  project()	{ PROJECT="$*"; }
  home()	{ PROJECT_HOME="$1"; }

  pkgname()	{ PACKAGE="$1"; }
  pkgversion()	{ VERSION="$1"; }
  pkgrelease()	{ RELEASE="$1"; }

  title()	{ TITLE="$*"; }
  description() { DESCRIPTION="$*"; }

  pkghome()	{ PACKAGE_HOME_URI="$1"; }
  srcname()	{ PACKAGE_SOURCE_URI_NAME="$1"; }
  srcpath()	{ PACKAGE_SOURCE_URI_PATH="$1"; }

  arch()	{ ARCH="$1"; }

  pkgtype()	{
    ARCHIVE_CMD="tar"
    case $1 in .*) PKGEXT="$1" ;; *) PKGEXT=".$1" ;; esac
    case $PKGEXT in
      .tar )		COMPRESS_CMD="none"			 ;;
      .tgz | .tar.gz )	COMPRESS_CMD="gzip"			 ;;
      .tar.bz2 ) 	COMPRESS_CMD="bzip2"			 ;;
      .tar.lzma ) 	COMPRESS_CMD="lzma ${OPTION_VERBOSE+-v}" ;;
      .tar.xz ) 	COMPRESS_CMD="xz ${OPTION_VERBOSE+-v}"   ;;

      *) warning "unknown package type '$1'; reverting to '.tar.gz'"
	 COMPRESS_CMD="gzip" PKGEXT=".tar.gz"
    esac
  }

  affiliate()	{
    affiliate_key=`expr 1 + ${affiliate_key-"0"}`
    AFFILIATES=${AFFILIATES+"$AFFILIATES "}$affiliate_key
    eval AFFILIATE_$affiliate_key='"$*"'
  }

  subpackage()	{
    SUBPACKAGE_PREFIX="$1:"
    # SUBPACKAGE_LIST=`echo $SUBPACKAGE_LIST "$1"`
  }

  component()	{
    COMPONENTS=`echo $COMPONENTS "$SUBPACKAGE_PREFIX$1"`
    eval `sanitized_name $SUBPACKAGE_PREFIX$1`_package_content='"$2"'
  }

  alias()	{ ALIASES=`echo $ALIASES "$*"`; }
  aliases()	{ alias "$*"; }

  action_load_package_specs(){
    ALIASES= AFFILIATES= affiliate_key=0 COMPONENTS=
    test "x$1" = x && action_load_package_specs_plugin_hook || . "$1"
    TAGGED_VERSION="$VERSION${RELEASE+"-$RELEASE"}"
  }

  action_load_package_specs_plugin_hook(){
    TAGGED_VERSION="$VERSION${RELEASE+"-$RELEASE"}"
    test -f $PACKAGE_SRCDIR/arch/$ARCH/$PACKAGE-$TAGGED_VERSION-$ARCH.$pkgspec \
      && . $PACKAGE_SRCDIR/arch/$ARCH/$PACKAGE-$TAGGED_VERSION-$ARCH.$pkgspec  \
      || true
  }

# Finally, make an initial tentative attempt to load a package specification,
# if any is available at the default location within the current directory.
#
  action_load_package_specs
#
# -----------------------------------------------------------------------------
# $RCSfile: pkgspec.sh,v $: end of file
