/*
 * 3D ʂƉ~ʂ̌߂NX
 *
 * Copyright 2000 by Information-technology Promotion Agency, Japan
 * Copyright 2000 by Precision Modeling Laboratory, Inc., Tokyo, Japan
 * Copyright 2000 by Software Research Associates, Inc., Tokyo, Japan
 *
 * $Id: JgclIntsPlnCon3D.java,v 1.10 2000/08/11 06:18:52 shikano Exp $
 */

package jp.go.ipa.jgcl;

import java.util.*;

/**
 * 3D ʂƉ~ʂ̌߂NX
 *
 * @version $Revision: 1.10 $, $Date: 2000/08/11 06:18:52 $
 * @author Information-technology Promotion Agency, Japan
 */

class JgclIntsPlnCon3D {
    private JgclPlane3D dA;
    private JgclConicalSurface3D dB;

    private JgclPoint3D Bv;		/* the vertex of B */
    private JgclPoint3D pBv2A;		/* projected Bv to A */
    private JgclVector3D eB2A;		/* vector from Bv to pBv2A */
    private JgclVector3D pBz2A;		/* projected B's z to A */

    private double edot;		/* dot_product(A's z, B's z) */
    private double eBdot;		/* dot_product(B's z, pBz2A) */
    private JgclVector3D ecrs;		/* cross_product(A's z, B's z) */

    private double sinBsa;		/* sin(B's semi angle) */
    private double tanBsa;		/* tan(B's semi angle) */

    private double a_tol;

    private void setupParams(JgclPlane3D dA, JgclConicalSurface3D dB) {
	this.dA = dA;
	this.dB = dB;

	sinBsa = Math.sin(dB.semiAngle());
	tanBsa = Math.tan(dB.semiAngle());
	a_tol = dA.getToleranceForAngle();

	/*
	 * vertex of B
	 */
	double ework = dB.radius() / tanBsa;
	Bv = dB.position().location().subtract(dB.position().z().multiply(ework));

	/*
	 * project vertex of B to A
	 */
	JgclVector3D evec = dA.position().location().subtract(Bv);
	ework = dA.position().z().dotProduct(evec);
	eB2A = dA.position().z().multiply(ework);
	pBv2A = Bv.add(eB2A);

	/*
	 * project Z-axis of B to A
	 */
	edot = dA.position().z().dotProduct(dB.position().z());

	if (Math.abs(edot) < Math.cos(a_tol)) {
	    pBz2A = dB.position().z().subtract(dA.position().z().multiply(edot)).unitized();
	    eBdot = dB.position().z().dotProduct(pBz2A);	/* always greater than 0 */
	} else {
	    pBz2A = JgclVector3D.zeroVector;
	    eBdot = 0.0;
	}

	ecrs = dA.position().z().crossProduct(dB.position().z());
    }

    JgclIntsPlnCon3D(JgclPlane3D dA, JgclConicalSurface3D dB) {
	super();
	setupParams(dA, dB);
    }

    private JgclSurfaceSurfaceInterference3D[] oneLine(boolean doExchange) {
	JgclLine3D res = new JgclLine3D(Bv, pBz2A);
	JgclIntersectionCurve3D ints = dA.curveToIntersectionCurve(res, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] sol = {ints};
	return sol;
    }

    private JgclSurfaceSurfaceInterference3D[] twoLines(boolean doExchange) {
	double ework = Math.sqrt(1.0 + (tanBsa * tanBsa) - (1.0 / (eBdot * eBdot)));
	JgclVector3D ecrs2 = ecrs.unitized().multiply(ework);

	JgclLine3D res1
	    = new JgclLine3D(Bv, pBz2A.divide(eBdot).add(ecrs2).unitized());
	JgclLine3D res2
	    = new JgclLine3D(Bv, pBz2A.divide(eBdot).subtract(ecrs2).unitized());
	JgclIntersectionCurve3D ints1 = dA.curveToIntersectionCurve(res1, dB, doExchange);
	JgclIntersectionCurve3D ints2 = dA.curveToIntersectionCurve(res2, dB, doExchange);

	JgclSurfaceSurfaceInterference3D[] sol = {ints1, ints2};
	return sol;
    }

    private JgclSurfaceSurfaceInterference3D[] onePoint(boolean doExchange) {
	JgclPoint3D res = Bv;
	JgclIntersectionPoint3D intsPnt = dA.pointToIntersectionPoint(res, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] intf = {intsPnt};
	return intf;
    }

    private JgclSurfaceSurfaceInterference3D[] oneCircle(boolean doExchange) {
	double ework = eB2A.dotProduct(dB.position().z());
	JgclAxis2Placement3D axis
	    = new JgclAxis2Placement3D(Bv.add(dB.position().z().multiply(ework)),
				       dA.position().z(), dA.position().x());
	JgclCircle3D res = new JgclCircle3D(axis, Math.abs(ework) * tanBsa);
	JgclIntersectionCurve3D ints = dA.curveToIntersectionCurve(res, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] sol = {ints};
	return sol;
    }

    private JgclSurfaceSurfaceInterference3D[] oneParabola(boolean doExchange) {
	/*
	 * loc = pBv2A - (X / tanBsa) * pBz2A, (X is a length of eB2A).
	 */
	JgclPoint3D loc = pBv2A;
	JgclVector3D Bz = dB.position().z();
	JgclVector3D pBz2A2 = pBz2A;
	
	if (eB2A.dotProduct(Bz) < 0.0) {
	    Bz = Bz.reverse();
	    pBz2A2 = pBz2A2.reverse();
	}

	double ework = eB2A.length();

	if (Math.abs(dB.semiAngle() - (Math.PI / 4.0)) > a_tol) {
	    double ework2 = ework / Math.tan(2.0 * dB.semiAngle());
	    loc = loc.add(pBz2A2.multiply(ework2));
	}

	/*
	 * axis is A's axis
	 * ref_dir is B's axis
	 */
	JgclAxis2Placement3D axis
	    = new JgclAxis2Placement3D(loc, dA.position().z(), Bz);

	/*
	 * focal dist = X * tanBsa / 2
	 */
	JgclParabola3D res = new JgclParabola3D(axis, ework * tanBsa / 2.0);
	JgclIntersectionCurve3D ints = dA.curveToIntersectionCurve(res, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] sol = {ints};
	return sol;
    }

    private JgclHyperbola3D oneHyperbola() {
	double ecos, esin;
	double ework, ework2, ework3;

	/*
	 * loc
	 */
	JgclPoint3D loc = pBv2A;

	ecos = edot;
	esin = Math.sqrt(1.0 - ecos * ecos);

	ework = eB2A.length();

	ework2 = Math.abs((esin * esin * tanBsa * tanBsa) - (ecos * ecos));

	ework3 = Math.abs((ecos * esin * ework * (1.0 + tanBsa * tanBsa)) / ework2);

	if (eB2A.dotProduct(dB.position().z()) > 0.0) {
	    loc = loc.subtract(pBz2A.multiply(ework3));
	} else {
	    loc = loc.add(pBz2A.multiply(ework3));
	}
	/*
	 * axis is A's axis
	 * ref_dir is B's axis
	 */
	JgclAxis2Placement3D axis
	    = new JgclAxis2Placement3D(loc, dA.position().z(), dB.position().z());

	return new JgclHyperbola3D(axis,
				   ework * tanBsa / ework2,
				   ework * tanBsa / Math.sqrt(ework2));
    }

    private JgclSurfaceSurfaceInterference3D[] hyperbola(boolean doExchange) {
	JgclHyperbola3D res1 = oneHyperbola();

	JgclPoint3D loc = dB.position().location();
	JgclVector3D vec = Bv.subtract(loc).multiply(2.0);
	loc = loc.add(vec);
	JgclAxis2Placement3D axis2 = new JgclAxis2Placement3D(loc,
							      dB.position().z().reverse(),
							      dB.position().x().reverse());
	JgclConicalSurface3D saved_dB = dB;
	JgclConicalSurface3D dB2
	    = new JgclConicalSurface3D(axis2, dB.radius(), dB.semiAngle());
	setupParams(dA, dB2);
	JgclHyperbola3D res2 = oneHyperbola();

	setupParams(dA, saved_dB);
	JgclIntersectionCurve3D ints1 = dA.curveToIntersectionCurve(res1, dB, doExchange);
	JgclIntersectionCurve3D ints2 = dA.curveToIntersectionCurve(res2, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] sol = {ints1, ints2};
	return sol;
    }

    private JgclSurfaceSurfaceInterference3D[] oneEllipse(boolean doExchange) {
	double ecos, esin;
	double ework, ework2, ework3;

	ecos = edot;
	esin = Math.sqrt(1.0 - ecos * ecos);

	ework = eB2A.length();

	ework2 = Math.abs((esin * esin * tanBsa * tanBsa) - (ecos * ecos));

	ework3 = Math.abs((ecos * esin * ework * (1.0 + tanBsa * tanBsa)) / ework2);
	if (eB2A.dotProduct(dB.position().z()) < 0.0)
	    ework3 = (- ework3);

	/*
	 * axis is A's axis
	 * ref_dir is B's axis
	 */
	JgclAxis2Placement3D axis
	    = new JgclAxis2Placement3D(pBv2A.add(pBz2A.multiply(ework3)),
				       dA.position().z(), dB.position().z());

	JgclEllipse3D res = new JgclEllipse3D(axis,
					      ework * tanBsa / ework2,
					      ework * tanBsa / Math.sqrt(ework2));
	JgclIntersectionCurve3D ints = dA.curveToIntersectionCurve(res, dB, doExchange);
	JgclSurfaceSurfaceInterference3D[] sol = {ints};
	return sol;
    }

    JgclSurfaceSurfaceInterference3D[]
    getInterference(boolean doExchange) {
	JgclSurfaceSurfaceInterference3D[] result;

	/*
	 * LINE / 2 LINES / POINT / CIRCLE / PARABOLA / HYPERBOLA / ELLIPSE
	 */
	if (eB2A.length() < dA.getToleranceForDistance()) {
	    /*
	     * vertex of B is on A.
	     */
	    if (Math.abs(Math.acos(eBdot) - dB.semiAngle()) < a_tol) {	// line
		return oneLine(doExchange);
	    }
	    if (Math.acos(eBdot) < dB.semiAngle()) {			// 2 lines
		return twoLines(doExchange);
	    }
	    return onePoint(doExchange);
	}

	/*
	 * vertex of B is NOT on A
	 */
	if (ecrs.length() < Math.sin(a_tol)) {				// circle
	    return oneCircle(doExchange);
	}
	if (Math.abs(Math.acos(eBdot) - dB.semiAngle()) < a_tol) {	 // parabola
	    return oneParabola(doExchange);				// one parbola
	}
	if (Math.acos(eBdot) < dB.semiAngle()) {			// hyperbola
	    return hyperbola(doExchange);
	}

	return oneEllipse(doExchange);					// ellipse
    }
}
