# encoding: utf-8

require 'facets/pathname'
require 'rumix'
begin
	require '__wxautoload' # generated by Rake
rescue LoadError
	warn "WARNING: Running to depend on WxRuby autoload. It is virtual operation. Please rake '__wxautoload.rb' "
end
require 'wxconstructor'


module Rumix
	def self.run_on_thread(&proc)
		stop_running_thread
	
		Thread.abort_on_exception = true
		thread = Thread.new{
			begin
				proc.call
			rescue Exception
				open(File.join(Rumix.base_dir, 'operation_error.log'), 'w'){|f|
					f.puts $!.to_s
					f.puts $!.backtrace
				}
				Kernel.raise $!
			end
		}
		timer = Wx::Timer.every(1) do
			Thread.pass
		end
		
		@running_thread = thread
		@running_timer = timer
		
		return thread
	end
	
	def self.stop_running_thread
		if @running_thread then
			@running_thread.kill
		end
		
		if @running_timer then
			@running_timer.stop
		end
	end
	
	module Widget
		TITLE_FONT = Wx::Font.new(12, Wx::FONTFAMILY_DEFAULT, Wx::FONTSTYLE_NORMAL, Wx::FONTWEIGHT_BOLD)
		MAIN_SIZER_BORDER = 12
		
		module WindowName
			module ManualOptionItem
				CHM_NEW_REMIX = 'manual_new_chm_remix'
				CHM_NEW = 'manual_new_chm'
				CHM_OLD = 'manual_old_chm'
				NONE = 'manual_none'
			end
			module ShellOptionItem
				NYAOS_CKW = 'shell_nyaos_ckw'
				NYAOS = 'shell_nyaos'
				NONE = 'shell_none'
			end
			module DefaultRubyOptionItem
				PREFIX = 'default_ruby_'
				NONE = 'default_ruby_none'
			end
		end
		

		class OptionItem
			attr_reader :name, :label, :description, :selected, :options, :bitmaps
		
			def initialize(name, label, desc = nil, selected_flag_or_selected_item_name = false, opts = {})
				case selected_flag_or_selected_item_name
				when String
					selected = (selected_flag_or_selected_item_name == name)
				else
					selected = (selected_flag_or_selected_item_name ? true : false)
				end
				@name, @label, @description, @selected, @options = name, label, desc, selected, opts
				@bitmaps = []
			end
		end
		
		class HyperlinkItem
			attr_reader :url, :label 
		
			def initialize(url, label)
				@url, @label = url, label
			end
		end


		
		class TitleText < Wx::StaticText
			def initialize(*args)
				super
				self.font = TITLE_FONT
			end
		end


	
		class OptionPanel < Wx::Panel
			def initialize(parent, items)
				super(parent)
				
				construct_children do |panel|
					# ラジオグループ全体を入れるSizer
					topsz = vbox_sizer do
						vertical_border 8
						expand
						items.each_with_index do |item, i|
							# 各ラジオボタンの関連パーツをまとめるSizer
							vbox_sizer do
								expand
								hbox_sizer do
									widget panel.option_widget_class, :name => item.name, :label => item.label, :style => (i == 0 ? Wx::RB_GROUP : 0) do |widget|
										widget.value = item.selected
									end
									item.bitmaps.each do |bitmap|	
										widget Wx::StaticBitmap, :label => bitmap
									end
								end
								
			
								if item.description then
									# 付帯情報領域と左マージンを含んだSizer
									hbox_sizer do
										stretch_spacer 1
										proportion 11
										# 付帯情報領域のSizer
										static_vbox_sizer('') do
											widget Wx::StaticText, :label => item.description do |text|
												text.name = item.options[:description_name] if item.options[:description_name]
											end
											
											if item.options[:path_name] then
												expand
												hbox_sizer do
													border 4
													proportion 1
													widget Wx::TextCtrl, :name => (item.options[:path_name] + '_text'), :value => item.options[:path_default]
													proportion 0
													widget Wx::Button, :label => '参照', :name => (item.options[:path_name] + '_button') do |button|
														panel.evt_button(button, :on_select_dir)
													end # widget
												end # sizer
											end # if
											
											if (links = item.options[:hyperlinks]) then
												expand
												widget HyperlinkPanel, links
											end
										end # static_vbox_sizer
										
									end # hbox_sizer
								end # if item.description
							end # vbox_sizer
						end # items.each_with_index
					end # topsz
					
					topsz.fit(panel)
							
				end
			end
			
			def on_select_dir(evt)
				text = find_window_by_name(evt.event_object.name.sub(/_button$/, '_text'))
				dialog = Wx::DirDialog.new(self, :default_path => text.value)
				if dialog.show_modal == Wx::ID_OK then
					text.value = dialog.path
				end
				
			end
	
		end
		
		
		class RadioOptionPanel < OptionPanel
			def option_widget_class 
				Wx::RadioButton
			end
		end
		
		class CheckOptionPanel < OptionPanel
			def option_widget_class 
				Wx::CheckBox
			end
		end
		
		class HyperlinkPanel < Wx::Panel
			attr_reader :hyperlink_ctrls
			attr_accessor :top_sizer
			def initialize(parent, links)
				super(parent)
				@hyperlink_ctrls = []
				update_links(links)
			end
			
			def update_links(links)
				@hyperlink_ctrls.each do |ctrl|
					ctrl.hide
					ctrl.destroy
				end
				@hyperlink_ctrls.clear
				
				construct_children do |panel|
					panel.top_sizer = hbox_sizer do
						border 4
						stretch_spacer 1
						links.each do |item|
							ctrl = widget Wx::HyperlinkCtrl, :url => item.url, :label => "[#{item.label}]", :style => Wx::HL_CONTEXTMENU | Wx::NO_BORDER | Wx::HL_ALIGN_RIGHT do |hyperlink|
								hyperlink.tool_tip = hyperlink.url
								hyperlink.visited_colour = hyperlink.normal_colour
							end
							panel.hyperlink_ctrls << ctrl
						end
					end
					
					panel.top_sizer.fit(panel)
				end

			end
			
		end


		
	
		class Wizard < Wx::Wizard

			def self.parse_options(argv)
				re = {}
				
				ropt_parsed = ROpt.parse(argv, 'd:h', 'dest:', 'ini:', 'system-safe', 'dry-run', 'help')
				if ropt_parsed then
					# -h, --help
					if ropt_parsed['help'] || ropt_parsed['h'] then
						show_help_for_gui
						exit
					end

					# -d, --dest
					dest_dir = ropt_parsed['dest'] || ropt_parsed['d']
					if dest_dir then
						re[:dest_dir] = Rumix.winpath(File.expand_path(dest_dir))
					end

					# -d, --dest
					ini_path = ropt_parsed['ini']
					if ini_path then
						re[:ini_path] = File.expand_path(ini_path)
					end


					# --dry-run and --system-safe 
					if ropt_parsed['dry-run'] then
						re[:operation_level] = OperationLevel::DRY_RUN
					elsif ropt_parsed['system-safe'] then
						re[:operation_level] = OperationLevel::SYSTEM_SAFE
					end

					
					return re
				else
					self.show_help_for_gui
					exit
				end
			end

			def self.show_help_for_gui
				file_name = (defined?(ExerbRuntime) ? ExerbRuntime.filename : File.basename($0))
				help_text = <<TEXT
Rumix #{Rumix::VERSION} - Ruby Starter Package with Installer

使い方: #{file_name} [options]

オプション:
  -d, --dest           インストール先のフォルダパスを指定します。

      --system-safe    システムセーフモード。レジストリや環境変数の書き込み、
                       スタートメニューのショートカット作成、
                       一部コマンドの実行など、OS環境全体にかかわる
                       変更を行いません。
                       (インストール先フォルダへのファイルの書き込みは
                        通常通り行います)

      --dry-run        dry runモード。上記システムセーフモードの制限に加えて
                       さらに実際のファイル書き込みも行いません。

  -h, --help           このヘルプメッセージを表示します。

TEXT

				$stderr.puts(Uconv.u8tosjis(help_text))
			end



		
			def initialize(gui_options = {}, *args)
				super(*args)
				@gui_options = gui_options

				@tool_page_shown = false
				@canceled = false
				
				pages = []
				config = Rumix::Config.new.load_ini_file(gui_options[:ini_path] || 'rumix.ini')
				osi = OSInterface.new(config, $stderr)
				
				# 起動時のオプションでインストール先が指定された場合や、前回のインストール情報がレジストリに残っている場合
				# インストール先から前回インストール時の情報を読み出す
				last_install_info = nil
				last_install_dir_path = nil
				if gui_options[:dest_dir] then
					last_install_dir_path = gui_options[:dest_dir]
				else
					last_install_dir_path = osi.read_install_path_from_registry
				end
				
				if last_install_dir_path then
					install_info_pathname = Pathname[last_install_dir_path] / '.rumix-inst/rumix-setting.yml'
					if install_info_pathname.exist? then
						begin
							last_install_info = YAML.load_file(install_info_pathname.to_s)
						rescue
						end
					end
				end

				# 各ペイン組み立て
				construct_children do |wizard|
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							border MAIN_SIZER_BORDER
							widget TitleText, :label => 'Welcome to Rumix'
							widget Wx::StaticText, :label => "#{config.rumix_edition_name} のインストールを開始します。\n「次へ」をクリックしてください。"

							proportion 1
							@location_args[1] |= (Wx::ALIGN_RIGHT | Wx::ALIGN_BOTTOM)
							
							path = 'res/image/ce_jacas/ruby_logo_text_bottom.png'
							if File.exist?(path) then
								widget Wx::StaticBitmap, :label => Wx::Bitmap.new(path, Wx::BITMAP_TYPE_PNG)
							end
							
							proportion 0
							vbox_sizer do
								ruby_desc = config.ruby_configs.map{|x| x.ruby_description}.join("\n")
								widget Wx::StaticText, :label => "インストールされる ruby:\n#{ruby_desc}"

								if gui_options[:operation_level] and gui_options[:operation_level] < OperationLevel::FULL then
									widget Wx::StaticText, :label => ""
									widget Wx::StaticText, :label => "★オペレーションレベル: #{gui_options[:operation_level]}"
								end
							
								links = []
								links << HyperlinkItem.new('http://rubyinstaller.org/', '配布元サイト（英語）')
								expand
								widget HyperlinkPanel, links
							end
							
						end
					end
					
					
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							border MAIN_SIZER_BORDER
							widget TitleText, :label => 'インストール先指定'
							widget Wx::StaticText, :label => "Rumixのインストール先を指定してください。\n指定した位置にフォルダが存在しない場合は、自動的に作成します。"
							
							expand
							hbox_sizer do
								border 4
								add_sizer_flag Wx::ALIGN_CENTER_VERTICAL
								
								proportion 1
								widget Wx::TextCtrl, :value => (last_install_dir_path || 'C:\\rumix'), :name => 'dest'
								proportion 0
								widget Wx::Button, :label => '参照' do |button|
									wizard.evt_button(button, :on_select_dest)
								end
							end
						end
					end

					# 標準ruby選択ページ（pikをインストールする場合のみ表示）
					if config.pik_should_be_installed? then
						pages << widget(Wx::WizardPageSimple) do
							vbox_sizer do
								expand
								border MAIN_SIZER_BORDER
								widget TitleText, :label => '標準のrubyを選択'
								widget Wx::StaticText, :label => <<TEXT
このパッケージでは、#{config.ruby_configs.size}種類のバージョンのrubyと
使用するrubyの切り替えを行うコマンドラインツール「pik」を、同時にインストールし
複数のrubyを切り替えて使うことができるように設定します。

システムが標準で使用するrubyを選択してください。
（選択したrubyのパスを、環境変数PATHに追加します）
TEXT

								# 初期選択項目の決定
								selected_item_name = WindowName::ManualOptionItem::CHM_NEW_REMIX
								if last_install_info then
									case last_install_info['manual_selected']
									when ManualType::CHM_NEW
										selected_item_name = WindowName::ManualOptionItem::CHM_NEW
									when ManualType::CHM_OLD
										selected_item_name = WindowName::ManualOptionItem::CHM_OLD
									when ManualType::NONE
										selected_item_name = WindowName::ManualOptionItem::NONE
									end
								end

								items = []
								config.ruby_configs.each_with_index do |ruby_conf, i|
									items << OptionItem.new("#{WindowName::DefaultRubyOptionItem::PREFIX}#{ruby_conf.ruby_type}", ruby_conf.ruby_short_description + (i == 0 ? ' [推奨]' : ''), "#{ruby_conf.ruby_short_description} を標準で使用します。", selected_item_name)
								end
								items << OptionItem.new(WindowName::DefaultRubyOptionItem::NONE, "標準のrubyを設定しない [パワーユーザー向け]", "標準のrubyを設定せず、環境変数PATHへrubyのパスを追加することも行いません。", selected_item_name)
								widget RadioOptionPanel, items
								
								links = []
								links << HyperlinkItem.new('https://github.com/vertiginous/pik', 'pik配布ページ (英語)')
								links << HyperlinkItem.new('http://www.ruby.or.jp/ja/tech/install/ruby/pik.html', 'pik解説ページ')
								widget HyperlinkPanel, links
							end
						end	
					end
		
=begin
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							expand
							border MAIN_SIZER_BORDER
							widget TitleText, :label => '外部ライブラリのインストール先指定'
							widget Wx::StaticText, :label => "rubyを使うのに必要な、各種外部ライブラリ（zlib, readline, iconv, ...）を\nインストールする場所を選択してください。"
							
							items = []
							items << OptionItem.new('ext_system', 'システムフォルダ [推奨]', "Windowsの標準システムフォルダにインストールします。\n  パス: #{Rumix.system_folder_path}", false, {:description_name => 'ext_system_description'})
							items << OptionItem.new('ext_ruby', 'rubyと同じ場所', wizard.get_ext_ruby_description('-'), false, {:description_name => 'ext_ruby_description'})
							items << OptionItem.new('ext_manual', '他の場所', "自分でパスを指定します。", false, {:path_name => 'ext_dest', :path_default => 'C:\\'})
							items << OptionItem.new('ext_no', 'インストールしない')
							widget RadioOptionPanel, items
						end
					end
=end
		
			
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							expand
							border MAIN_SIZER_BORDER
							widget TitleText, :label => 'Rubyリファレンスマニュアルの選択'
							widget Wx::StaticText, :label => "インストールするRubyリファレンスマニュアルの種類を選択してください。"

							# 初期選択項目の決定
							selected_item_name = WindowName::ManualOptionItem::CHM_NEW_REMIX
							if last_install_info then
								case last_install_info['manual_selected']
								when ManualType::CHM_NEW
									selected_item_name = WindowName::ManualOptionItem::CHM_NEW
								when ManualType::CHM_OLD
									selected_item_name = WindowName::ManualOptionItem::CHM_OLD
								when ManualType::NONE
									selected_item_name = WindowName::ManualOptionItem::NONE
								end
							end

							new_chm_remix_links = [
								HyperlinkItem.new('http://ruby.morphball.net/refm-remix.html', '配布ページ'),
								HyperlinkItem.new('http://ruby.morphball.net/theme-sample/default/', 'サンプル'),
							]
							new_chm_remix_desc = "「リファレンスマニュアル刷新計画」により、作成が進んでいる\n新しいリファレンスマニュアルに、目次表示の分類分けや表示形式の変更など\nいくつかの変更を加えたものです。"
							new_chm_links = [
								HyperlinkItem.new('http://www.ruby-lang.org/ja/documentation/', '配布ページ'),
								HyperlinkItem.new('http://doc.ruby-lang.org/ja/1.9.3/doc/index.html', 'サンプル'),
							]
							new_chm_desc =  "「リファレンスマニュアル刷新計画」により、作成が進んでいる\n新しいリファレンスマニュアルです。"
							
							items = []
							items << OptionItem.new(WindowName::ManualOptionItem::CHM_NEW_REMIX, '新リファレンスマニュアル chm版リミックス [推奨]', new_chm_remix_desc, selected_item_name, :hyperlinks => new_chm_remix_links)
							items << OptionItem.new(WindowName::ManualOptionItem::CHM_NEW, '新リファレンスマニュアル（chm版）', new_chm_desc, selected_item_name, :hyperlinks => new_chm_links)

							items << OptionItem.new(WindowName::ManualOptionItem::NONE, 'インストールしない', nil, selected_item_name)
							
							widget RadioOptionPanel, items
							
						end
					end	
					
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							expand
							border MAIN_SIZER_BORDER
							widget TitleText, :label => 'コマンドラインツールの選択'
							widget Wx::StaticText, :label => "同時にインストールするコマンドラインツールの種類を選択してください。"
							
							# 初期選択項目の決定
							selected_item_name = WindowName::ShellOptionItem::NYAOS_CKW
							if last_install_info then
								case last_install_info['shell_selected']
								when ShellType::NYAOS
									selected_item_name = WindowName::ShellOptionItem::NYAOS
								when ShellType::NONE
									selected_item_name = WindowName::ShellOptionItem::NONE
								end
							end
							
							# 選択項目を設定
							items = []
							items << OptionItem.new(WindowName::ShellOptionItem::NYAOS_CKW, "NYAOS #{NYAOS_VERSION} + ckw-mod #{CKW_VERSION} [推奨]", "コマンドの実行を便利にしてくれる、コマンドラインシェルNYAOSと\n外観や使い勝手を改善してくれる、端末ソフトウェアckwのセットです。", selected_item_name)
							#items.last.bitmaps << Wx::Image.new("d:/temp/luacat.png", Wx::BITMAP_TYPE_PNG).rescale(16, 16, Wx::IMAGE_QUALITY_NORMAL).convert_to_bitmap
							#items.last.bitmaps << Wx::Image.new("d:/nyaos/icon.ico", Wx::BITMAP_TYPE_ICO).rescale(16, 16, Wx::IMAGE_QUALITY_NORMAL).convert_to_bitmap

							
							items << OptionItem.new(WindowName::ShellOptionItem::NYAOS, "NYAOS #{NYAOS_VERSION}", "コマンドの実行を便利にしてくれる、コマンドラインシェルです。", selected_item_name)
							items << OptionItem.new(WindowName::ShellOptionItem::NONE, 'インストールしない', nil, selected_item_name)
							widget RadioOptionPanel, items
							
							links = []
							links << HyperlinkItem.new('http://www.nyaos.org/', 'NYAOS配布ページ')
							links << HyperlinkItem.new('http://deflis.github.com/ckw-mod/', 'ckw-mod配布ページ')
							widget HyperlinkPanel, links
						end
					end	

					unless config.tool_list.empty? then
					
						pages << widget(Wx::WizardPageSimple) do |page|
							hyperlink_panel = nil
							vbox_sizer do
								expand
								border MAIN_SIZER_BORDER
								widget TitleText, :label => 'インストールパッケージの選択'
								widget Wx::StaticText, :label => "同時にインストールする、ツールやライブラリを選んでください。\nよく分からない場合は、とりあえず全部入れておくのがお勧めです。"
								
								hbox_sizer do
									border 4
									expand
								
									choices = config.tool_list.map{|x| x.name}
									list_box = widget Wx::CheckListBox, :name => 'tool_list', :choices => choices, :size => [-1, 160] do |list|
										config.tool_list.each_with_index do |tool, i|
											if last_install_info then
												# 最終インストール情報が見つかった場合、最後のチェック状態と同じ状態にする
												if last_install_info['tool_selected'][tool.id.to_s] then
													list.check(i)
												end
											else
												list.check(i)
											end
										end
										
										list.evt_listbox(list) do |evt|
											guide_url, dist_url = nil, nil
											guide_link_label = '[解説ページ]'
											dist_link_label = '[配布ページ]'
											
											desc = ''
										
											case config.tool_list[evt.index].id
											when :devkit
												desc = "　Rubyではパッケージ管理システム「RubyGems」を用いて、新しいツールやライブラリのインストールを行うことができます。その際に、通常はWindows向けのパッケージが用意されていない一部のパッケージも、インストールできるようにしてくれるツールキットです。\n\n　（拡張ライブラリをコンパイルするためのツール類が付属しています。インストール先に、追加で約170MBの空き容量が必要です）"
												dist_url = 'http://rubyinstaller.org/add-ons/devkit/'
												dist_link_label = '[配布ページ（英語）]'
												guide_url = 'http://d.hatena.ne.jp/kitamomonga/20110208/ruby_gem_install_requires_make_on_windows'
											when :ocra
												desc = '　Rubyのスクリプトや拡張ライブラリを、単一で動作可能なexeファイル（Windows実行形式ファイル）に変換するソフトウェアです。'
												dist_link_label = '[配布ページ（英語）]'
												dist_url = 'http://ocra.rubyforge.org/'
												guide_url = 'http://blog.livedoor.jp/maru_tak/archives/51253027.html'
											when :exerb
												desc = '　Rubyのスクリプトや拡張ライブラリを、単一で動作可能なWindows実行形式ファイル（exeファイル）に変換するソフトウェアです。'
												dist_url = 'http://exerb.sourceforge.jp/'
											when :rake
												desc = '　Ruby-Make（Makeによく似たビルドツール）です。ファイルを扱う定型的な処理を、一連のRubyスクリプトとして定義して、簡単に実行することができます。'
												guide_url = 'http://www2s.biglobe.ne.jp/~idesaku/sss/tech/rake/'
												dist_url = 'http://rake.rubyforge.org/'
												dist_link_label = '[配布ページ（英語）]'
											when :facets
												desc = '　Rubyを便利にしてくれる、小さな拡張スクリプトの詰め合わせライブラリです。String, Array, Hash, Kernelなどの組み込みクラスに、便利なメソッドが追加されます。'
												dist_url = 'http://rubyworks.github.com/facets/'
												dist_link_label = '[配布ページ（英語）]'
											when :visualuruby
												desc = '　Windows上で動くGUIアプリケーションを楽に作るためのライブラリです。リファレンスマニュアルと、マウスで手軽にウインドウを配置できるツール「FormDesigner」も同梱しています。'
												dist_url = 'http://www.osk.3web.ne.jp/~nyasu/software/vrproject.html'
											when :refe
												desc = "　コマンドラインから日本語のRubyリファレンスマニュアルを検索・表示することができるツールです（BitClustというツールの一部として動作します）。Ruby #{config.default_ruby_config.ruby_version} に対応した、リファレンスマニュアルのデータを含んでいます。"
												dist_url = 'http://redmine.ruby-lang.org/wiki/rurema'
												guide_url = 'http://www.loveruby.net/w/ReFe.html'
											when :rdoc_data
												desc = "　Rubyに標準添付されているツール「ri」用のドキュメントデータです。Rubyの組み込みクラスや、添付ライブラリに関する情報を英語で検索することができます。"
												dist_url = 'http://redmine.ruby-lang.org/wiki/rurema'
												guide_url = 'http://www.loveruby.net/w/ReFe.html'
											when :ansicon_win32console
												desc = "　Windows上のコマンドライン端末上で、色や特殊な制御文字を解釈できるようにするツール2つのセットです（これらのツールがインストールされていると、一部のRuby製ツールでは、コマンドライン端末に色つきの文字を出力するようになります）。"
												dist_url = 'https://github.com/adoxa/ansicon'
												dist_link_label = '[ANSICON配布ページ（英語）]'
												guide_url = 'https://github.com/luislavena/win32console'
												guide_link_label = '[Win32::Console配布ページ（英語）]'
											when :sqlite3
												desc = "　軽量なデータベースエンジン「SQLite3」（dll版）と、そのSQLite3をrubyから扱うことができる「sqlite3-ruby」のセットです。"
											end
											
											page.find_window_by_name('tool_description').value = desc
												
											guide_link_ctrl, dist_link_ctrl = page.find_window_by_name('tool_link_panel').hyperlink_ctrls
											if guide_url then
												guide_link_ctrl.url = guide_url
												guide_link_ctrl.tool_tip = guide_url
												guide_link_ctrl.label = guide_link_label
												guide_link_ctrl.show
											else
												guide_link_ctrl.hide
											end
											
											if dist_url then
												dist_link_ctrl.url = dist_url
												dist_link_ctrl.tool_tip = dist_url
												dist_link_ctrl.label = dist_link_label
												dist_link_ctrl.show
											else
												dist_link_ctrl.hide
											end
											
											# 表示を更新してサイザーの再配置
											dist_link_ctrl.refresh
											guide_link_ctrl.refresh
											hyperlink_panel.top_sizer.layout
										end
									end
									
									
									proportion 1
									widget(Wx::TextCtrl, :style => Wx::TE_READONLY | Wx::TE_MULTILINE,
									       :name => 'tool_description', :value => 'ツールを選択すると、ここに解説が表示されます。')
									
									
								end # end of hbox sizer
								
								links = [HyperlinkItem.new('dummy', '解説ページ'), HyperlinkItem.new('dummy', '配布ページ（英語）')]
								hyperlink_panel = widget HyperlinkPanel, links do |panel|
									panel.name = 'tool_link_panel'
								end
								
								

							end # end of vbox sizer
						end	
					end
					
					
					
	

					pages << widget(Wx::WizardPageSimple) do |page|
						vbox_sizer do
							expand
							border MAIN_SIZER_BORDER
							widget TitleText, :label => 'オプション'
							widget Wx::StaticText, :label => "よく分からない場合は、チェックを入れたまま次に進んでください。"
							
							option_selected = nil
							if last_install_info and last_install_info['option_selected'] then
								option_selected = last_install_info['option_selected']
							end
							
							items = []
							unless config.pik_should_be_installed? then
								items << OptionItem.new('add_path_env', "環境変数PATHに ruby.exe のパスを追加する [推奨]", nil, option_selected.nil? || option_selected['add_path_env'])
							end
							items << OptionItem.new('add_start_menu', "スタートメニューにRumixを追加する", nil, option_selected.nil? || option_selected['add_start_menu'])
							items << OptionItem.new('no_force_overwriting', "すでにファイルが存在する場合、新しいファイルのみ上書き", nil, option_selected.nil? || !option_selected['force_overwriting'])

							widget CheckOptionPanel, items
							
							static_vbox_sizer('パワーユーザー向け') do
								horizontal_border 8
								widget CheckOptionPanel, [OptionItem.new('no_rdoc_and_no_ri', "gemのインストール時にrdoc, ri用のドキュメントを自動生成しない (大幅な時間短縮)", nil, option_selected && option_selected['no_rdoc_and_no_ri'])]
							end

						end
					end	

		
					
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							expand
						
							border MAIN_SIZER_BORDER
							widget TitleText, :label => '最終確認'
							widget Wx::StaticText, :label => "実行する内容を確認してください。"
							widget Wx::ListBox, :name => 'operation_list', :size => [-1, 200]
							widget Wx::StaticText, :label => "「インストール」を押すと開始します。"
						end
					end
					
					pages << widget(Wx::WizardPageSimple) do
						vbox_sizer do
							expand
						
							border MAIN_SIZER_BORDER
							widget TitleText, :name => 'installing_title', :label => 'インストール'
							widget Wx::StaticText, :style => Wx::CLIP_CHILDREN, :name => 'installing_console', :label => ''
							widget Wx::Gauge, :name => 'progress_gauge', :range => 100, :style => Wx::GA_SMOOTH
							widget Wx::StaticText, :style => Wx::CLIP_CHILDREN, :name => 'total_installing_console', :label => ''
							widget Wx::Gauge, :name => 'total_progress_gauge', :range => 100, :style => Wx::GA_SMOOTH
							widget Wx::StaticText, :style => Wx::CLIP_CHILDREN, :name => 'debug_console', :label => ''
						end
					end

					
					(0..(pages.size - 2)).each do |i|
						Wx::WizardPageSimple.chain(pages[i], pages[i+1])
					end
			
					sz = wizard.page_area_sizer
					pages.each do |page|
						sz.add(page)
					end
				end
				
				@pages = pages
				
				evt_wizard_finished(self, :on_finish)
				evt_wizard_page_changed(self, :on_page_change)
				evt_wizard_cancel(self, :on_cancel)
				#evt_close(:on_close)

				run_wizard(@pages.first)
				destroy
				
			end
			
			def on_select_dest
				dialog = Wx::DirDialog.new(self, :message => 'インストール先の選択：', :default_path => find_window_by_name('dest').value)
				if dialog.show_modal == Wx::ID_OK then
					find_window_by_name('dest').value = dialog.path
				end
				
			end
			
			def update_operation_list
				list = find_window_by_name('operation_list')
				list.clear
				
				make_rumix_config.operation_list.each do |op|
					list.append(op)
				end
			end
			
			def make_rumix_config
				conf = Rumix::Config.new(find_window_by_name('dest').value.gsub('\\', '/'))
				conf.load_ini_file(@gui_options[:ini_path] || 'rumix.ini')

				# コマンドライン等から指定したオプションの処理
				if @gui_options[:operation_level] then
					conf.operation_level = @gui_options[:operation_level]
				end

				# 標準ruby設定と、環境編数PATHへの追加設定（pikをインストールするか否かによって処理が異なる）
				if conf.pik_should_be_installed? then
					# pikをインストールする場合
					if find_window_by_name(WindowName::DefaultRubyOptionItem::NONE).value then
						# 標準rubyを選択しなかった
						conf.selected_ruby_type = nil
						conf.add_path_env = false
					else
						# 標準rubyを選択した
						selected_ruby_conf = conf.ruby_configs.find{|x| find_window_by_name(WindowName::DefaultRubyOptionItem::PREFIX + x.ruby_type).value }
						conf.selected_ruby_type = selected_ruby_conf.try.ruby_type # ruby type is String or nil
						conf.add_path_env = true
					end
				else
					# pikをインストールしない場合
					conf.add_path_env = (find_window_by_name('add_path_env').value ? true : false)
				end

				# 
				if find_window_by_name(WindowName::ManualOptionItem::CHM_NEW_REMIX).value then
					conf.man_type = ManualType::CHM_NEW_REMIX
				elsif find_window_by_name(WindowName::ManualOptionItem::CHM_NEW).value then
					conf.man_type = ManualType::CHM_NEW
				elsif find_window_by_name(WindowName::ManualOptionItem::NONE).value then
					conf.man_type = ManualType::NONE
				end
				
				if find_window_by_name(WindowName::ShellOptionItem::NYAOS_CKW).value then
					conf.shell_type = ShellType::NYAOS_CKW
				elsif find_window_by_name(WindowName::ShellOptionItem::NYAOS).value then
					conf.shell_type = ShellType::NYAOS
				elsif find_window_by_name(WindowName::ShellOptionItem::NONE).value then
					conf.shell_type = ShellType::NONE
				end
				
				tool_list_window = find_window_by_name('tool_list')
				if tool_list_window then
					conf.tool_list.each_with_index do |tool, i|
						if tool_list_window.is_checked(i) then
							conf.installing_tool_ids << tool.id
						end
					end
				end

				# オプションスイッチの選択内容をConfigに設定
				if find_window_by_name('add_start_menu').value then
					conf.add_start_menu = true
				end
				
				conf.force_overwriting = !(find_window_by_name('no_force_overwriting').value)
				
				
				return conf

			end
			
			def on_finish
				Kernel.exit 0
			end
			

			
			# キャンセル時の処理
			def on_cancel
				puts "canceled."

				# 更新用スレッドを殺す（これが残っていると、更新用スレッドの中で表示しないウインドウを更新しようとしてエラーになる）
				Thread.kill(@updating_worker) if @updating_worker

				# キャンセルフラグをONにする
				# メイン処理が走っている途中にキャンセルしても、メイン処理が終わるまで走り続けてしまうため
				# このフラグでメイン処理を強制的に中断するようにしている
				@canceled = true
			end
			
			def on_page_change(evt)
				buttons = {}
				self.children.each do |widget|
					if widget.label =~ /Finish|終了/ then
						widget.label = '終了'
						buttons[:finish] = widget
					elsif widget.label =~ /Next|次へ|インストール/ then
						if evt.page == @pages[-2] then
							widget.label = 'インストール'
						else
							widget.label = '次へ(&N) >'
						end
						buttons[:next] = widget
					elsif widget.label =~ /Back|戻る/ then
						widget.label = '< 戻る(&B)'
						buttons[:back] = widget
					elsif widget.label =~ /Cancel|中止/ then
						widget.label = '中止(&C)'
						buttons[:cancel] = widget
					end
				end
				
				text = find_window_by_name('ext_ruby_description')
				#text.label = get_ext_ruby_description(File.join(find_window_by_name('dest').value, 'ruby/bin').gsub('/', '\\'))


				if (panel = evt.page.find_window_by_name('tool_link_panel')) then
					# ツール選択画面用。このタイミングでhideしないと、うまく動かない（showしても表示されなくなる）。
					unless @tool_page_shown then
						panel.hyperlink_ctrls.each{|x| x.hide}
						@tool_page_shown = true
					end

				elsif evt.page == @pages[-2] then
					update_operation_list
				elsif evt.page == @pages.last then
					buttons[:back].disable
					buttons[:finish].disable
					
					installing_console = find_window_by_name('installing_console')
					total_installing_console = find_window_by_name('total_installing_console')
					gauge = find_window_by_name('total_progress_gauge')
					task_gauge = find_window_by_name('progress_gauge')
					
					Wx::Timer.after(100) do
						app = Wx.get_app
						dispatcher = nil
						open('rumix.log', 'w'){|log_io|
							config = make_rumix_config
							app.dispatch while app.pending
							
							# タスクの数を調べる
							installing_console.label = 'ファイル数を調査中...'
							total_installing_console.label = "全体の進捗: 0%"
							total = 0
							
							# インストールするファイル数を調査する
							arc_info = YAML.load_file('package/archives_info.yml')

							config.ruby_configs.each do |ruby_config|
								total += get_zip_file_number(arc_info, ruby_config.ruby_zip_file_name)
								app.dispatch while app.pending
							end

							config.ruby_manual_zip_file_names.each do |name|
								total += get_zip_file_number(arc_info, name)
								app.dispatch while app.pending
							end
							
							total += get_zip_file_number(arc_info, 'uninstaller.zip')
							app.dispatch while app.pending
								
							case config.shell_type
							when ShellType::NYAOS_CKW
								total += get_zip_file_number(arc_info, 'nyaos.zip')
								total += get_zip_file_number(arc_info, 'ckw-mod.zip')
							when ShellType::NYAOS
								total += get_zip_file_number(arc_info, 'nyaos.zip')
							end
							app.dispatch while app.pending

							# pik
							if config.pik_should_be_installed? then
								total += get_gem_weight(arc_info, config, 'pik')
							end
								
							extra_gem_weight_total = 0
							config.installing_tool_ids.each do |tool_id|
								case tool_id
								when :devkit
									config.devkit_zip_file_names.each do |devkit_zip_name|
										total += get_zip_file_number(arc_info, devkit_zip_name)
									end
								when :refe
									total += get_zip_file_number(arc_info, 'refmdb-1.8.7.zip')
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'bitclust-core')
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'rack')
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'refe2')
								when :ocra
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'ocra')
								when :facets
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'facets')
								when :bundler
									extra_gem_weight_total += get_gem_weight(arc_info, config, 'bundler')
								end
							end
							total += extra_gem_weight_total
							app.dispatch while app.pending

							
							i = 0
							task_progress = nil
							task_total = nil
							gauge.range = total
							
							desc = ''
							@old_label = ''
							@old_total_label = ''
							progress_io = open('rumix_progress.log', 'w')
							
							#Wx::Timer.every(25) do
							Thread.abort_on_exception = true
							@updating_worker = Thread.new do		
								loop do
									refresh_progress_gauge(app, installing_console, total_installing_console, gauge, task_gauge,
									                       desc, i, total, task_progress, task_total, progress_io)
								end
							end
							
							refresh_proc = proc do
								refresh_progress_gauge(app, installing_console, total_installing_console, gauge, task_gauge,
								                       desc, i, total, task_progress, task_total, progress_io)
							end

							catch(:application_loop) do
								Rumix::Application.run(config, log_io){|event_id, args|
									app.dispatch while app.pending

									throw(:application_loop) if @canceled


									case event_id
									when :on_uninstaller_install_start
										desc = 'アンインストーラをインストール中...'

									when :on_ruby_install_start
										installing_ruby_config = args.first
										desc = "#{installing_ruby_config.ruby_short_description}をインストール中..."
										task_total = get_zip_file_number(arc_info, installing_ruby_config.ruby_zip_file_name)
										task_progress = 0

									when :on_ruby_install_complete
										10.times do
											refresh_proc.call
										end
										task_total = nil
									when :on_devkit_install_start
										devkit_type, zip_name = args
										desc = "DevKit (#{devkit_type}版) をインストール中..."
										
										task_total = get_zip_file_number(arc_info, zip_name)
										task_progress = 0
									when :on_devkit_setup_start
										desc = 'DevKitを設定中...'
										wakeup_dispatcher(task_gauge)
									when :on_devkit_install_complete
										kill_dispatcher
										task_total = nil
									when :on_ansicon_install_start
										desc = 'ANSICONをインストール中...'
									when :on_win32console_install_start
										desc = 'win32console gemをインストール中...'
										wakeup_dispatcher(task_gauge)
									when :on_win32console_install_complete
										kill_dispatcher
									when :on_gem_install_start
										gem_title = args.first
										desc = "#{gem_title}をインストール中..."
										wakeup_dispatcher(task_gauge)
									when :on_gem_install_complete
										gem_names = args
										i += get_gem_weight(arc_info, config, *gem_names)
										kill_dispatcher
									when :on_pik_install_start
										desc = "pikをインストール中..."
										wakeup_dispatcher(task_gauge)
									when :on_pik_install_complete
										i += get_gem_weight(arc_info, config, 'pik')
										kill_dispatcher
									when :on_ext_install_start
										desc = '外部ライブラリをインストール中...'
									when :on_man_install_start
										desc = 'リファレンスマニュアルをインストール中...'
									when :on_man_install_complete
									when :on_refe_document_install_start
										desc = "ReFe2に必要なドキュメントデータをインストール中..."
										task_total = get_zip_file_number(arc_info, 'refmdb-1.8.7.zip')
										task_progress = 0
									when :on_refe_document_install_complete
										task_total = nil
									when :on_tool_install_start

										tool_name = config.tool_list.find{|x| x.id == args[0]}.name
										desc = "#{tool_name}をインストール中..."
										
										# GUIインストーラ上では、Threadを使用して進捗ゲージを更新し
										# ウインドウが固まってしまわないようにする
										wakeup_dispatcher(task_gauge)

									when :on_tool_install_complete
										kill_dispatcher
										
									when :on_path_env_add_start
										desc = "PATH環境変数を設定中..."
									when :on_start_menu_add_start
										desc = "スタートメニューへショートカットを追加中..."
									when :on_path_env_add_complete, :on_start_menu_add_complete
									when :on_child_installed
										i += 1
										if task_total then
											task_progress += 1
										end
									when :on_complete
										i = total
										desc = "Rumixのインストールを完了しました。"
										refresh_proc.call
									end
								}
								total_installing_console.label = '全体の進捗: 100%'
								task_gauge.range = 100
								task_gauge.value = 100
								task_gauge.disable
								buttons[:finish].enable
								buttons[:cancel].disable
							end

							Thread.kill(@updating_worker) if @updateing_worker
						}
					end
						
				end
			end
			
			# 進捗ゲージ更新用のスレッドを起動
			def wakeup_dispatcher(task_gauge)
				kill_dispatcher
				app = Wx.get_app
				#Thread.abort_on_exception = true
				#@dispatcher_thread = Thread.new{
				#	loop do
				#		task_gauge.pulse
				#		app.dispatch while app.pending
				#		sleep(0.05)
				#	end
				#}
			end
			
			def kill_dispatcher
				if @dispatcher_thread then
					#Thread.kill(@dispatcher_thread)
				end
			end

			# 進捗ゲージ更新
			def refresh_progress_gauge(app, installing_console, total_installing_console, gauge, task_gauge,
				                         desc, progress, total, task_progress, task_total, progress_io)
				label = "#{desc}"
				total_label = "全体の進捗: #{progress * 100 / total}%"
				unless @old_label == label then
					@old_label = label
					installing_console.label = label
				end
				unless @old_total_label == total_label then
					@old_total_label = total_label
					total_installing_console.label = total_label
				end
				gauge.value = progress

				if task_total then
					task_gauge.range = task_total
					task_gauge.value = task_progress
				else
					task_gauge.pulse
				end

				# デバッグ用
				if task_total then
					progress_io.puts "PROGRESS: #{label}, task = #{task_progress} / #{task_total} (#{task_progress * 100 / task_total}%), total = #{progress} / #{total} (#{progress * 100 / total}%)"
				else
					progress_io.puts "PROGRESS: #{label}, task = (unknown), total = #{progress} / #{total} (#{progress * 100 / total}%)"
				end
				#log_io.puts "PROGRESS: task = #{task_progress} / #{task_total} (#{task_progress * 100 / task_total}%), total = #{progress} / #{total} (#{progress * 100 / total}%)"
				
				app.dispatch while app.pending
				sleep(0.05)
			end
			
			def get_ext_ruby_description(path)
				"Rumixに含まれるrubyと同じ場所にインストールします。\n  パス: #{path}"
			end

			# 対象のzipファイル内のファイル数を取得
			def get_zip_file_number(arc_info, zip_file_name)
				n = (arc_info[zip_file_name] || {})['file_count'] 
				unless n then
					Kernel.raise "zip file not found - #{zip_file_name}"
				end

				return n
			end

			# 対象のgemのウェイト（インストール時間に占める大まかな重み）を取得
			def get_gem_weight(arc_info, config, *gem_names)
				file_count = 0
				gem_names.each do |gem_name|
					file_count += (arc_info[File.basename(config.get_gem_file_path(gem_name))] || {})['file_count'] 
					unless file_count then
						Kernel.raise "gem file not found - #{gem_name}"
					end
				end

				return file_count * 5 * config.ruby_configs.size
			end
			
		end


	end
end
